/* multilink.c - support for Liebert UPS models via MultiLink cable.

   Copyright (C) 2001  Rick Lyons <rick@powerup.com.au>
   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/termios.h>

#define INFO_MAX 8

#include "config.h"
#include "proto.h"
#include "shared.h"
#include "version.h"
#include "upscommon.h"
#include "common.h"

#define	ML_ONBATTERY	0x55

	int	shmok = 1;
extern	char	*pidfn;

void initinfo (void)
{
	create_info(INFO_MAX, shmok);

	/* setup the basics */

	addinfo (INFO_MFR,"Liebert", FLAG_STRING, 0);
	addinfo (INFO_MODEL,"MultiLink", FLAG_STRING, 0);
	addinfo (INFO_STATUS, "", 0, 0);
}

/* normal idle loop - keep up with the current state of the UPS */
void updateinfo (void)
{
	int	flags, ob, bl, ret;
	char	temp[VALSIZE];
	char	c;

	ob = 0;

	c = ML_ONBATTERY;
	write(upsfd, &c, 1);
	if (read(upsfd, &c, 1) == 1) {
		while (read(upsfd, &c, 1) == 1)
			continue;
		if (c == ML_ONBATTERY)
			ob = 1;
	}
	
	ret = ioctl (upsfd, TIOCMGET, &flags);

	if (ret != 0) {
		upslog(LOG_INFO, "ioctl failed");
		return;
	}

	bl = (flags & TIOCM_CD);

	strcpy (temp, "");

	if (bl)
		strcat (temp, "LB ");	/* low battery */

	if (ob)
		strcat (temp, "OB");	/* on battery */
	else
		strcat (temp, "OL");	/* on line */

	setinfo(INFO_STATUS, "%s", temp);

	writeinfo();
}

/* install pointers to functions for msg handlers called from msgparse */
void setuphandlers(void)
{
	/* TODO: future */
}

void usage(char *prog)
{
	printf ("usage: %s [-h] [-m <model>] [-M <mfr>] <device>\n", prog);
	printf ("Example: %s /dev/ttyS0\n", prog);
}

void help(char *prog)
{
	printf ("usage: %s [-h] [-m <model>] [-M <mfr>] <device>\n", prog);
	printf ("\n");
	printf ("-h         - display this help\n");
	printf ("-m <model> - set model name to <model>\n");
	printf ("-M <mfr>   - set manufacturer name to <mfr>\n");
	printf ("<device>   - /dev entry corresponding to UPS port\n");
}

/*  Open the serial port.  The upscommon routines aren't used because
    they lose bits off the MCR required to make TX/RX work, and we
    set up the port to make use of VMIN/VTIME.
*/
void
open_serial_ml(char *port)
{
	int	flags;
	struct termios tio;

	signal(SIGALRM, openfail);
	alarm(3);

	if ((upsfd = open(port, O_RDWR)) == -1)
		fatal("Unable to open %s", port);

	alarm(0);

	lockport(upsfd, port);

	if (ioctl(upsfd, TIOCMGET, &flags))
		fatal("get ioctl");
	flags |= TIOCM_RTS;
	if (ioctl(upsfd, TIOCMSET, &flags))
		fatal("set ioctl");

	tcgetattr(upsfd, &tio);
	cfmakeraw(&tio);
	tio.c_cc[VMIN] = 0;
	tio.c_cc[VTIME] = 1;
	tcsetattr(upsfd, TCSANOW, &tio);
}

int main (int argc, char **argv)
{
	char	*portname, *prog, *forcemodel = NULL, *forcemfr = NULL;
	int	i;

	printf ("Network UPS Tools - Liebert MultiLink UPS driver 1.0 (%s)\n", UPS_VERSION);
	openlog ("multilink", LOG_PID, LOG_FACILITY);
	prog = argv[0];
	
	while ((i = getopt(argc, argv, "+hm:M:")) != EOF) {
		switch (i) {
			case 'h':
				help (prog);
				exit (1);
				break;
			case 'm':
				forcemodel = optarg;
				break;
			case 'M':
				forcemfr = optarg;
				break;
			default:
				usage (prog);
				break;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc != 1) {
		help (prog);
		exit (1);
	}

	droproot();

	portname = NULL;
	for (i = strlen(argv[0]); i >= 0; i--)
		if (argv[0][i] == '/') {
			portname = &argv[0][i+1];
			break;
		}

	if (portname == NULL) {
		printf ("Unable to abbreviate %s\n", argv[0]);
		exit (1);
	}

	snprintf (statefn, sizeof(statefn), "%s/multilink-%s", STATEPATH,
	          portname);
	open_serial_ml(argv[0]);

	initinfo();

	/* replace model/mfr info if specified on command line */
	if (forcemfr)
		setinfo(INFO_MFR, "%s", forcemfr);

	if (forcemodel)
		setinfo(INFO_MODEL, "%s", forcemodel);

	createmsgq();   /* try to create IPC message queue */

	setuphandlers();

	background();

	pidfn = xmalloc(SMALLBUF);
	snprintf(pidfn, sizeof(pidfn), "multilink-%s", portname);
	writepid(pidfn);

	for (;;) {
		updateinfo();

		/* wait up to 2 seconds for a message from the upsd */

		if (getupsmsg(2))
			upslogx(LOG_INFO, "Received a message from upsd");
	}
}
