/* hidups.c - prototype HID UPS driver for Network UPS Tools
 
   Copyright (C) 2001  Russell Kroll <rkroll@exploits.org>
 
   Based on evtest.c v1.10 - Copyright (c) 1999-2000 Vojtech Pavlik
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or 
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "hidups.h"
#include "main.h"

	int	fd;

void upsdrv_shutdown(void)
{
	/* XXX: replace with a proper shutdown function */
	fatalx("shutdown not supported");
}

void instcmd(int auxcmd, int dlen, char *data) {}

void upsdrv_help(void) {}

void upsdrv_makevartable(void) {}

void upsdrv_banner(void)
{
	printf("Network UPS Tools: HID UPS driver 0.11 (%s)\n\n", UPS_VERSION);

	experimental_driver = 1;
}

static inline int find_application(int fd, unsigned usage) 
{
	int	i = 0, ret;

	while ((ret = ioctl(fd, HIDIOCAPPLICATION, i)) > 0 &&
	       ret != usage) 
		i++;

	return (ret == usage);
}

void parse_event(struct hiddev_event *ev)
{
	int	sc = 0;
	static	int	ol = -1, ob = -1, lb = -1;
	char	stattmp[16];

	/* XXX: deal with bogosity by throwing it out */
	if (ev->value > 3000000) {
		upslogx(LOG_INFO, "Ignoring bogus event 0x%x (%d)",
			ev->hid, ev->value);
		return;
	}

	/* x86 page == ups-specific, ignore these for now */
	if ((ev->hid & 0x860000) == 0x860000) {
		upslogx(LOG_INFO, "Ignoring x86 page event 0x%x (%d)",
			ev->hid, ev->value);
		return;
	}

	switch (ev->hid) {
		case UPS_BATTVOLT:
			setinfo(INFO_BATTVOLT, "%2.1f", ev->value / 100.0);
			break;
		case BATT_RUNTIME_TO_EMPTY:
			setinfo(INFO_RUNTIME, "%d", ev->value);
			break;
		case BATT_REMAINING_CAPACITY:
			setinfo(INFO_BATTPCT, "%d", ev->value);
			break;

		/* OB/OL/LB: update temp storage and flag changed */
		case BATT_DISCHARGING:
			if ((ev->value == 0) || (ev->value == 1)) {
				ob = ev->value;
				sc = 1;
			}
			else
				upslogx(LOG_WARNING,
					"Got bogus value for BATT_DISCHARGING: %d",
					ev->value);
			break;

		case BATT_AC_PRESENT:
			if ((ev->value == 0) || (ev->value == 1)) {
				ol = ev->value;
				sc = 1;
			}
			else
				upslogx(LOG_WARNING, 
					"Got bogus value for BATT_AC_PRESENT: %d",
					ev->value);
			break;

		case UPS_SHUTDOWN_IMMINENT:
			if ((ev->value == 0) || (ev->value == 1)) {
				lb = ev->value;
				sc = 1;
			}
			else
				upslogx(LOG_WARNING,
					"Got bogus value for UPS_SHUTDOWN_IMMINENT: %d", 
					ev->value);
			break;

		/* things that we don't care about */
		case BATT_BELOW_RCL:
		case BATT_CHARGING:
			break;

		default:
			upslogx(LOG_INFO, "Unhandled event: 0x%x (%d)",
				ev->hid, ev->value);
	}

	/* deal with any status changes */
	if (sc == 0)
		return;

	*stattmp = '\0';	

	if (ol == 1)
		strlcat(stattmp, "OL ", sizeof(stattmp));
	if (ob == 1)
		strlcat(stattmp, "OB ", sizeof(stattmp));
	if (lb == 1)
		strlcat(stattmp, "LB ", sizeof(stattmp));

	if (stattmp[strlen(stattmp) - 1] == ' ')
		stattmp[strlen(stattmp) - 1] = 0;

	setinfo(INFO_STATUS, "%s", stattmp);
}

int getvalue(int type)
{
	struct	hiddev_usage_ref uref;

	/* TODO: build a report table so we don't need HID_REPORT_ID_UNKNOWN */

	memset(&uref, 0, sizeof(uref));
	uref.report_type = HID_REPORT_TYPE_FEATURE;
	uref.report_id = HID_REPORT_ID_UNKNOWN;
	uref.field_index = 0;
	uref.usage_index = 0;
	uref.usage_code = type;
	uref.value = 0;

	if (ioctl(fd, HIDIOCGUSAGE, &uref) >= 0)
		return uref.value;
	else
		return -1;
}

static char *getstring(int type)
{
	struct	hiddev_usage_ref uref;
	struct	hiddev_string_descriptor sdesc;
	static	char	str[256];

	/* TODO: build a report table so we don't need HID_REPORT_ID_UNKNOWN */

	memset(&uref, 0, sizeof(uref));
	memset(&sdesc, 0, sizeof(sdesc));
	uref.report_type = HID_REPORT_TYPE_FEATURE;
	uref.report_id = HID_REPORT_ID_UNKNOWN;
	uref.usage_code = type;

	snprintf(str, sizeof(str), "Unknown");
	if (ioctl(fd, HIDIOCGUSAGE, &uref) == 0) {
		if ( 0 != (sdesc.index = uref.value)) {
		     if (ioctl(fd, HIDIOCGSTRING, &sdesc) > 0)
			  snprintf(str, sizeof(str), "%s", sdesc.value);
		     else
			  upslog(LOG_ERR, "ioctl HIDIOCGSTRING");
		}
	}

	return str;
}

	/* results of querying some x86 page values on my APC UPS */

	/* note to APC: i can decode your "secret protocol" just by        *
	 * looking at it.  you're not helping anyone by keeping it closed! */

	/* 0x860060 == "441HMLL" - looks like a 'capability' string	*/
	/*	    == locale 4, 4 choices, 1 byte each			*/
	/*          == line sensitivity (high, medium, low, low)	*/

	/* 0x860013 == 44200155090 - capability again			*/
	/*          == locale 4, 4 choices, 2 bytes, 00, 15, 50, 90	*/
	/*          == minimum charge to return online			*/

	/* 0x860062 == D43133136127130					*/
	/*          == locale D, 4 choices, 3 bytes, 133, 136, 127, 130	*/
	/*          == high transfer voltage				*/

	/* 0x860064 == D43103100097106					*/
	/*          == locale D, 4 choices, 3 bytes, 103, 100, 097, 106	*/
	/*          == low transfer voltage				*/

	/* 0x860066 == 441HMLL (see 860060)				*/	

	/* 0x860074 == 4410TLN						*/
	/*          == locale 4, 4 choices, 1 byte, 0, T, L, N		*/
	/*          == alarm setting (5s, 30s, low battery, none)	*/

	/* 0x860077 == 443060180300600					*/
	/*          == locale 4, 4 choices, 3 bytes, 060,180,300,600	*/
	/*          == wake-up delay (after power returns)		*/

void upsdrv_updateinfo(void) 
{
	fd_set	fdset;
	struct	timeval	tv;
	int	rd, i;
	struct	hiddev_event ev[64];

	FD_ZERO(&fdset);
	FD_SET(fd, &fdset);
	tv.tv_sec = 0;
	tv.tv_usec = 0;
	rd = select(fd+1, &fdset, NULL, NULL, &tv);

	/* XXX: alarm around this read */
	if (rd > 0) {
		rd = read(fd, ev, sizeof(ev));

		if (rd < (int) sizeof(ev[0])) {
			if (rd < 0)
				fatal("read");
			else
				upslog(LOG_INFO, "short read from device");
		}
	    
		for (i = 0; i < rd / sizeof(ev[0]); i++)
			parse_event(&ev[i]);
	}	/* if rd > 0 */

	writeinfo();
}

void addhidvalue(int query, int infotype)
{
	int	val;

	val = getvalue(query);

	if (val < 0)
		return;

	/* XXX: deal with "3 million" kernel bogosity for now */
	if (val > 3000000) {
		int	i;

		for (i = 0; i < 5; i++) {
			val = getvalue(query);
			if (val < 3000000)
				break;
		}

		if (val > 3000000) {
			upslogx(LOG_WARNING, "Unable to add 0x%x: got bogus value %d",
				query, val);
			return;
		}
	}

	addinfo(infotype, "", 0, 0);
	setinfo(infotype, "%d", val);
}

void upsdrv_initinfo(void) 
{
	int	val;
	char	*str, *ptr;

	addinfo(INFO_MFR, "Generic", 0, 0);

	str = getstring(UPS_IPRODUCT);

	if (str) {

		/* try to trim this back to something reasonable */
		if (!strncmp(str, "BackUPS Pro", 11)) {
			ptr = strstr(str, "FW");
			if (ptr)
				*(ptr - 1) = '\0';

			/* we can be pretty sure of this now */
			setinfo(INFO_MFR, "%s", "APC");
		}

		/* non-Pro models seem to have USB ports too now */
		if ((!strncmp(str, "BackUPS ", 8)) && (isdigit(str[8]))) {
			ptr = strstr(&str[8], " ");
			if (ptr)
				*(ptr - 1) = '\0';

			/* we can be pretty sure of this now */
			setinfo(INFO_MFR, "%s", "APC");
		}

		addinfo(INFO_MODEL, str, 0, 0);
	}
	else
		addinfo(INFO_MODEL, "Generic USB UPS", 0, 0);

	str = getstring(UPS_ISERIAL);

	if (str)
		addinfo(INFO_SERIAL, str, 0, 0);

	/* seed the status register */

	val = getvalue(BATT_AC_PRESENT);

	if (val == 1)
		addinfo(INFO_STATUS, "OL", 0, 0);
	else
		addinfo(INFO_STATUS, "OB", 0, 0);

	val = getvalue(UPS_BATTVOLT);

	if (val > 0) {
		addinfo(INFO_BATTVOLT, "", 0, 0);
		setinfo(INFO_BATTVOLT, "%2.1f", val / 100.0);
	}

	/* XXX: set capacitymode to percent */
	addhidvalue(BATT_REMAINING_CAPACITY, INFO_BATTPCT);
	addhidvalue(BATT_RUNTIME_TO_EMPTY, INFO_RUNTIME);
	addhidvalue(UPS_LOADPCT, INFO_LOADPCT);
}

void upsdrv_initups(void)
{
	char	name[256];

	if ((fd = open(device_path, O_RDONLY)) < 0)
		fatal("hiddev open %s", device_path);

	if ((!find_application(fd, UPS_USAGE)) &&
		(!find_application(fd, POWER_USAGE)))
		fatalx("%s is not a UPS\n", device_path);

	ioctl(fd, HIDIOCGNAME(sizeof(name)), name);
	printf("Detected %s\n", name);
	printf("on port %s\n", device_path);

	ioctl(fd, HIDIOCINITREPORT, 0);
}

/* tell main how many entries we need */
int upsdrv_infomax(void)
{
	return 32;
}
