/* cyberpower.c - Network UPS Tools driver for Cyber Power Systems units

   Copyright (C) 2001  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              
*/

#include "main.h"

#include <sys/ioctl.h>
#include <sys/termios.h>

/* adjust bizarre UPS data to observed voltage data */
int voltconvert(unsigned char in)
{
	int v_end[43] = { 36, 51, 55, 60, 65, 70, 75, 80, 85, 91, 98, 103, 108, 113, 118, 123, 128, 133, 138, 143, 148, 153, 158, 163, 168, 173, 178, 183, 188, 193, 198, 203, 208, 213, 218, 223, 228, 233, 238, 243, 248, 253, 255 };
	int v_adj[43] = {  3,  4,  5,  4,  3,  2,  1,  0, -1, -2, -3,  -4,  -5,  -6,  -7,  -8,  -9, -10, -11, -12, -13, -14, -15, -16, -17, -18, -19, -20, -21, -22, -23, -24, -25, -26, -27, -28, -29, -30, -31, -32, -33, -34, -35 };
	int	i;

	if (in < 27)
		return 0;

	for (i = 0; i < 19; i++) { 
		if (in <= v_end[i]) {
			return (in + v_adj[i]);
		}
	}

	return 0;
}

/* map UPS data to realistic percentages */
int battconvert(unsigned char in)
{
	int b_val[26] = {0, 1, 1, 2, 3, 4, 6, 8, 10, 12, 15, 18, 22, 26, 30, 35, 40, 46, 52, 58, 66, 73, 81, 88, 99, 100 }; /* XXX - for load of 0 */

	if (in > 185)
		return 100;

	if (in < 160)
		return 0;

	return (b_val[in - 160]);
}

/* more wacky mapping - get the picture yet? */

struct {
	int	st;
	int	end;
	int	sz;
	int	base;
}	temptab[] =
{
	{   0,  39, 5,  0 },
	{  40,  43, 4,  8 }, 
	{  44,  78, 5,  9 },
	{  79,  82, 4, 16 },
	{  83, 117, 5, 17 },
	{ 118, 121, 4, 24 },
	{ 122, 133, 3, 25 },
	{ 134, 135, 2, 29 },
	{ 136, 143, 4, 30 },
	{ 144, 146, 3, 32 },
	{ 147, 150, 4, 33 },
	{ 151, 156, 3, 34 },
	{ 157, 164, 2, 36 },
	{ 165, 170, 3, 40 },
	{ 171, 172, 2, 42 },
	{ 173, 175, 3, 43 },
	{ 176, 183, 2, 44 },
	{ 184, 184, 1, 48 },
	{ 185, 188, 2, 49 },
	{ 189, 190, 2, 51 },
	{ 191, 191, 1, 52 },
	{ 192, 193, 2, 53 },
	{ 194, 194, 1, 54 },
	{ 195, 196, 2, 55 },
	{ 197, 197, 1, 56 },
	{ 198, 199, 2, 57 },
	{ 200, 200, 1, 58 },
	{ 201, 202, 2, 59 },
	{ 203, 203, 1, 60 },
	{ 204, 205, 2, 61 },
	{ 206, 206, 1, 62 },
	{ 207, 208, 2, 63 },
	{ 209, 209, 1, 64 },
	{ 210, 211, 2, 65 },
	{ 212, 212, 1, 66 },
	{ 213, 213, 1, 67 },
	{ 214, 214, 1, 68 },
	{ 215, 215, 1, 69 },
	{ 216, 255, 40, 70 },
	{   0,   0, 0,  0 },
};

float tempconvert(unsigned char in)
{
	int	i, j, found, count;

	found = -1;
	for (i = 0; temptab[i].sz != 0; i++)
		if ((temptab[i].st <= in) && (temptab[i].end >= in))
			found = i;

	if (found == -1) {
		upslogx(LOG_ERR, "tempconvert: unhandled value %d", in);
		return 0;
	}

	count = temptab[found].end - temptab[found].st + 1;

	for (i = 0; i < count; i++) {
		j = temptab[found].st + (i * temptab[found].sz);

		if ((in - j) < temptab[found].sz) {
			return ((float)((in - j) / temptab[found].sz) + 
				temptab[found].base + i);
		}
	}

	upslogx(LOG_ERR, "tempconvert: fell through with %d", in);
	return 0;
}

void sendtoups(char ch)
{
	write(upsfd, &ch, 1);
	usleep(50000);
}

/* power down the attached load immediately */
void upsdrv_shutdown(void)
{
	/* XXX */
}

void upsdrv_updateinfo(void)
{
	int	ret, count;
	char	ch, buf[SMALLBUF], stbuf[SMALLBUF];

	sendtoups('D');
	sendtoups(13);

	count = 0;

	memset(buf, '\0', sizeof(buf));
	while (count < 14) {
		ret = read(upsfd, &ch, 1);
		buf[count++] = ch;
	}

	if (buf[0] != '#') {
		upslogx(LOG_ERR, "Invalid start char 0x%02x", buf[0] & 0xff);
		return;
	}

	if ((buf[4] != 46) || (buf[8] != 46)) {
		upslogx(LOG_ERR, "Invalid separator in response");
		return;
	}

	setinfo(INFO_UPSTEMP, "%2.1f", tempconvert(buf[6]));
	setinfo(INFO_BATTPCT, "%03d", battconvert(buf[5]));
	setinfo(INFO_LOADPCT, "%03d", (buf[3] & 0xff) * 2);
	setinfo(INFO_UTILITY, "%03d", voltconvert(buf[1]));

	memset(stbuf, '\0', sizeof(stbuf));

	if (buf[9] & 2)
		strcat(stbuf, "OFF ");

	if (buf[9] & 64)
		strcat(stbuf, "LB ");

	if (buf[9] & 128)
		strcat(stbuf, "OB ");
	else
		strcat(stbuf, "OL ");

	if (stbuf[strlen(stbuf)-1] == ' ')
		stbuf[strlen(stbuf)-1] = '\0';

	setinfo(INFO_STATUS, "%s", stbuf);
	writeinfo();
}

void instcmd(int auxcmd, int dlen, char *data)
{
	switch(auxcmd) {
		default:
			upslogx(LOG_INFO, "instcmd: unknown type 0x%04x",
				auxcmd);
	}
}

/* install pointers to functions for msg handlers called from msgparse */
void setuphandlers(void)
{
	upsh.instcmd = instcmd;
}

void upsdrv_banner(void)
{
	printf("Network UPS Tools - CyberPower driver 0.01 (%s)\n", UPS_VERSION);

	experimental_driver = 1;
}

void upsdrv_help(void)
{
}

void upsdrv_makevartable(void)
{
}

/* prep the serial port */
void upsdrv_initups(void)
{
	int dtr_bit = TIOCM_DTR;
	int rts_bit = TIOCM_RTS;
	struct termios tio;

	open_serial(device_path, B1200);

	/* dtr high, rts high */
	ioctl(upsfd, TIOCMBIS, &rts_bit);
	ioctl(upsfd, TIOCMBIS, &dtr_bit);

	tcgetattr(upsfd, &tio);
	tio.c_cflag = 0 | CS8 | CLOCAL | ~CRTSCTS | CREAD;
	cfmakeraw(&tio);
	tio.c_cflag &= ~(PARODD|CSTOPB|HUPCL|CRTSCTS);

	cfsetispeed(&tio, B1200);
	cfsetospeed(&tio, B1200);
	tcsetattr(upsfd, TCSANOW, &tio);
}

void getbaseinfo()
{
	addinfo(INFO_MODEL, "AVR700", 0, 0);	/* XXX */
	addinfo(INFO_UPSTEMP, "", 0, 0);
	addinfo(INFO_BATTPCT, "", 0, 0);
	addinfo(INFO_LOADPCT, "", 0, 0);
	addinfo(INFO_UTILITY, "", 0, 0);
	addinfo(INFO_STATUS, "", 0, 0);

	/* poll once to put in some good data */
	upsdrv_updateinfo();
}

void upsdrv_initinfo(void)
{
	/* XXX: probe */

#if 0
	res = init_communication();
	if (res == -1) {
		printf("Unable to detect a CyberPower UPS on port %s\n", 
			device_path);
		printf("Check the cabling, port name or model name and try again\n");
		exit(1);
	}
#endif

	addinfo(INFO_MFR, "CyberPower", 0, 0);

	getbaseinfo();

	printf("Detected %s on %s\n", getdata(INFO_MODEL), device_path); 

	setuphandlers(); 
}

int upsdrv_infomax(void) { return 16; }
