/* common.c - common useful functions

   Copyright (C) 2000  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "common.h"
#include "shared.h"

#include <ctype.h>
#include <syslog.h>
#include <pwd.h>
#include <grp.h>

	int	nut_debug_level = 0;
	int     upslog_flags = UPSLOG_STDERR;

void xbit_set(int *val, int flag)
{
	*val = (*val |= flag);
}

void xbit_clear(int *val, int flag)
{
	*val = (*val ^= (*val & flag));
}

int xbit_test(int val, int flag)
{
	return ((val & flag) == flag);
}

/* close ttys and become a daemon */
void background(void)
{
	int	pid;

	if ((pid = fork()) < 0)
		fatal("Unable to enter background");

	xbit_set(&upslog_flags, UPSLOG_SYSLOG);
	xbit_clear(&upslog_flags, UPSLOG_STDERR);

	close(0);
	close(1);
	close(2);

	if (pid != 0) 
		_exit(0);		/* parent */

	/* child */

	/* make fds 0-2 point somewhere defined */
	if (open("/dev/null", O_RDWR) != 0)
		fatal("open /dev/null");
	dup(0);
	dup(0);

#ifdef HAVE_SETSID
	setsid();		/* make a new session to dodge signals */
#endif

	upslogx(LOG_INFO, "Startup successful");
}


void switch_to_user(const char *user, const char *group)
{
	struct passwd *pw;
	struct group  *gr;
	gid_t gid;

	if (getuid() == 0)
		if (seteuid(0))
			; /* error */

	if ((pw = getpwnam(user)) == NULL)
		fatalx("getpwnam() failed: user %s does not exist", user);
	gid = pw->pw_gid;

	if (initgroups(user, pw->pw_gid) == -1)
		fatal("initgroups");

	if (group) {
		if ((gr = getgrnam(group)) == NULL)
			fatalx("getgrnam() failed: group %s does not exist", group);
		gid = gr->gr_gid;
	}

	if (setgid(gid) == -1)
		fatal("setgid");

        /* change uid second */
	if (setuid(pw->pw_uid) == -1)
		fatal("setuid");
}

/* change uid/gid if running as root */
void droproot(void)
{
	if (getuid() == 0 || geteuid() == 0)
		switch_to_user(RUN_AS_USER, RUN_AS_GROUP);
}

/* drop off a pidfile for this process */
void writepid(const char *name)
{
	char	fn[SMALLBUF];
	FILE	*pidf;
	int	mask;

	/* use full path if present, else build filename in PIDPATH */
	if (*name == '/')
		strlcpy(fn, name, sizeof(fn));
	else
		snprintf(fn, sizeof(fn), "%s/%s.pid", PIDPATH, name);

	mask = umask(022);
	pidf = fopen(fn, "w");

	if (pidf) {
		fprintf(pidf, "%d\n", (int) getpid());
		fclose(pidf);
	} else {
		upslog(LOG_NOTICE, "writepid: fopen %s", fn);
	}

	umask(mask);
}

/* open pidfn, get the pid, then send it sig */
void sendsignalfn(const char *pidfn, int sig)
{
	char	buf[SMALLBUF];
	FILE	*pidf;
	int	pid, ret;

	pidf = fopen(pidfn, "r");
	if (!pidf) {
		upslog(LOG_NOTICE, "fopen %s", pidfn);
		return;
	}

	fgets(buf, sizeof(buf), pidf);
	buf[strlen(buf)-1] = '\0';

	pid = strtol(buf, (char **)NULL, 10);

	if (pid < 2) {
		upslogx(LOG_NOTICE, "Ignoring invalid pid number %d", pid);
		return;
	}

	/* see if this is going to work first */
	ret = kill(pid, 0);

	if (ret < 0) {
		perror("kill");
		return;
	}

	/* now actually send it */
	ret = kill(pid, sig);

	if (ret < 0) {
		perror("kill");
		return;
	}
}

int snprintfcat(char *dst, size_t size, const char *fmt, ...)
{
	va_list ap;
	int len = strlen(dst);
	int ret;

	size--;
	assert(len <= size);

	va_start(ap, fmt);
	ret = vsnprintf(dst + len, size - len, fmt, ap);
	va_end(ap);

	dst[size] = '\0';
	return len + ret;
}

/* lazy way to send a signal if the program uses the PIDPATH */
void sendsignal(const char *progname, int sig)
{
	char	fn[SMALLBUF];

	snprintf(fn, sizeof(fn), "%s/%s.pid", PIDPATH, progname);

	sendsignalfn(fn, sig);
}

const char *xbasename(const char *file)
{
	const char *p = strrchr(file, '/');

	if (p == NULL)
		return file;
	return p + 1;
}

int findinfo(const itype *info, int size, int offset, int type)
{
	int x;

	for (x = offset; x < size; x++) {
		if (info[x].type == type)
			return x;
	}

	return -1;
}

void vupslog(int priority, const char *fmt, va_list va, int use_strerror)
{
	char buf[LARGEBUF];

	if (vsnprintf(buf, sizeof(buf), fmt, va) >= sizeof(buf))
		; /* overflow should be handled */

	if (use_strerror) {
		strlcat(buf, ": ", sizeof(buf));
		strlcat(buf, strerror(errno), sizeof(buf));
	}

	if (xbit_test(upslog_flags, UPSLOG_STDERR))
		fprintf(stderr, "%s\n", buf);
	if (xbit_test(upslog_flags, UPSLOG_SYSLOG))
		syslog(priority, "%s", buf);
}

/* logs the formatted string to any configured logging devices + the output of strerror(errno) */
void upslog(int priority, const char *fmt, ...)
{
	va_list va;

	va_start(va, fmt);
	vupslog(priority, fmt, va, 1);
	va_end(va);
}

/* logs the formatted string to any configured logging devices */
void upslogx(int priority, const char *fmt, ...)
{
	va_list va;

	va_start(va, fmt);
	vupslog(priority, fmt, va, 0);
	va_end(va);
}

void upsdebug(int level, const char *fmt, ...)
{
	va_list va;
	
	if (nut_debug_level < level)
		return;

	va_start(va, fmt);
	vupslog(LOG_DEBUG, fmt, va, 1);
	va_end(va);
}

void upsdebugx(int level, const char *fmt, ...)
{
	va_list va;
	
	if (nut_debug_level < level)
		return;

	va_start(va, fmt);
	vupslog(LOG_DEBUG, fmt, va, 0);
	va_end(va);
}

void vfatal(const char *fmt, va_list va, int use_strerror)
{
	if (xbit_test(upslog_flags, UPSLOG_STDERR_ON_FATAL))
		xbit_set(&upslog_flags, UPSLOG_STDERR);
	if (xbit_test(upslog_flags, UPSLOG_SYSLOG_ON_FATAL))
		xbit_set(&upslog_flags, UPSLOG_SYSLOG);

	vupslog(LOG_ERR, fmt, va, use_strerror);
}

void fatal(const char *fmt, ...)
{
	va_list va;

	va_start(va, fmt);
	vfatal(fmt, va, 1);
	va_end(va);

	exit(1);
}

void fatalx(const char *fmt, ...)
{
	va_list va;

	va_start(va, fmt);
	vfatal(fmt, va, 0);
	va_end(va);

	exit(1);
}

static const char *oom_msg = "Out of memory";

void *xmalloc(size_t size)
{
	void *p = malloc(size);

	if (p == NULL)
		fatal("%s", oom_msg);
	return p;
}

void *xcalloc(size_t number, size_t size)
{
	void *p = calloc(number, size);

	if (p == NULL)
		fatal("%s", oom_msg);
	return p;
}

void *xrealloc(void *ptr, size_t size)
{
	void *p = realloc(ptr, size);

	if (p == NULL)
		fatal("%s", oom_msg);
	return p;
}

char *xstrdup(const char *string)
{
	char *p = strdup(string);

	if (p == NULL)
		fatal("%s", oom_msg);
	return p;
}

#ifndef HAVE_CFMAKERAW

int cfmakeraw(struct termios *termios_p)
{
	termios_p->c_iflag &= ~(IGNBRK|BRKINT|PARMRK|ISTRIP
				|INLCR|IGNCR|ICRNL|IXON);
	termios_p->c_oflag &= ~OPOST;
	termios_p->c_lflag &= ~(ECHO|ECHONL|ICANON|ISIG|IEXTEN);
	termios_p->c_cflag &= ~(CSIZE|PARENB);
	termios_p->c_cflag |= CS8;

	return 0;
}

#endif
