/* upsct - simple "client" to test read/write variable access

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "common.h"

#include <netdb.h>
#include <netinet/in.h>
#include <sys/socket.h>

#include "upsfetch.h"

	int	fd;

void do_enum(char *varname)
{
	char	out[SMALLBUF], temp[SMALLBUF];

	snprintf(out, sizeof(out), "ENUM %s\n", varname);

	if (upssendraw(fd, out) < 0) {
		printf("Enumerate %s failed: %s\n", 
			varname, upsstrerror(upserror));
		exit(1);
	}

	if (upsreadraw(fd, temp, sizeof(temp)) < 0) {
		printf("Enumerate %s failed: %s\n",
			varname, upsstrerror(upserror));
		exit(1);
	}

	if (strncmp(temp, "ENUM", 4) != 0) {
		printf("Bogus reply from server for %s\n", varname);
		return;
	}

	if (upsreadraw(fd, temp, sizeof(temp)) < 0) {
		printf("Can't get next ENUM value: %s\n",
			upsstrerror(upserror));
		exit(1);
	}

	while (strcmp(temp, "END") != 0) {
		printf("Option: %s\n", strstr(temp, "\""));

		if (upsreadraw(fd, temp, sizeof(temp)) < 0) {
			printf("Can't get next ENUM value: %s\n",
				upsstrerror(upserror));
			exit(1);
		}
	}
}

void usage(char *prog)
{
	printf("Network UPS Tools upsct2 %s\n\n", UPS_VERSION);
        printf("usage: %s [-h]\n", prog);
	printf("       %s [-s <variable>] [-u <username>] [-p <password>] <ups>\n\n", prog);
	printf("Demo program to set variables within UPS hardware.\n");
        printf("\n");
        printf("  -h            display this help text\n");
	printf("  -s <variable>	specify variable to be changed\n");
	printf("		use -s VAR=VALUE to avoid prompting for value\n");
        printf("  -u <username> set username for command authentication\n");
        printf("  -p <password> set password for command authentication\n");
        printf("\n");
        printf("  <ups>         UPS identifier - myups@localhost, etc.\n");
	printf("\n");
	printf("Call without -s to show all possible read/write variables.\n");
	exit(1);
}

void do_setvar(char *varname, int fd, char *upsname, char *user, char *pass)
{
	char	newval[SMALLBUF], temp[SMALLBUF], *ptr;

	if (!user) {
		printf("Username: ");
		user = xmalloc(SMALLBUF);
		fgets(user, SMALLBUF, stdin);

		/* deal with that pesky newline */
		if (strlen(user) > 1)
			user[strlen(user) - 1] = '\0';
	}

	if (!pass)
		pass = getpass("Password: " );

	/* Check if varname is in VAR=VALUE form */
	if ((ptr = strchr(varname, '=')) != NULL) {
		*ptr++ = 0;
		strlcpy(newval, ptr, sizeof(newval));
	} else {
		printf("Enter new value for %s: ", varname);
		fflush(stdout);
		fgets(newval, sizeof(newval), stdin);
		newval[strlen(newval) - 1] = '\0';
	}

	snprintf(temp, sizeof(temp), "USERNAME %s\n", user);

	if (upssendraw(fd, temp) < 0) {
		printf("Can't set username: %s\n", upsstrerror(upserror));
		exit(1);
	}

	if (upsreadraw(fd, temp, sizeof(temp)) < 0) {
		if (upserror == UPSF_UNKNOWNCMD) {
			printf("Set username failed due to an unknown command.\n");
			printf("You probably need to upgrade upsd.\n");
			exit(1);
		}

		printf("Set username failed: %s\n", upsstrerror(upserror));
		exit(1);
	}

	snprintf(temp, sizeof(temp), "PASSWORD %s\n", pass);

	if (upssendraw(fd, temp) < 0) {
		printf("Can't set password: %s\n", upsstrerror(upserror));
		exit(1);
	}

	if (upsreadraw(fd, temp, sizeof(temp)) < 0) {
		printf("Set password failed: %s\n", upsstrerror(upserror));
		exit(1);
	}

	snprintf(temp, sizeof(temp), "SET %s %s\n", varname, newval);

	if (upssendraw(fd, temp) < 0) {
		printf("Can't set variable: %s\n", upsstrerror(upserror));
		exit(1);
	}

	if (upsreadraw(fd, temp, sizeof(temp)) < 0) {
		printf("Set variable failed: %s\n", upsstrerror(upserror));
		exit(1);
	}

	exit(0);
}	

int main(int argc, char **argv)
{
	char	*upsname, *ptr, *host, vars[SMALLBUF], temp[SMALLBUF], out[SMALLBUF], *v,
		type[16], *setvar, *prog, *password = NULL, *username = NULL;
	int	typelen, i, ret;

	setvar = NULL;
	prog = argv[0];

	while ((i = getopt(argc, argv, "+s:p:u:")) != EOF) {
		switch (i) {
		case 's':
			setvar = xstrdup(optarg);
			break;
		case 'p':
			password = optarg;
			break;
		case 'u':
			username = optarg;
			break;
		default:
			usage(prog);
			break;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc < 1)
		usage(prog);

	/* handle upsname@hostname syntax and split up parts */
	ptr = strstr(argv[0], "@");
	if (ptr != NULL) {
		ptr[0] = 0;
		upsname = argv[0];
		host = ptr + 1;
	}
	else {
		upsname = NULL;
		host = argv[0];
	}

	fd = upsconnect(host);
	if (fd < 0) {
		printf("Unable to connect to %s - %s\n", host,
		       upsstrerror(upserror));
		exit(1);
	}

	/* setting a variable? */
	if (setvar)
		do_setvar(setvar, fd, upsname, username, password);

	/* if not, get the list of supported read/write variables */

	printf("host: %s\n", host);

	if (upsname) {
		char	buf[SMALLBUF];

		snprintf(buf, sizeof(buf), "LISTRW %s\n", upsname);
		ret = upssendraw(fd, buf);
	}
	else
		ret = upssendraw(fd, "LISTRW\n");

	if (ret < 0) {
		printf("Can't list read/write variables: %s\n",
			upsstrerror(upserror));
		exit(1);
	}
	
	if (upsreadraw(fd, vars, sizeof(vars)) < 0) {
		printf("Unable to get variable list - %s\n",
		       upsstrerror(upserror));
		exit(1);
	}

	v = vars;
	ptr = strchr(v, ' ');
	if (!ptr) {
		printf("Broken string from server: %s\n", vars);
		exit(1);
	}

	*ptr++ = '\0';
	v = ptr;

	/* possibly skip '@ups' */
	if (ptr[0] == '@') {
		ptr = strchr(ptr, ' ');
		if (!ptr) {
			printf("No R/W variables available on this UPS.\n");
			exit(0);
		}
		ptr++;
		v = ptr;
	}

	while (v != NULL) {
		ptr = strchr(v, ' ');
		if (ptr)
			*ptr++ = '\0';

		/* get description */
		snprintf(out, sizeof(out), "VARDESC %s\n", v);

		if (upssendraw(fd, out) < 0) {
			printf("Can't get description of %s: %s\n",
				v, upsstrerror(upserror));
			exit(1);
		}

		if (upsreadraw(fd, temp, sizeof(temp)) < 0) {
			printf("Can't get description of %s: %s\n",
				v, upsstrerror(upserror));
			exit(1);
		}

		printf("[%s] %s ", v, strstr(temp, "\""));

		/* now get type */
		snprintf(out, sizeof(out), "VARTYPE %s\n", v);

		if (upssendraw(fd, out) < 0) {
			printf("Can't get type of %s: %s\n",
				v, upsstrerror(upserror));
			exit(1);
		}

		if (upsreadraw(fd, temp, sizeof(temp)) < 0) {
			printf("Can't get type of %s: %s\n",
				v, upsstrerror(upserror));
			exit(1);
		}

		sscanf(temp, "%*s %s %i", type, &typelen); 
		printf("(%s:%i)\n", type, typelen);

		if (!strcmp(type, "ENUM"))
			do_enum(v);

		if (!strcmp(type, "STRING")) {
			if (getupsvarfd(fd, upsname, v, temp, 
				sizeof(temp)) < 0) {

				printf("Can't get value of %s: %s\n",
					temp, upsstrerror(upserror));
				exit(1);
			}
				
			printf("Value: %s\n", temp);
		}
		
		v = ptr;
	}
	
	return 0;
}
