/*
 * -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
 * top.c - Display top network usage using curses.
 *
 * Luca Deri     <deri@ntop.org>
 * Rocco Carbone <rocco@ntop.org>
 * -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/*
 * ntop header file(s)
 */
#include "ntop.h"

#include "intop.h"

/*
 * Static variables.
 */
static int idleFlag = 1;
static int columnSort = 0;
static int screenNumber = 0;
static int localAddrFlag = 1;
static int percentMode = 0;
static int sortSendMode = 0;

/*
 * Shamelessly ripped from ntop's report.c source file
 * and adapted to do the job.
 */

void clrscr ()
{
  attrset (A_NORMAL);
  erase ();
}


/*
 * Well formatted Bytes routine.
 */
char * cformatBytes (TrafficCounter numBytes)
{
#define BUFFER_SIZE   24
  static char outStr [BUFFER_SIZE] [32];
  static short bufIdx = 0;

  bufIdx = (bufIdx + 1) % BUFFER_SIZE;

  if (numBytes < 1024)
    sprintf (outStr [bufIdx], "%lu", (unsigned long) numBytes);
  else if (numBytes < 1048576)
    sprintf (outStr[bufIdx], "%.1f Kb", ((float) (numBytes) / 1024));
  else
    {
      float tmpMBytes = ((float) numBytes) / 1048576;

      if (tmpMBytes < 1024)
	sprintf (outStr[bufIdx], "%.1f MB", tmpMBytes);
      else
	{
	  tmpMBytes /= 1024;

	  if (tmpMBytes < 1024)
	    sprintf (outStr[bufIdx], "%.1f GB", tmpMBytes);
	  else
	    sprintf (outStr[bufIdx], "%.1f TB", ((float) (tmpMBytes) / 1024));
	}
    }

  return (outStr [bufIdx]);
}


/*
 * Well formatted Packets routine.
 */
char * cformatPkts (TrafficCounter pktNr)
{
  static short bufIdx = 0;
  static char staticBuffer [5] [32];

  bufIdx = (bufIdx + 1) % 5;

  if (pktNr < 1000)
    sprintf (staticBuffer [bufIdx], "%lu", (unsigned long) pktNr);
  else if (pktNr < 1000000)
    sprintf (staticBuffer [bufIdx], "%lu,%03lu",
	     (unsigned long) (pktNr / 1000),
	     ((unsigned long) pktNr) % 1000);
  else
    {
      unsigned long a, b, c;

      a = (unsigned long) (pktNr / 1000000);
      b = (unsigned long) ((pktNr - a * 1000000) / 1000);
      c = ((unsigned long) pktNr) % 1000;
      sprintf (staticBuffer [bufIdx], "%lu,%03lu,%03lu", a, b, c);
    }

  return (staticBuffer [bufIdx]);
}


char * cformatThroughput (float numBytes)
{
  static char outStr [5] [32];
  static short bufIdx = 0;

  float numBits;

  bufIdx = (bufIdx + 1) % 5;

  if (numBytes < 0)
    numBytes = 0; /* Sanity check */
  numBits = numBytes * 8;

  if (numBits < 100)
    numBits = 0; /* Avoid very small decimal values */

  if (numBits < 1024)
    {
      sprintf (outStr [bufIdx], "%.1f ", numBits);
    }
  else if (numBits < 1048576)
    {
      sprintf (outStr [bufIdx], "%.1f Kbps", ((float) (numBits) / 1024));
    }
  else
    {
      sprintf (outStr [bufIdx], "%.1f Mbps", ((float) (numBits) / 1048576));
    }

  return (outStr [bufIdx]);
}


char cformatStatus (HostTraffic * el)
{

  if (el->lastBytesSent == el->bytesSent && el->lastBytesReceived == el->bytesReceived)
    return ('I'); /* Idle */
  else if (el->lastBytesSent != el->bytesSent && el->lastBytesReceived != el->bytesReceived)
    return ('B'); /* Both */
  else if (el->lastBytesSent != el->bytesSent)
    return ('S'); /* Send */
  else
    return ('R'); /* Receive */
}


static void intop_printHeader (char * interface)
{
  char buf [BUF_SIZE] = {0};
  char tmpBuf [24], * c = "", * d = "", * e = "";
  int idx;
  char headerline [256];

  /* clear the screen */
  clrscr ();

  strcpy (headerline, intop_banner (interface));

  /* avoid long lines on small screens */
  headerline [80] = '\0';
  headerline [COLS - 2] = '\0';
  mvprintw (0, 40 - strlen (headerline) / 2, headerline);

  switch (screenNumber)
    {
    case 0:
      c = "TCP";
      d = "UDP";
      e = "ICMP";
      break;

    case 1:
      c = "DLC";
      d = "IPX";
      e = "Decnet";
      break;

    case 2:
      c = "(R)ARP";
      d = "ATalk";
      e = "OSPF";
      break;

    case 3:
      c = "NBios";
      d = "IGMP";
      e = "OSI";
      break;

    case 4:
      c = "QNX";
      d = "Other";
      e = "";
      break;

    default:
      idx = (screenNumber - MAX_NUM_PROTOS_SCREENS) * 3;
      if (idx < numIpProtosToMonitor)
	c = protoIPTrafficInfos [idx];
      else
	c = "";

      ++ idx;
      if (idx < numIpProtosToMonitor)
	d = protoIPTrafficInfos [idx];
      else
	d = "";

      ++ idx;
      if (idx < numIpProtosToMonitor)
	e = protoIPTrafficInfos [idx];
      else
	e = "";
    }


  switch (columnSort)
    {
      /* case 0: = no sort */
    case 1:
      sprintf (tmpBuf, "-%s-", c);
      c = tmpBuf;
      break;

    case 2:
      sprintf (tmpBuf, "-%s-", d);
      d = tmpBuf;
      break;

    case 3:
      sprintf (tmpBuf, "-%s-", e);
      e = tmpBuf;
      break;
    }

  if (sortSendMode)
    sprintf (buf, " %-24.24s %s%9s %9s %9s %9s %8s ",
	     "Host", "Act", "Rcvd", "-Sent-", c, d, e);
  else
    sprintf (buf, " %-24.24s %s%9s %9s %9s %9s %8s ",
	     "Host", "Act", "-Rcvd-", "Sent", c, d, e);

 /* avoid long lines on small screens */
  buf [COLS - 2] = '\0';
  attrset (A_REVERSE);
  mvprintw (2, 0, buf);
  attrset (A_NORMAL);
  refresh ();
}


static void yafillDomainName (HostTraffic * el)
{
  u_int i;

  if (theDomainHasBeenComputed(el)
      || (el->hostSymIpAddress == NULL))
    return;

#ifdef MULTITHREADED
  accessMutex (&addressResolutionMutex, "yafillDomainName");
#endif
  
  if ((el->hostSymIpAddress[0] == '*')
      || (el->hostNumIpAddress[0] == '\0')
      || (isdigit(el->hostSymIpAddress[strlen(el->hostSymIpAddress)-1]) &&
	  isdigit(el->hostSymIpAddress[0]))) {
    /* NOTE: theDomainHasBeenComputed(el) = 0 */
    el->fullDomainName = el->dotDomainName = "";
#ifdef MULTITHREADED
    releaseMutex (&addressResolutionMutex);
#endif
    return;
  }

  FD_SET( THE_DOMAIN_HAS_BEEN_COMPUTED_FLAG, &el->flags);
  el->fullDomainName = el->dotDomainName = ""; /* Reset values... */

  i = strlen(el->hostSymIpAddress)-1;

  while (i > 0)
    if (el->hostSymIpAddress[i] == '.')
      break;
    else
      i--;

  if ((i > 0)
      && strcmp(el->hostSymIpAddress, el->hostNumIpAddress)
      && (strlen(el->hostSymIpAddress) > (i+1)))
    el->dotDomainName = &el->hostSymIpAddress[i+1];
  else
    {
    /* Let's use the local domain name */

      if ((domainName[0] != '\0')
	  && (strcmp(el->hostSymIpAddress, el->hostNumIpAddress))) {
	int len  = strlen(el->hostSymIpAddress);
	int len1 = strlen(domainName);

	if ((len > len1)
	    && (strcmp(&el->hostSymIpAddress[len-len1-1], domainName) == 0))
	  el->hostSymIpAddress[len-len1-1] = '\0';

	el->fullDomainName = domainName;
	el->dotDomainName = shortDomainName;
      }
      else
	{
	  el->fullDomainName = el->dotDomainName = "";
	}

#ifdef MULTITHREADED
      releaseMutex (&addressResolutionMutex);
#endif
      return;
    }
  
  for (i=0; el->hostSymIpAddress[i] != '\0'; i++)
    el->hostSymIpAddress[i] = tolower(el->hostSymIpAddress[i]);

  i = 0;
  while (el->hostSymIpAddress[i] != '\0')
    if (el->hostSymIpAddress[i] == '.')
      break;
    else
      i++;

  if ((el->hostSymIpAddress[i] == '.')
      && (strlen(el->hostSymIpAddress) > (i+1)))
    el->fullDomainName = &el->hostSymIpAddress[i+1];

  
#ifdef MULTITHREADED
    releaseMutex (&addressResolutionMutex);
#endif
}



static int yacmpFctn (const void * _a, const void * _b)
{
  HostTraffic ** a = (HostTraffic **) _a;
  HostTraffic ** b = (HostTraffic **) _b;
  TrafficCounter a_=0, b_=0;
  float fa_=0, fb_=0;
  int idx;
  short oldColumnSort, floatCompare = 0;

  if ((a == NULL) && (b != NULL))
    {
      printf ("WARNING (1)\n");
      return (1);
    }
  else if ((a != NULL) && (b == NULL))
    {
      printf ("WARNING (2)\n");
      return (-1);
    }
  else if ((a == NULL) && (b == NULL))
    {
      printf ("WARNING (3)\n");
      return (0);
    }

  if (columnSort == HOST_DUMMY_IDX_VALUE)
    {
      int rc;

      /* Host name */
#ifdef MULTITHREADED
      accessMutex (&addressResolutionMutex, "yacmpFctn");
#endif
    if ((*a)->hostSymIpAddress[0] != '\0')
      rc = strcasecmp ((*a)->hostSymIpAddress, (*b)->hostSymIpAddress);
    else
      rc = strcasecmp ((*a)->ethAddressString, (*b)->ethAddressString);

#ifdef MULTITHREADED
    releaseMutex (&addressResolutionMutex);
#endif
    return(rc);
    }
  else if (columnSort == DOMAIN_DUMMY_IDX_VALUE)
    {
      int rc;

      yafillDomainName (*a); yafillDomainName (*b);

      rc = strcasecmp ((*a)->dotDomainName, (*b)->dotDomainName);
      if (rc == 0)
      return (strcasecmp((*a)->fullDomainName, (*b)->fullDomainName));
      else
	return rc;
    }

  oldColumnSort = columnSort;

  if (screenNumber == DUMMY_IDX_VALUE)
    {
      /* dirty trick */
      idx = columnSort-1;
      if (idx == -1)
	{
	  idx = 0;
	  columnSort = 0;
	}
      else
	columnSort = 1;
    }
  else
    idx = (screenNumber-MAX_NUM_PROTOS_SCREENS)*3;

  switch (columnSort)
    {
    case 0:
      if (sortSendMode)
	{
	  a_ = (*a)->bytesSent, b_ = (*b)->bytesSent;
	}
      else
	{
	  a_ = (*a)->bytesReceived, b_ = (*b)->bytesReceived;
	}
      break;

    case 1:
      if (sortSendMode)
	{
	  switch (screenNumber)
	    {
	    case 0:
	      a_ = (*a)->tcpSentLocally+(*a)->tcpSentRemotely;
	      b_ = (*b)->tcpSentLocally+(*b)->tcpSentRemotely;
	      break;
	    case 1:
	      a_ = (*a)->dlcSent, b_ = (*b)->dlcSent;
	      break;
	    case 2:
	      a_ = (*a)->arp_rarpSent, b_ = (*b)->arp_rarpSent;
	      break;
	    case 3:
	      a_ = (*a)->netbiosSent, b_ = (*b)->netbiosSent;
	      break;
	    case MAX_NUM_PROTOS_SCREENS:
	      fa_ = (*a)->actualSentThpt, fb_ = (*b)->actualSentThpt, floatCompare = 1;
	      break;
	    default:
	      if (idx < numIpProtosToMonitor)
		{
		  a_ = (*a)->protoIPTrafficInfos[idx].sentLocally
		    +(*a)->protoIPTrafficInfos[idx].sentRemotely;
		  b_ = (*b)->protoIPTrafficInfos[idx].sentLocally
		    +(*b)->protoIPTrafficInfos[idx].sentRemotely;
		}
	      else
		{
		  int i;

		  a_ = 0, b_ = 0;

		  for(i=0; i<numIpProtosToMonitor; i++)
		    {
		      a_ += (*a)->protoIPTrafficInfos[i].sentLocally
			+(*a)->protoIPTrafficInfos[i].sentRemotely;
		      b_ += (*b)->protoIPTrafficInfos[i].sentLocally
			+(*b)->protoIPTrafficInfos[i].sentRemotely;
		    }

		  if ((*a)->bytesSent > a_)
		    a_ = (*a)->bytesSent-a_;
		  else
		    a_ = 0;

		  if ((*b)->bytesSent > b_)
		    b_ = (*b)->bytesSent-b_;
		  else
		    b_ = 0;
		}
	    }
	}
      else
	{
	  switch (screenNumber)
	    {
	    case 0:
	      a_ = (*a)->tcpReceivedLocally+(*a)->tcpReceivedFromRemote;
	      b_ = (*b)->tcpReceivedLocally+(*b)->tcpReceivedFromRemote;
	      break;

	    case 1:
	      a_ = (*a)->dlcReceived, b_ = (*b)->dlcReceived;
	      break;

	    case 2:
	      a_ = (*a)->arp_rarpReceived, b_ = (*b)->arp_rarpReceived;
	      break;

	    case 3:
	      a_ = (*a)->netbiosReceived, b_ = (*b)->netbiosReceived;
	      break;

	    case MAX_NUM_PROTOS_SCREENS:
	      fa_ = (*a)->actualRcvdThpt,
		fb_ = (*b)->actualRcvdThpt, floatCompare = 1;
	      break;

	    default:
	      if (idx < numIpProtosToMonitor)
		{
		  a_ = (*a)->protoIPTrafficInfos[idx].receivedLocally
		    +(*a)->protoIPTrafficInfos[idx].receivedFromRemote;
		  b_ = (*b)->protoIPTrafficInfos[idx].receivedLocally
		    +(*b)->protoIPTrafficInfos[idx].receivedFromRemote;
		}
	      else
		{
		  int i;

		  a_ = 0, b_ = 0;

		  for (i=0; i<numIpProtosToMonitor; i++)
		    {
		      a_ += (*a)->protoIPTrafficInfos[i].receivedLocally
			+(*a)->protoIPTrafficInfos[i].receivedFromRemote;
		      b_ += (*b)->protoIPTrafficInfos[i].receivedLocally
			+(*b)->protoIPTrafficInfos[i].receivedFromRemote;
		    }

		  if ((*a)->bytesReceived > a_)
		    a_ = (*a)->bytesReceived-a_;
		  else
		    a_ = 0;

		  if ((*b)->bytesReceived > b_)
		    b_ = (*b)->bytesReceived-b_;
		  else
		    b_ = 0;

		  /*
		    printf("=>%d (%s)<=>%d (%s)<=\n",
		    (int)a_, (*a)->hostSymIpAddress,
		    (int)b_, (*b)->hostSymIpAddress);
		  */
		}
	    }
	}
      break;

    case 2:
    if (sortSendMode)
      {
	switch (screenNumber)
	  {
	  case 0:
	    a_ = (*a)->udpSentLocally +(*a)->udpSentRemotely;
	    b_ = (*b)->udpSentLocally +(*b)->udpSentRemotely;
	    break;

	  case 1:
	    a_ = (*a)->ipxSent, b_ = (*b)->ipxSent;
	    break;

	  case 2:
	    a_ = (*a)->appletalkSent, b_ = (*b)->appletalkSent;
	    break;

	  case 3:
	    a_ = (*a)->igmpSent, b_ = (*b)->igmpSent;
	    break;

	  case MAX_NUM_PROTOS_SCREENS:
	    fa_ = (*a)->averageSentThpt,
	      fb_ = (*b)->averageSentThpt, floatCompare = 1;
	    break;

	  default:
	    if(++idx < numIpProtosToMonitor)
	      {
		a_ = (*a)->protoIPTrafficInfos[idx].sentLocally
		  +(*a)->protoIPTrafficInfos[idx].sentRemotely;
		b_ = (*b)->protoIPTrafficInfos[idx].sentLocally
		  +(*b)->protoIPTrafficInfos[idx].sentRemotely;
	      }
	    else
	      {
		int i;

		a_ = 0, b_ = 0;

		for (i=0; i<numIpProtosToMonitor; i++)
		  {
		    a_ += (*a)->protoIPTrafficInfos[i].sentLocally
		      +(*a)->protoIPTrafficInfos[i].sentRemotely;
		    b_ += (*b)->protoIPTrafficInfos[i].sentLocally
		      +(*b)->protoIPTrafficInfos[i].sentRemotely;
		  }

		if((*a)->bytesSent > a_)
		  a_ = (*a)->bytesSent-a_;
		else
		  a_ = 0;

		if((*b)->bytesSent > b_)
		  b_ = (*b)->bytesSent-b_;
		else
		  b_ = 0;
	      }
	  }
      }
    else
      {
	switch (screenNumber)
	  {
	  case 0:
	    a_ = (*a)->udpReceivedLocally +(*a)->udpReceivedFromRemote;
	    b_ = (*b)->udpReceivedLocally +(*b)->udpReceivedFromRemote;
	    break;

	  case 1:
	    a_ = (*a)->ipxReceived, b_ = (*b)->ipxReceived;
	    break;

	  case 2:
	  a_ = (*a)->appletalkReceived, b_ = (*b)->appletalkReceived;
	  break;

	  case 3:
	    a_ = (*a)->igmpReceived, b_ = (*b)->igmpReceived;
	    break;

	  case MAX_NUM_PROTOS_SCREENS:
	    fa_ = (*a)->averageRcvdThpt,
	      fb_ = (*b)->averageRcvdThpt, floatCompare = 1;
	    break;

	  default:
	    if(++idx < numIpProtosToMonitor)
	      {
		a_ = (*a)->protoIPTrafficInfos[idx].receivedLocally
		  +(*a)->protoIPTrafficInfos[idx].receivedFromRemote;
		b_ = (*b)->protoIPTrafficInfos[idx].receivedLocally
		  +(*b)->protoIPTrafficInfos[idx].receivedFromRemote;
	      }
	    else
	      {
		int i;

		a_ = 0, b_ = 0;

		for (i=0; i<numIpProtosToMonitor; i++)
		  {
		    a_ += (*a)->protoIPTrafficInfos[i].receivedLocally
		      +(*a)->protoIPTrafficInfos[i].receivedFromRemote;
		    b_ += (*b)->protoIPTrafficInfos[i].receivedLocally
		      +(*b)->protoIPTrafficInfos[i].receivedFromRemote;
		  }

		if((*a)->bytesReceived > a_)
		  a_ = (*a)->bytesReceived-a_;
		else
		  a_ = 0;

		if((*b)->bytesReceived > b_)
		  b_ = (*b)->bytesReceived-b_;
		else
		  b_ = 0;
	      }
	  }
      }
    break;

    case 3:
      if(sortSendMode)
	{
	  switch(screenNumber)
	    {
	    case 0:
	      a_ = (*a)->icmpSent, b_ = (*b)->icmpSent;
	      break;

	    case 1:
	      a_ = (*a)->decnetSent, b_ = (*b)->decnetSent;
	      break;

	    case 2:
	      a_ = (*a)->ospfSent, b_ = (*b)->ospfSent;
	      break;

	    case 3:
	      a_ = (*a)->osiSent, b_ = (*b)->osiSent;
	      break;

	    case 4:
	      a_ = (*a)->qnxSent, b_ = (*b)->qnxSent;
	      break;

	    case 5: /* MAX_NUM_PROTOS_SCREENS: */
	      fa_ = (*a)->peakSentThpt,
		fb_ = (*b)->peakSentThpt, floatCompare = 1;
	      break;

	    default:
	      idx+=2;
	      if (idx < numIpProtosToMonitor)
		{
		  a_ = (*a)->protoIPTrafficInfos[idx].sentLocally
		    +(*a)->protoIPTrafficInfos[idx].sentRemotely;
		  b_ = (*b)->protoIPTrafficInfos[idx].sentLocally
		    +(*b)->protoIPTrafficInfos[idx].sentRemotely;
		}
	      else
		{
		  int i;

		  a_ = 0, b_ = 0;

		  for (i=0; i<numIpProtosToMonitor; i++)
		    {
		      a_ += (*a)->protoIPTrafficInfos[i].sentLocally
			+(*a)->protoIPTrafficInfos[i].sentRemotely;
		      b_ += (*b)->protoIPTrafficInfos[i].sentLocally
			+(*b)->protoIPTrafficInfos[i].sentRemotely;
		    }

		  if((*a)->bytesSent > a_)
		    a_ = (*a)->bytesSent-a_;
		  else
		    a_ = 0;

		  if((*b)->bytesSent > b_)
		    b_ = (*b)->bytesSent-b_;
		  else
		    b_ = 0;
		}
	    }
	}
      else
	{
	  switch (screenNumber)
	    {
	    case 0:
	      a_ = (*a)->icmpReceived, b_ = (*b)->icmpReceived;
	      break;
	    case 1:
	      a_ = (*a)->decnetReceived, b_ = (*b)->decnetReceived;
	      break;
	    case 2:
	      a_ = (*a)->ospfReceived, b_ = (*b)->ospfReceived;
	      break;
	    case 3:
	      a_ = (*a)->osiReceived, b_ = (*b)->osiReceived;
	      break;
	    case MAX_NUM_PROTOS_SCREENS:
	      fa_ = (*a)->peakRcvdThpt,
		fb_ = (*b)->peakRcvdThpt, floatCompare = 1;
	      break;
	    default:
	      idx+=2;
	      if(idx < numIpProtosToMonitor)
		{
		  a_ = (*a)->protoIPTrafficInfos[idx].receivedLocally
		    +(*a)->protoIPTrafficInfos[idx].receivedFromRemote;
		  b_ = (*b)->protoIPTrafficInfos[idx].receivedLocally
		    +(*b)->protoIPTrafficInfos[idx].receivedFromRemote;
		}
	      else
		{
		  int i;

		  a_ = 0, b_ = 0;

		  for (i=0; i<numIpProtosToMonitor; i++)
		    {
		      a_ += (*a)->protoIPTrafficInfos[i].receivedLocally
			+(*a)->protoIPTrafficInfos[i].receivedFromRemote;
		      b_ += (*b)->protoIPTrafficInfos[i].receivedLocally
			+(*b)->protoIPTrafficInfos[i].receivedFromRemote;
		    }

		  if((*a)->bytesReceived > a_)
		    a_ = (*a)->bytesReceived-a_;
		  else
		    a_ = 0;

		  if((*b)->bytesReceived > b_)
		    b_ = (*b)->bytesReceived-b_;
		  else
		    b_ = 0;
		}
	    }
	  break;
	}
      break;

    case 4:
      if(sortSendMode)
	switch(screenNumber)
	  {
	  case 0:
	    a_ = (*a)->qnxSent, b_ = (*b)->qnxSent;
	    break;
	  default:
	    fa_ = (*a)->actualSentPktThpt, fb_ = (*b)->actualSentPktThpt, floatCompare = 1;
	    break;
	  }
      else
	switch(screenNumber)
	  {
	  case 0:
	    a_ = (*a)->qnxReceived, b_ = (*b)->qnxReceived;
	    break;
	  case MAX_NUM_PROTOS_SCREENS:
	    fa_ = (*a)->actualRcvdPktThpt, fb_ = (*b)->actualRcvdPktThpt, floatCompare = 1;
	    break;
	  }
      break;

    case 5:
      if (sortSendMode)
	switch (screenNumber)
	  {
	  case 0:
	    a_ = (*a)->otherSent, b_ = (*b)->otherSent;
	    break;
	  case MAX_NUM_PROTOS_SCREENS:
	    fa_ = (*a)->averageSentPktThpt, fb_ = (*b)->averageSentPktThpt, floatCompare = 1;
	    break;
	  }
      break;

    case 6:
      if(sortSendMode)
	fa_ = (*a)->peakSentPktThpt, fb_ = (*b)->peakSentPktThpt, floatCompare = 1;
      else
	fa_ = (*a)->peakRcvdPktThpt, fb_ = (*b)->peakRcvdPktThpt, floatCompare = 1;
    }

  columnSort = oldColumnSort;

  if(floatCompare == 0)
    {
      if (a_ < b_)
	{
	  return (1);
	}
      else if (a_ > b_)
	{
	  return (-1);
	}
      else
	{
	  return (0);
	}
    }
  else
    {
      if (fa_ < fb_)
	{
	  return (1);
	}
      else if (fa_ > fb_)
	{
	  return (-1);
	}
      else
	{
	  return (0);
	}
    }
}


static void getProtocolDataSent (TrafficCounter * c, TrafficCounter * d,
				 TrafficCounter * e, HostTraffic * el)
{
  int idx;

  switch (screenNumber)
    {
    case 0:
      (*c) = el->tcpSentLocally + el->tcpSentRemotely;
      (*d) = el->udpSentLocally + el->udpSentRemotely;
      (*e) = el->icmpSent;
      break;

    case 1:
      (*c) = el->dlcSent;
      (*d) = el->ipxSent;
      (*e) = el->decnetSent;
      break;

    case 2:
      (*c) = el->arp_rarpSent;
      (*d) = el->appletalkSent;
      (*e) = el->ospfSent;
      break;

    case 3:
      (*c) = el->netbiosSent;
      (*d) = el->igmpSent;
      (*e) = el->osiSent;
      break;

    case 4:
      (*c) = el->qnxSent;
      (*d) = el->otherSent;
      (*e) = 0;
      break;

    default:
      idx = (screenNumber-MAX_NUM_PROTOS_SCREENS)*3;
      if(idx < numIpProtosToMonitor)
	(*c) = el->protoIPTrafficInfos[idx].sentLocally
	  + el->protoIPTrafficInfos[idx].sentRemotely;
      else
	(*c) = 0;

      ++idx;
      if(idx < numIpProtosToMonitor)
	(*d) = el->protoIPTrafficInfos[idx].sentLocally
	  + el->protoIPTrafficInfos[idx].sentRemotely;
      else
	(*d) = 0;

      ++idx;
      if(idx < numIpProtosToMonitor)
	(*e) = el->protoIPTrafficInfos[idx].sentLocally
	  + el->protoIPTrafficInfos[idx].sentRemotely;
      else
	(*e) = 0;
    }
}


static void getProtocolDataReceived (TrafficCounter * c, TrafficCounter * d,
				     TrafficCounter * e, HostTraffic * el)
{
  int idx;

  switch (screenNumber)
    {
    case 0:
      (*c) = el->tcpReceivedLocally + el->tcpReceivedFromRemote;
      (*d) = el->udpReceivedLocally + el->udpReceivedFromRemote;
      (*e) = el->icmpReceived;
      break;

    case 1:
      (*c) = el->dlcReceived;
      (*d) = el->ipxReceived;
      (*e) = el->decnetReceived;
      break;

    case 2:
      (*c) = el->arp_rarpReceived;
      (*d) = el->appletalkReceived;
      (*e) = el->ospfReceived;
      break;

    case 3:
      (*c) = el->netbiosReceived;
      (*d) = el->igmpReceived;
      (*e) = el->osiReceived;
      break;

    case 4:
      (*c) = el->qnxReceived;
      (*d) = el->otherReceived;
      (*e) = 0;
      break;

    default:
      idx = (screenNumber-MAX_NUM_PROTOS_SCREENS)*3;
      if(idx < numIpProtosToMonitor)
	(*c) = el->protoIPTrafficInfos[idx].receivedLocally
	  + el->protoIPTrafficInfos[idx].receivedFromRemote;
      else
	(*c) = 0;

      ++idx;
      if (idx < numIpProtosToMonitor)
	(*d) = el->protoIPTrafficInfos[idx].receivedLocally
	  + el->protoIPTrafficInfos[idx].receivedFromRemote;
      else
	(*d) = 0;

      ++idx;
      if(idx < numIpProtosToMonitor)
	(*e) = el->protoIPTrafficInfos[idx].receivedLocally
	  + el->protoIPTrafficInfos[idx].receivedFromRemote;
      else
	(*e) = 0;
    }
}


static char * cgetHostName (HostTraffic * el, short cutName)
{
  static char buf[5][80];
  char *tmpStr;
  static short bufIdx=0;

  if (broadcastHost (el))
    return ("broadcast");

  bufIdx = (bufIdx + 1) % 5;

#ifdef MULTITHREADED
  accessMutex (&addressResolutionMutex, "cgetHostName");
#endif

  switch (numericFlag)
    {
    case 0: /* symbolic */
      tmpStr = el->hostSymIpAddress;

      if(tmpStr == NULL)
	{
	  /* The DNS is still getting the entry name */
	  if (el->hostNumIpAddress[0] == '\0')
	    strncpy (buf[bufIdx], el->hostNumIpAddress, sizeof(buf[bufIdx]));
	  else
	    strncpy (buf[bufIdx], el->ethAddressString, sizeof(buf[bufIdx]));
	}
      else if (tmpStr[0] != '\0')
	{
	  strncpy (buf[bufIdx], tmpStr, sizeof(buf[bufIdx]));
	  if (cutName)
	    {
	      int i;

	      for (i=0; buf[bufIdx][i] != '\0'; i++)
		if ((buf[bufIdx][i] == '.')
		    && (!(isdigit(buf[bufIdx][i-1])
			  && isdigit(buf[bufIdx][i+1]))))
		  {
		    buf[bufIdx][i] = '\0';
		    break;
		  }
	    }
	}
      else
	strcpy (buf[bufIdx], el->ethAddressString);
      break;

    case 1:   /* numeric */
      if (el->hostNumIpAddress[0] != '\0')
	strcpy (buf[bufIdx], el->hostNumIpAddress);
      else
	strcpy (buf[bufIdx], el->ethAddressString);
      break;

    case 2: /* ethernet address */
      if(el->ethAddressString[0] != '\0')
	strcpy(buf[bufIdx], el->ethAddressString);
      else
	strcpy(buf[bufIdx], el->hostNumIpAddress);
      break;

    case 3: /* ethernet constructor */
      if (el->ethAddressString[0] != '\0')
	strcpy(buf[bufIdx], getVendorInfo(el->ethAddress, 0));
      else
	strcpy (buf[bufIdx], el->hostNumIpAddress);
      break;
    }

#ifdef MULTITHREADED
  releaseMutex(&addressResolutionMutex);
#endif

  return (buf[bufIdx]);
}



/*
 * Shamelessly ripped from ntop's report.c source file
 * and adapted to do the job.
 *
 * 1. copy from hostTraffic to dstTable
 * 2. sort dstTable
 * 3. print
 */
static void cHostsTraffic (int current, int reportType,
			   int sortedColumn, int revertOrder)
{
  u_int idx, numEntries = 0;
  int printedEntries = 0;
  HostTraffic * el;
  HostTraffic * dstTable [HASHNAMESIZE];

  char buf [BUF_SIZE] = {0};
  char buf2[BUF_SIZE] = {0};
  char buf1 [80] = {0};

  float sentPercent = 0, rcvdPercent = 0;


  memset (dstTable, 0, HASHNAMESIZE * sizeof (HostTraffic *));


  for (idx = 1; idx < device [current] . actualHashSize; idx ++)
    {
      /*
       * Skip unused entries.
       */
      if (! (el = device [current] . hash_hostTraffic [idx]))
	continue;


      if (! broadcastHost (el))
	{
	  if (localAddrFlag)
	    {
	      if (subnetPseudoLocalHost (el) && el->hostNumIpAddress[0] != '\0')
		{
		  if (! idleFlag)
		    {
		      if ((sortSendMode && el->bytesSent > 0) ||
			  (! sortSendMode && el->bytesReceived > 0))
			{
			  dstTable [numEntries ++] = el;
			}
		    }
		  else if (el->lastBytesSent != el->bytesSent ||
			   el->lastBytesReceived != el->bytesReceived)
		    {
		      if (reportType == 1 && el->hostNumIpAddress[0] == '\0')
			continue;
		      dstTable [numEntries ++] = el;
		    }
		}
	    }
	  else
	    {
	      if (! idleFlag)
		{
		  if ((sortSendMode && el->bytesSent > 0) ||
		      (! sortSendMode && el->bytesReceived > 0))
		    {
		      if (reportType == 1 && el->hostNumIpAddress[0] == '\0')
			continue;
		      dstTable [numEntries ++] = el;
		    }
		}
	      else if (el->lastBytesSent != el->bytesSent ||
		       el->lastBytesReceived != el->bytesReceived)
		{
		  if (reportType == 1 && el->hostNumIpAddress[0] == '\0')
		    continue;
		  dstTable [numEntries ++] = el;
		}
	    }
	}
    }


  /*
   * Done! We have a fresh table
   */
  if (numEntries > 0)
    {
      /*
       * The switch below is needed to:
       * - sort data according to the selected column
       * - 'recycle' (somebody would call this "code reuse") the cmpFctn function
       */

      if (sortedColumn == HOST_DUMMY_IDX_VALUE)
	columnSort = HOST_DUMMY_IDX_VALUE; /* Host name */
      else if (sortedColumn == DOMAIN_DUMMY_IDX_VALUE)
	columnSort = DOMAIN_DUMMY_IDX_VALUE; /* domain name */
      else if (reportType == 0 /* Interactive mode */)
	{
	  switch (sortedColumn)
	    {
	    case 0:
	      /* Nothing to do */
	      break;
	    case 1: /* TCP */
	      screenNumber = 0, columnSort = 1;
	      break;
	    case 2: /* UDP */
	      screenNumber = 0, columnSort = 2;
	      break;
	    case 3: /* ICMP */
	      screenNumber = 0, columnSort = 3;
	      break;
	    case 4: /* DLC */
	      screenNumber = 1, columnSort = 1;
	      break;
	    case 5: /* IPX */
	      screenNumber = 1, columnSort = 2;
	      break;
	    case 6: /* Decnet */
	      screenNumber = 1, columnSort = 3;
	      break;
	    case 7: /* (R)ARP */
	      screenNumber = 2, columnSort = 1;
	      break;
	    case 8: /* AppleTalk */
	      screenNumber = 2, columnSort = 2;
	      break;
	    case 9: /* OSPF */
	      screenNumber = 2, columnSort = 3;
	      break;
	    case 10: /* NetBios */
	      screenNumber = 3, columnSort = 1;
	      break;
	    case 11: /* IGMP */
	      screenNumber = 3, columnSort = 2;
	      break;
	    case 12: /* OSI */
	      screenNumber = 3, columnSort = 3;
	      break;
	    case 13: /* QNX */
	      screenNumber = 0, columnSort = 4;
	      break;
	    case 14: /* Other */
	      screenNumber = 0, columnSort = 5;
	      break;
	    }
	}
      else if (reportType == 1)
	{
	  screenNumber = DUMMY_IDX_VALUE /* dirty trick */, columnSort = sortedColumn;
	}
      else if (reportType == 2) /* Thpt */
	{
	  if (sortedColumn == 0)
	    sortedColumn = 1;
	  screenNumber = MAX_NUM_PROTOS_SCREENS /* dirty trick */, columnSort = sortedColumn;
	}


      /*
       * Sort the temporary table now.
       */
      quicksort (dstTable, numEntries, sizeof (HostTraffic *), yacmpFctn);


      /*
       * Avoid large printing on small screeens.
       */
      if (numEntries > (u_int) (LINES - 4))
	numEntries = (u_int) (LINES - 4);


      for (idx = 0; idx < numEntries; idx ++)
	{
	  if (revertOrder)
	    el = dstTable [numEntries - idx - 1];
	  else
	    el = dstTable [idx];

	  if (el)
	    {
	      char * tmpName;

	      if (percentMode == 1)
		{
		  sentPercent = (100 * (float) el->bytesSent) / device[current].ethernetBytes;
		  rcvdPercent = (100 * (float) el->bytesReceived) / device[current].ethernetBytes;
		}

	      tmpName = cgetHostName (el, 0);

	      if (percentMode == 1)
		{
		  float a, b;
		  TrafficCounter c, d, e;

		  /* % mode */
		  if (! sortSendMode)
		    {
		      a = rcvdPercent;
		      b = sentPercent;
		      getProtocolDataReceived (&c, &d, &e, el);
		    }
		  else
		    {
		      a = rcvdPercent;
		      b = sentPercent;
		      getProtocolDataSent (&c, &d, &e, el);
		    }

		  sprintf (buf, "%-26.26s%c %8.1f  %8.1f  %9s %9s %8s",
			   tmpName,
			   cformatStatus (el),
			   a, b,
			   cformatBytes (c),
			   cformatBytes (d),
			   cformatBytes (e));
		}
	      else
		{
		  TrafficCounter a, b, c, d, e;

		  a = el -> bytesReceived, b = el -> bytesSent;

		  if (! sortSendMode)
		    getProtocolDataReceived (&c, &d, &e, el);
		  else
		    getProtocolDataSent (&c, &d, &e, el);

		  if (! strcmp (tmpName, "0.0.0.0") || tmpName[0] == '\0')
		    tmpName = el -> ethAddressString;

		  if (percentMode == 0)
		    sprintf (buf, "%-26.26s%c %9s %9s %9s %9s %8s",
			     tmpName, cformatStatus (el),
			     cformatBytes (a),
			     cformatBytes (b),
			     cformatBytes (c),
			     cformatBytes (d),
			     cformatBytes (e));
		  else /* Throughput */
		    {
		      sprintf (buf, "%-26.26s%c %9s %9s %9s %9s %8s",
			       tmpName, cformatStatus (el),
			       cformatThroughput (el -> actualSentThpt),
			       cformatThroughput (el -> actualRcvdThpt),
			       cformatBytes (c),
			       cformatBytes (d),
			       cformatBytes (e));
		    }
		}


	      /*
	       * avoid long lines on small screens
	       */
	      buf [COLS-2] = '\0';
	      mvprintw (3 + idx, 1, buf);

	      if (percentMode == 1)
		{
		  mvaddch (3 + idx, 37, '%');
		  mvaddch (3 + idx, 47, '%');
		}
	    }
	  printedEntries ++;
	}
    }
  else
    idx = 0;

  /* First iteration */
  sprintf (buf, "%-80s", "");
  for (idx += 3; idx < (u_int) LINES; idx ++)
    mvprintw (idx, 1, buf);

  sprintf (buf, "%lu Pkts/%s [IP %s/Other %s]",
	   (unsigned long) device [current] . ethernetPkts,
	   cformatBytes (device [current] . ethernetBytes),
	   cformatBytes (device [current] . ipBytes),
	   cformatBytes (device [current] . ethernetBytes - device [current] . ipBytes));

  sprintf (buf1, "Thpt: %.11s/%-.11s",
	   cformatThroughput (device [current] . actualThpt),
	   cformatThroughput (device [current] . peakThroughput));

  sprintf (buf2, "%-50.50s %27.27s", buf, buf1);
  buf2 [78] = '\0';
  buf2 [COLS - 2] = '\0'; /* avoid long lines on small screens */

  mvprintw (1, 0, buf2);
  refresh ();

  lastRefreshTime = actTime;
}


/*
 * The help() function while running the 'top' command.
 *
 * Shamelessly ripped from ntop's ntop.c source file
 * and adapted to do the job.
 */
static void top_help (char * interface)
{
  char buf [BUFSIZ];
  int row = 0;

  /* clear the screen */
  clrscr ();
  refresh ();


  sprintf (buf, "%s", intop_banner (interface));
  mvprintw (row ++, 0, buf);
  row += 2;

  sprintf (buf, "While %s is running interactively, the following keys are active:",
	   progname);

  mvprintw (row ++, 0, buf);
  sprintf (buf, "'q' - quit the curses mode");

  mvprintw (row ++, 7, buf);
  sprintf (buf, "'c' - clear the screen");

  mvprintw (row ++, 7, buf);
  sprintf (buf, "'l' - toggle hosts display (local subnet <-> all)");

  mvprintw (row ++, 7, buf);
  sprintf (buf, "'d' - toggle idle (idle <-> send/receive)");

  mvprintw (row ++, 7, buf);
  sprintf (buf, "'p' - toggle traffic values (bytes <-> %% <-> thpt)");

  mvprintw (row ++, 7, buf);
  sprintf (buf, "'r' - reset statistics");

  mvprintw (row ++, 7, buf);
  sprintf (buf, "'n' - toggle address format (num <-> sym <-> MAC <-> Nw Board Manufact.)");

  mvprintw (row ++, 7, buf);
  sprintf (buf, "'t' - toggle sort (sent <-> received)");

  mvprintw (row ++, 7, buf);
  sprintf (buf, "'y' - toggle columns sort");

  mvprintw (row ++, 7, buf);
  sprintf (buf, "'h' - show this help");

  mvprintw (row ++, 7, buf);
  sprintf (buf, "' ' - toggle protocol");

  mvprintw (row ++, 7, buf);

  mvprintw (++ row, 0, "Strike a key to continue...");
  refresh ();

  while (getchar () == EOF)
    ;
}


/*
 * Get a char from the standard input and process it.
 * Return:
 *   1 if, and only if, the 'q' (quit) key has been pressed
 *   0 if a valid key has been recognized
 *  -1 otherwise
 */
static int handleuserkey (char * interface)
{
  unsigned int c;
  int numScreens;
  int choice = 0;

  switch (c = getch ())
    {
    case 'q':
    case 'Q':
      /* done! */
      choice = 1;
      break;

    case 'h':
    case 'H':
      /* help screen */
      top_help (interface);
      break;

    case 'c':
    case 'C':
      /* screen refresh */
      clrscr ();
      refresh ();
      sleep (1);   /* just to give the user impression of cleaning */
      break;

    case 'l':
    case 'L':
      /* toggle host list */
      localAddrFlag = ! localAddrFlag;
      break;

    case 'd':
    case 'D':
      /* toggle idle hosts */
      idleFlag = ! idleFlag;
      break;

    case 'p':
    case 'P':
      /* toggle traffic percentage */
      percentMode = (percentMode + 1) % 3;
      break;

    case 'r':
    case 'R':
      resetStats ();
      break;

    case 'n':
    case 'N':
      /* toggle numeric <-> sym ... address format */
      numericFlag = (numericFlag + 1) % 4;
      break;

    case 't':
    case 'T':
      sortSendMode = ! sortSendMode;
      break;

    case 'y':
    case 'Y':
      columnSort = (columnSort + 1) % 4;
      break;

    case ' ': /* space */
      numScreens = ((numIpProtosToMonitor + 2 /* avoid round problems */) / 3);
      if (numIpProtosToMonitor % 3)
	numScreens ++;
      numScreens += 3; /* Protocol Screens */
      screenNumber = (screenNumber + 1) % numScreens;
      break;

    default:  /* unsupported key */
      choice = -1;
      break;
    }

  return (choice);
}


/*
 * Show top network usage, like top does for processes.
 */
int intop_top (int argc, char * argv [])
{
  /*
   * Notice the command name.
   */
  char * commandname = argv [0];

  int c;
#define USAGE(xxx) \
  printf ("Usage: %s [-h] [-i interface] [-r refreshrate]\n", xxx)
  char * optstring = "hi:r:";

  /*
   * Reserve here space for the local variables.
   */
  int done = 0;
  struct timeval timeout;
  fd_set readmask;
  int nfound;

  char * interface = NULL;
  int current;
  int rr = 5;         /* refresh rate in seconds */

  int choice = 0;

  optind = 0;
  optarg = NULL;

  /*
   * Parse command line options to the application via standard system calls.
   */
  while ((c = getopt (argc, argv, optstring)) != -1)
    {
      switch (c)
	{
	case 'h':
	  USAGE (commandname);
	  return (0);

	default:
	  USAGE (commandname);
	  return (-1);

	case 'i':
	  interface = optarg;
	  break;

	case 'r':
	  rr = atoi (optarg);
	  break;
	}
    }

  if (optind < argc)
    {
      printf ("\nWrong option(s): \" ");
      while (optind < argc)
	printf ("%s ", argv [optind ++]);
      printf ("\"\n");
      USAGE (commandname);
      printf ("\n");
      return (-1);
    }

  /*
   * Safe to play with the 'active' interface (if any)
   * in case no specific one was chosen by the user.
   */
  if (! interface && ! (interface = intop_interface (active)))
    {
      printf ("No interface is currently enabled for packet sniffing.\n");
      return (-1);
    }

  /*
   * Lookup for the given name in the table of enabled interfaces.
   */
  current = intop_lookup_interface (interface);
  if (current == -1)
    {
      printf ("%s: unknown interface %s.\n", commandname, interface);
      return (-1);
    }

  /*
   * Avoid to display no available information.
   */
  if (get_intop_flags (current) != INTERFACE_ENABLED)
    {
      printf ("%s: no interface is currently enabled for packet sniffing.\n",
	      commandname);
      return (-1);
    }

  /*
   * Go for fun!
   */


  /*
   * Avoid to be suspended until I have the time to write a good handler :-(
   */
  signal (SIGTSTP, SIG_IGN);

  /*
   * similar to ntop's initCurses().
   * I need a separate function to avoid the creation of a new
   * thread responsible for screen printing and refresh.
   */
  initscr();                 /* initialize the curses library */
  cbreak();                  /* take input chars one at a time, no wait for \n */
  noecho();                  /* don't echo input */
  nodelay(stdscr, TRUE);
  nonl();                    /* tell curses not to do NL->CR/NL on output */
  intrflush (stdscr, FALSE);
  keypad (stdscr, TRUE);
  attrset (A_NORMAL);

  /*
   * Initial startup.
   */
  clrscr ();
  intop_printHeader (interface);
  cHostsTraffic (intop_lookup_interface (interface), 0, 0, 0);
  refresh ();

  /*
   * Print and refresh the screen.
   * Handle user requests, until the quit command has been issued.
   */
  while (! done)
    {
      FD_ZERO (& readmask);
      FD_SET (fileno (stdin), & readmask);

      timeout . tv_sec  = rr;   /* wait max for refresh rate seconds */
      timeout . tv_usec = 0;

      nfound = select (fileno (stdin) + 1, & readmask, NULL, NULL, & timeout);

      switch (nfound)
	{
	case -1:
	  done = 1;
	  break;

	case 0:
	  clrscr ();
	  intop_printHeader (interface);
	  cHostsTraffic (intop_lookup_interface (interface), 0, 0, 0);
	  refresh ();
	  break;

	default:
	  switch (choice = handleuserkey (interface))
	    {
	    case 1:     /* quit key */
	      done = 1;
	      break;

	    case 0:    /* some valid key has been issued */
	      clrscr ();
	      intop_printHeader (interface);
	      cHostsTraffic (intop_lookup_interface (interface), 0, 0, 0);
	      refresh ();
	      break;

	    case -1: /* nothing to do */
	      break;
	    }
	  break;
	}
    }

 /*
  * Terminate the curses.
  */
  clrscr ();
  echo ();
  nl ();
  intrflush (stdscr, TRUE);
  endwin ();

  /*
   * Restart the suspend default behavoiur.
   */
  signal (SIGTSTP, SIG_DFL);


  return (0);
}


