%{
/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */
%}

%{
// prevent redefinition of YYSTYPE in parser.h
#define __PARSE_TAB_H

// these classes must be defined prior to YYSTYPE
#include "Fragment.h"
#include "BlockStmt.h"
#include "Symbol.h"
#include "Expr.h"
#include "FunctionDef.h"
#include "CallStmt.h"
#include "AsmStmt.h"
#include "LabeledStmt.h"
#include "DeclareStmt.h"

class Clause;
class LocationNode;

%}

%union {
	int			fInt;
	Fragment*	fFragment;
	Stmt*		fStmt;
	BlockStmt*	fBlock;
	Symbol*		fSymbol;
	char*		fString;
	Expr*		fExpr;
	FunctionDef*	fFunction;
	CallStmt*	fCall;
	AsmStmt*	fAsmStmt;
	Field*		fField;
	LabeledStmt*	fLabeledStmt;
	DeclareStmt*	fDeclareStmt;
	LocationNode*	fLocation;
}


%{
#include <stdlib.h>
#include "IfStmt.h"
#include "WhileStmt.h"
#include "AsmStmt.h"
#include "DoStmt.h"
#include "RepeatStmt.h"
#include "AssignStmt.h"
#include "TaskStmt.h"
#include "JumpStmt.h"
#include "SwitchStmt.h"
#include "ExprStmt.h"
#include "RCX_Constants.h"
#include "PreProc.h"
#include "parser.h"
#include "Program.h"
#include "parse_util.h"
#include "Program.h"
#include "IncDecExpr.h"
#include "AtomExpr.h"
#include "TypeExpr.h"
#include "Error.h"
#include "MonitorStmt.h"
#include "EventSrcExpr.h"
#include "AssignMathStmt.h"
#include "ForStmt.h"
#include "RelExpr.h"
#include "NegateExpr.h"
#include "LogicalExpr.h"

#define yylex()		(gPreProc->Get(yylval))

static LexLocation sSavedLoc;

%}


%left OR
%left AND
%left '|'
%left '^'
%left '&'
%left REL_EQ REL_NE
%left '<' '>'  REL_LE REL_GE
%left LEFT RIGHT
%left '-' '+'
%left '*' '/' '%'
%right UMINUS '~' INDIRECT '!'
%left INCDEC

%nonassoc ABS SIGN TYPE EVENT_SRC
%nonassoc LOWER_THAN_ELSE
%nonassoc ELSE

%nonassoc LOWER_THAN_EXPR_SHIFT
%nonassoc ')'

%token <fSymbol> ID
%token <fInt> NUMBER ASSIGN TASKOP JUMP TASK SUB INCDEC
%token <fString> STRING
%token REL_GE REL_LE REL_EQ REL_NE

%token PP_DEFINE PP_INCLUDE NL WS PP_ARG PP_ERROR PP_IFDEF PP_IF PP_ELSE PP_ELIF PP_ENDIF PP_UNDEF PP_PRAGMA PP_GLOM
%token IF WHILE DO FOR REPEAT SWITCH CASE DEFAULT MONITOR CATCH ACQUIRE
%token ASM
%token INT INLINE T_VOID T_CONST SENSOR TASKID
%token CTRUE CFALSE

%type <fStmt> stmt block var_item handler evt_handler opt_handler control_stmt expr_stmt misc_stmt opt_expr_stmt
%type <fDeclareStmt> var_decl
%type <fBlock> stmt_list var_list handler_list
%type <fInt> S arg_type
%type <fExpr> expr opt_expr
%type <fFunction> args arg_list
%type <fCall> params param_list
%type <fAsmStmt> asm_list
%type <fField> asm_item
%type <fLabeledStmt> label
%type <fLocation> loc

%{
static void yyerror(const char *msg);

%}


%%

S : unit_list	{ $$ = 0; }
	;


unit_list : unit_list unit
	|
	;

unit :	INT var_list ';'		{ gProgram->AddGlobalDecls($2); }
	|	TASK ID  opt_arg block	{ (void)new Fragment(kRCX_TaskFragment, $2, $4); }
	|	SUB ID opt_arg block	{ (void)new Fragment(kRCX_SubFragment, $2, $4); }
				
	|	INLINE ID  { Error(kDep_Inline, $2->GetKey()).RaiseLex();
			$<fFunction>$ = BeginFunction(new FunctionDef(), $2); } '{' stmt_list '}' { EndFunction($<fFunction>3, $5); }
	|	T_VOID ID '(' args ')' { BeginFunction($4, $2); } '{' stmt_list '}'  { EndFunction($4, $8); }
	;


opt_arg : '(' ')'
	|				{ Error(kDep_NoArgList).RaiseLex(); }
	;

args : arg_list
	| T_VOID	{ $$ = new FunctionDef(); }
	|			{ $$ = new FunctionDef(); }
	;

arg_list : arg_list ',' arg_type ID	{ $$ = $1; 	DefineArg($$, $4, $3); }
	| arg_type ID					{ $$ = new FunctionDef(); DefineArg($$,$2,$1); }
	;

arg_type : INT		{ $$ = FunctionDef::kIntegerArg; }
	| T_CONST INT 	{ $$ = FunctionDef::kConstantArg; }
	| INT '&'		{ $$ = FunctionDef::kReferenceArg; }
	| T_CONST INT '&'	{ $$ = FunctionDef::kConstRefArg; }
	| SENSOR		{ $$ = FunctionDef::kSensorArg; }
	;


var_list : var_list ',' var_item	{ $1->Add($3); $$ = $1; }
	| var_item						{ $$ = new BlockStmt(); $$->Add($1); }
	;

var_item : var_decl		{ $$ = $1; }
	| var_decl '=' expr	{ $$ = $1; $1->SetInitialValue($3); }
	;

var_decl : ID loc { $$ = MakeDeclareStmt($1, $2, 0); }
	| ID loc '[' expr ']'{ $$ = MakeDeclareStmt($1, $2, $4); }
	;

block : '{' { BeginScope(); } stmt_list '}'		{ $$ = EndScope($3); }
	;

stmt_list : stmt_list  stmt 	{ $1->Add($2); $$ = $1; }
	|							{ $$ = new BlockStmt(); }
	;

stmt
	:	';'				{ $$ = new BlockStmt(); }
	|	block
	|	label stmt		{ $$ = $1; $1->SetStmt($2); }
	|	error ';'		{  yyerrok; $$ = new BlockStmt(); }
	|	control_stmt
	|	misc_stmt
	|	expr_stmt ';'
	;

expr_stmt
	:	expr					{ $$ = new ExprStmt($1); }
	|	expr ASSIGN expr		{ CheckLValue($1); $$ = new AssignMathStmt($1, (RCX_VarCode)$2, $3); }
	|	expr '=' expr			{ CheckLValue($1); $$ = new AssignStmt($1, $3); }
	;

control_stmt
	:	WHILE '(' expr ')' stmt					{ $$ = new WhileStmt($3, $5); }
	|	DO stmt WHILE '(' expr ')' ';'		{ $$ = new DoStmt($5, $2); }
	|	REPEAT '(' expr ')' stmt				{ $$ = new RepeatStmt($3, $5); }
	|	SWITCH '(' expr ')' stmt				{ $$ = new SwitchStmt($3, $5); }
	|	IF	'(' expr ')' stmt	%prec LOWER_THAN_ELSE	{ $$ = new IfStmt($3, $5); }
	|	IF '(' expr ')' stmt ELSE stmt		{ $$ = new IfStmt($3, $5, $7); }
	|	MONITOR loc '(' expr ')' block handler_list	{ $$ = new MonitorStmt($4, $6, $7, $2->GetLoc()); delete $2; }
	|	ACQUIRE loc '(' expr ')' block opt_handler	{ $$ = MakeAcquireStmt($4, $6, $7, $2); }
	|	FOR '(' opt_expr_stmt ';' opt_expr ';' opt_expr_stmt ')' stmt	{ $$ = new ForStmt($3, $5, $7, $9); }
	;
	
misc_stmt
	:	ASM '{' asm_list '}' ';'	{ $$ = $3; }
	|	saveloc JUMP ';'			{ $$ = new JumpStmt($2, sSavedLoc); }
	|	TASKOP saveloc ID ';'  		{ $$ = new TaskStmt((UByte)$1, $3, sSavedLoc); }
	|	loc ID '(' params ')' ';'	{ $$ = $4; $4->SetName($2); $4->SetLocation($1->GetLoc()); delete $1; }
	|	INT	var_list ';'			{ $$ = $2; }
	;


opt_expr_stmt : expr_stmt
	|				{ $$ = 0; }
	;

opt_expr : expr
	| 	{ $$ = 0; }
	;


opt_handler : handler
	|	{ $$ = 0; }
	;


handler_list :	handler_list evt_handler	{ $1->Add($2); $$ = $1; }
	| 										{ $$ =  new BlockStmt(); }
	;
	
evt_handler : CATCH block		{ $$ = $2; }
	| CATCH loc '(' expr ')' block	{ $$ = MakeCatchStmt($4, $6, $2); }
	;
		
	
handler : CATCH block	{ $$ = $2; }
	;

label : saveloc CASE expr ':'	{ $$ = MakeCaseStmt($3, sSavedLoc); }
	  | saveloc DEFAULT ':'		{ $$ = new LabeledStmt(LabeledStmt::kDefaultValue, sSavedLoc); }
	  ;

params : param_list
	|				{ $$ = new CallStmt(); }
	;

param_list : param_list ',' expr	{ $$ = $1; $$->AddParam($3); }
	| expr							{ $$ = new CallStmt(); $$->AddParam($1); }
	;

expr : NUMBER saveloc		{ $$ = new AtomExpr(kRCX_ConstantType, $1, sSavedLoc); }
	|	expr '+' expr	{ $$ = MakeBinaryExpr($1, '+', $3); }
	|	expr '-' expr	{ $$ = MakeBinaryExpr($1, '-', $3); }
	|	expr '*' expr	{ $$ = MakeBinaryExpr($1, '*', $3); }
	|	expr '/' expr	{ $$ = MakeBinaryExpr($1, '/', $3); }
	|	expr '&' expr	{ $$ = MakeBinaryExpr($1, '&', $3); }
	|	expr '|' expr	{ $$ = MakeBinaryExpr($1, '|', $3); }
	|	expr '%' expr	{ $$ = MakeBinaryExpr($1, '%', $3); }
	|	expr LEFT expr	{ $$ = MakeBinaryExpr($1, LEFT, $3); }
	|	expr RIGHT expr	{ $$ = MakeBinaryExpr($1, RIGHT, $3); }
	|	expr '^' expr	{ $$ = MakeBinaryExpr($1, '^', $3); }
	|	expr REL_EQ expr	{ $$ = new RelExpr($1, RelExpr::kEqualTo, $3); }
	|	expr REL_LE expr	{ $$ = new RelExpr($1, RelExpr::kLessOrEqual, $3); }
	|	expr REL_GE expr	{ $$ = new RelExpr($1, RelExpr::kGreaterOrEqual, $3); }
	|	expr REL_NE expr	{ $$ = new RelExpr($1, RelExpr::kNotEqualTo, $3); }
	|	expr '<' expr	{ $$ = new RelExpr($1, RelExpr::kLessThan, $3); }
	|	expr '>' expr	{ $$ = new RelExpr($1, RelExpr::kGreaterThan, $3); }
	|	CTRUE saveloc			{ $$ = new AtomExpr(kRCX_ConstantType, 1, sSavedLoc); }
	|	CFALSE saveloc			{ $$ = new AtomExpr(kRCX_ConstantType, 0, sSavedLoc); }
	|	'!' expr		{ $$ = new NegateExpr($2); }
	|	expr AND expr	{ $$ = new LogicalExpr($1, LogicalExpr::kLogicalAnd, $3); }
	|	expr OR expr	{ $$ = new LogicalExpr($1, LogicalExpr::kLogicalOr, $3); }

	|	'-'	expr %prec UMINUS	{ $$ = MakeBinaryExpr(new AtomExpr(kRCX_ConstantType, 0, $2->GetLoc()), '-', $2); }
	|	'~' expr %prec UMINUS	{ $$ = MakeUnaryExpr('~', $2); }
	
	|	ABS '(' expr ')' 	{ $$ = MakeUnaryExpr(ABS, $3); }
	|	SIGN '(' expr ')'	{ $$ = MakeUnaryExpr(SIGN, $3); }
	
	| '(' expr ')'			{ $$ = $2; }
	| loc ID 				{ $$ = MakeVarExpr($2, $1); }
	| loc ID '[' expr ']'	{ $$ = MakeArrayExpr($2, $1, $4); }
	| '@' expr %prec UMINUS			{ $$ = MakeValueExpr($2); }
	
	| expr INCDEC				{ $$ = new IncDecExpr(GetLValue($1), $2, false, $1->GetLoc()); }
	| INCDEC expr %prec UMINUS	{ $$ = new IncDecExpr(GetLValue($2), $1, true, $2->GetLoc()); }
	
	| TYPE '(' expr ')'			{ $$ = new TypeExpr($3); }
	| EVENT_SRC '(' expr ')'	{ $$ = new EventSrcExpr($3); }
	| loc TASKID				{ $$ = MakeTaskIdExpr($1); }
	| INDIRECT '(' expr ')' '[' expr ']'		{ $$ = MakeIndirectExpr($3, $6); }
	;

saveloc :  %prec LOWER_THAN_EXPR_SHIFT		{ LexCurrentLocation(sSavedLoc); }
	;

loc : %prec LOWER_THAN_EXPR_SHIFT	{ $$ = new LocationNode(); }
	;
	
asm_list : asm_list ',' asm_item	{ $$ = $1; $1->Add($3); }
	| asm_item	{ $$ = new AsmStmt(); $$->Add($1); }
	|			{ $$ = new AsmStmt(); }
	;


asm_item : expr		{ $$ = MakeConstField($1); }
	| '&' expr	{ $$ = new EAField($2); }
	| '&' expr ':' expr	{ $$ = new EAField($2, GetConstantValue($4)); }
	;
	
%%


void yyerror(const char *msg)
{
	Error(kErr_Parser, msg).RaiseLex();
}


