/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>
#include "Program.h"
#include "RCX_Image.h"
#include "RCX_Link.h"
#include "Symbol.h"
#include "PreProc.h"
#include "parser.h"
#include "Macro.h"
#include "RCX_Cmd.h"
#include "RCX_Log.h"
#include "SRecord.h"
#include "AutoFree.h"
#include "DirList.h"
#include "Buffer.h"
#include "Error.h"
#include "Compiler.h"
#include "RCX_Disasm.h"

// use this to check for memory leaks in the compiler
//#define CHECK_LEAKS

#ifdef CHECK_LEAKS
#include <DebugNew.h>
#endif

// some win32 consoles are lame...use stdout instead of stderr for usage, etc.
#ifdef WIN32
#define STDERR stdout
#else
#define STDERR stderr
#endif

FILE *gErrorStream = stderr;

#define VERSION	"2.0.2"

#define kMaxPrintedErrors	10

typedef const char * const * ArgArray;

class Args
{
public:
			Args(int argc, ArgArray argv) : fStart(argv), fCurrent(argv), fEnd(argv+argc) {}

	int		Remain() const	{ return fEnd - fCurrent; }
	const char *Next()	{ return *fCurrent++; }
	int		NextInt()	{ return atoi(*fCurrent++); }
	
private:
	ArgArray	fStart;
	ArgArray	fCurrent;
	ArgArray	fEnd;
};


class AutoLink : public RCX_Link
{
public:
				AutoLink() : fSerialPort(0), fOpen(false) {}
				~AutoLink()	{ Close(); }

	RCX_Result	Open();
	void		Close();
	RCX_Result	Send(const RCX_Cmd *cmd, bool retry=true);
	
	void		SetSerialPort(const char *sp)	{ fSerialPort = sp; }

	bool		DownloadProgress(int soFar, int total);

private:
	const char*	fSerialPort;
	bool		fOpen;		
};



class MyCompiler : public Compiler, public ErrorHandler
{
public:
				MyCompiler()	 {}
				
	Buffer *GetBuffer(const char *name);

	void	AddError(const Error &e, const LexLocation *loc);
	void	AddDir(const char *dirspec)	{ fDirs.Add(dirspec); }

private:
	DirList	fDirs;
} gMyCompiler;



#define kMaxFirmware 65536
#define kOldIncludePathEnv	"NQCC_INCLUDE"
#define kNewIncludePathEnv	"NQC_INCLUDE"
#define kLowBattery	6600

#define kRCXFileExtension ".rcx"
#define kNQCFileExtension ".nqc"



// error codes in addition to RCX_Result codes
#define kUsageError	(kRCX_LastError - 1)
#define kQuietError	(kRCX_LastError - 2)

// codes for the actions
enum
{
	kFirstActionCode = 256,
	kDatalogCode = kFirstActionCode,
	kDatalogFullCode,
	kClearMemoryCode,
	kFirmwareCode,
	kNearCode,
	kFarCode,
	kWatchCode,
	kSleepCode,
	kRunCode,
	kProgramCode,
	kMessageCode,
	kRawCode,
	kCompileStdinCode
};

// these must be in the same order as the codes for the long options
static const char *sActionNames[] = {
	"datalog", "datalog_full", "clear", "firmware", "near", "far", "watch",
	"sleep", "run", "pgm", "msg", "raw", ""
};


static char *sTestArgs[]={"nqc", "-l", "::foo.nqc"};
//static char *sTestArgs[]={"nqc", "-v",  "-d", "test.nqc"};
//static char *sTestArgs[]={"nqc", "-firmware", "foo"};
//static char *sTestArgs[]={"nqc", "-l", "::regression:2.0a1:bugbot1.nqc"};


struct Request
{
	const char *fSourceFile;
	const char *fOutputFile;
	const char *fListFile;
	bool	fListing;
	bool	fDownload;
	int		fFlags;
};

static int GetActionCode(const char *arg);
static RCX_Result ProcessCommandLine(int argc, ArgArray argv);
static void PrintError(RCX_Result error, const char *filename = 0);
static void PrintUsage();
static void DefineMacro(const char *text);
static RCX_Result ProcessFile(const char *sourceFile, const Request &req);
static char *CreateFilename(const char *source, const char *oldExt, const char *newExt);
static int CheckExtension(const char *s1, const char *ext);
static RCX_Image *Compile(const char *sourceFile,  int flags);
static RCX_Result GenerateListing(RCX_Image *image, const char *filename);
static RCX_Result Download(RCX_Image *image);
static RCX_Result UploadDatalog(bool verbose);
static RCX_Result DownloadFirmware(const char *filename);
static RCX_Result SetWatch(const char *timeSpec);
static RCX_Result SetErrorFile(const char *filename);
static RCX_Result SendRawCommand(const char *text);
static RCX_Result ClearMemory();

AutoLink gLink;
bool	gTargetCM = false;

int main(int argc, char **argv)
{
	RCX_Result result;

	if (argc == 0)
	{
		// special case for debugging under the Metrowerks console
		argv = sTestArgs;
		argc = sizeof(sTestArgs) / sizeof(char *);
	}

	// add any default include paths
	gMyCompiler.AddDir(getenv(kOldIncludePathEnv));
	gMyCompiler.AddDir(getenv(kNewIncludePathEnv));
	
	result = ProcessCommandLine(argc, argv);
	gLink.Close();
	
	PrintError(result);

	if (gErrorStream != stderr &&
		gErrorStream != stdout)
		fclose(gErrorStream);
	
#ifdef DEBUG_NEW
	printf("%d allocations, %d total bytes, %d max\n", gDebugNewAllocCount, gDebugNewAllocCurr, gDebugNewAllocMax);
#endif
	
	return (result==kRCX_OK) ? 0 : -1;
}


RCX_Result ProcessCommandLine(int argc, ArgArray argv)
{
	bool optionsOK = true;
	bool fileProcessed = false;
	Request req = { 0, 0, 0, false, false, 0 };
	Args args(argc, argv);
	RCX_Result result = kRCX_OK;
	RCX_Cmd cmd;
	
	// pop initial arg
	(void)args.Next();
	
	while(args.Remain() && !RCX_ERROR(result))
	{
		const char* a=args.Next();
		bool isOption = (a[0] == '-');
		
#ifdef WIN32
		if (a[0]=='/') isOption = true;
#endif

		if (isOption)
		{
			int code = GetActionCode(a+1);

			if (code)
			{
				optionsOK = false;
			}
			else
			{
				if (!optionsOK) return kUsageError;
				code = a[1];
			}
			
			switch(code)
			{
				// options
				case '1':
					req.fFlags |= Compiler::kCompat_Flag;
					break;
				case 'D':
					DefineMacro(a+2);
					break;
				case 'E':
					result = SetErrorFile(a+2);
					break;
				case 'I':
					gMyCompiler.AddDir(a+2);
					break;
				case 'L':
					req.fListing = true;
					req.fListFile = a[2] ? a+2 : 0;
					break;
				case 'O':
					req.fOutputFile = a+2;
					break;
				case 'S':
					gLink.SetSerialPort(a+2);
					break;
				case 'U':
					Compiler::Get()->Undefine(a+2);
					break;
				case 'c':
					gLink.SetTarget(RCX_Link::kTarget_CyberMaster);
					gTargetCM = true;
					req.fFlags |= Compiler::kTargetCM_Flag;
					break;
				case 'd':
					req.fDownload = true;
					break;
				case 'l':
					req.fListing = true;
					break;
				case 'n':
					req.fFlags |= Compiler::kNoSysFile_Flag;
					break;
				case 't':
					if (!args.Remain()) return kUsageError;
					gLink.SetRxTimeout(args.NextInt());
					break;
				case 'v':
					gLink.SetVerbose(true);
					break;

				// these are deprecated
				case 'o':
					if (!args.Remain()) return kUsageError;
					req.fOutputFile = args.Next();
					break;
				case 'e':
					result = SetErrorFile("");
					break;
				case 's':
					if (!args.Remain()) return kUsageError;
					gLink.SetSerialPort(args.Next());
					break;

				// actions
				case kCompileStdinCode:
					result = ProcessFile(nil, req);
					fileProcessed = true;
					break;
				case kDatalogCode:
					result = UploadDatalog(false);
					break;
				case kDatalogFullCode:
					result = UploadDatalog(true);
					break;
				case kClearMemoryCode:
					result = ClearMemory();
					break;
				case kFirmwareCode:
					if (!args.Remain()) return kUsageError;
					result = DownloadFirmware(args.Next());
					break;
				case kNearCode:
					result = gLink.Send(cmd.Set(kRCX_IRModeOp, 0));
					break;
				case kFarCode:
					result = gLink.Send(cmd.Set(kRCX_IRModeOp, 1));
					break;
				case kWatchCode:
					if (!args.Remain()) return kUsageError;
					result = SetWatch(args.Next());
					break;
				case kSleepCode:
					if (!args.Remain()) return kUsageError;
					result = gLink.Send(cmd.Set(kRCX_AutoOffOp, (UByte)args.NextInt()));
					break;
				case kRunCode:
					result = gLink.Send(cmd.Set(kRCX_StartTaskOp, 0));
					break;
				case kProgramCode:
					if (!args.Remain()) return kUsageError;
					result = gLink.Send(cmd.Set(kRCX_SelectProgramOp, (UByte)(args.NextInt()-1)));
					break;
				case kMessageCode:
					if (!args.Remain()) return kUsageError;
					result = gLink.Send(cmd.Set(kRCX_Message, (UByte)(args.NextInt())), false);
					break;
				case kRawCode:
					if (!args.Remain()) return kUsageError;
					result = SendRawCommand(args.Next());					
					break;
				default:
					return kUsageError;
			}
		}
		else if (!fileProcessed)
		{
			result = ProcessFile(a, req);

#ifdef CHECK_LEAKS
			int firstAllocCount = gDebugNewAllocCount;
			Compiler::Get()->Reset();
			result = ProcessFile(a, req);
			printf("%d leaked allocations\n", gDebugNewAllocCount - firstAllocCount);
#endif
			optionsOK = false;
			fileProcessed = true;
		}
		else
			return kUsageError;
	}
	
	// check if we did anything (compile and/or actions)
	if (optionsOK) return kUsageError;
	
	return result;
}


RCX_Result SetWatch(const char *timeSpec)
{
	int hour;
	int minute;
	RCX_Cmd cmd;
	
	if (strcmp(timeSpec, "now")==0)
	{
		time_t t;
		struct tm *tmp;
		
		time(&t);
		tmp = localtime(&t);
		hour = tmp->tm_hour;
		minute = tmp->tm_min;
	}
	else
	{
		int t = atoi(timeSpec);
		hour = t / 100;
		minute = t % 100;
	}
	
	return gLink.Send(cmd.Set(kRCX_SetWatchOp, (UByte)hour, (UByte)minute));
}


RCX_Result ProcessFile(const char *sourceFile, const Request &req)
{
	RCX_Image *image;
	RCX_Result result = kRCX_OK;

	if (CheckExtension(sourceFile, kRCXFileExtension))
	{
		// load RCX image file
		image = new RCX_Image();
		result = image->Read(sourceFile);
		if (RCX_ERROR(result))
		{
			PrintError(result, sourceFile);
			delete image;
			return kQuietError;
		}
	}
	else
	{
		// compile file
		image = Compile(sourceFile, req.fFlags);
		if (!image)
		{
			int errors = ErrorHandler::Get()->GetCount();
			
			if (errors)
				fprintf(gErrorStream, "# %d error%s during compilation\n", errors, errors==1 ? "" : "s");
			return kQuietError;
		}
		
		const char *outputFile = req.fOutputFile;
		char *newFilename = 0;
		
		if (!req.fDownload && !req.fListing && !req.fOutputFile && sourceFile)
			outputFile = newFilename = CreateFilename(sourceFile, kNQCFileExtension, kRCXFileExtension);
		
		if (outputFile)
			image->Write(outputFile);

		if (newFilename)
			delete [] newFilename;
	}
	
	if (req.fListing)
		result = GenerateListing(image, req.fListFile);
	
	if (req.fDownload)
	{
		RCX_Result r2 = Download(image);
		if (result == kRCX_OK) result = r2;
	}
	
	delete image;
	return result;
}


RCX_Result GenerateListing(RCX_Image *image, const char *filename)
{
	FILE *fp;
	
	if (filename)
	{
		fp = fopen(filename, "w");
		if (!fp)
		{
			fprintf(STDERR, "Error: could not generate listing to file %s\n", filename);
			return kQuietError;
		}
	}
	else
		fp = stdout;

	RCX_StdioPrinter dst(fp);
	image->Print(&dst);
	
	if (fp != stdout)
		fclose(fp);

	return kRCX_OK;	
}


RCX_Result Download(RCX_Image *image)
{
	RCX_Result result = kRCX_OK;
	RCX_Cmd cmd;
	
	fprintf(STDERR, "Downloading Program:");
	result = gLink.Open();
	if (result != kRCX_OK) goto ErrorReturn;
	
	result = image->Download(&gLink);
	if (result != kRCX_OK) goto ErrorReturn;	
	
	fprintf(STDERR, "complete\n");
	
	result = gLink.GetBatteryLevel();
	if (!RCX_ERROR(result))
	{
		fprintf(STDERR, "Battery Level = %3.1f V\n", (double)result / 1000);
		if (result < kLowBattery)
			fprintf(STDERR, "*** Warning: batteries are low ***\n");
	}
	
	return kRCX_OK;
	
ErrorReturn:
	fprintf(STDERR, "error\n");
	return result;
}


RCX_Image *Compile(const char *sourceFile, int flags)
{
	Buffer *mainBuf;

	if (sourceFile)
	{
		FILE *fp  = fopen(sourceFile, "rb");
		if (!fp)
		{
			fprintf(gErrorStream, "Error: could not open file \"%s\"\n", sourceFile);
			return nil;
		}
		
		mainBuf = new Buffer();
		mainBuf->Create(sourceFile, fp);
		fclose(fp);
	}
	else
	{
		mainBuf = new Buffer();
		mainBuf->Create("<stdin>", stdin);
	}
		
	return Compiler::Get()->Compile(mainBuf, flags);
}


RCX_Result UploadDatalog(bool verbose)
{
	RCX_Log log;
	RCX_Result result;
	int i;
	
	fprintf(STDERR, "Uploading Datalog");
	
	result = gLink.Open();
	if (RCX_ERROR(result)) return result;
	
	result = log.Upload(&gLink);
	if (RCX_ERROR(result)) return result;
	
	fprintf(STDERR, "\n");
	
	for(i=0; i<log.GetLength(); i++)
	{
		char line[256];
		log.SPrintEntry(line, i, verbose);
		printf("%s\n", line);
	}
	
	return kRCX_OK;
}


RCX_Result ClearMemory()
{
	RCX_Result result;
	RCX_Cmd cmd;
	
	// halt all tasks
	result = gLink.Send(cmd.Set(kRCX_StopAllOp));
	if (RCX_ERROR(result)) return result;
	
	for(UByte p=0; p<5; p++)
	{
		// select and delete program
		result = gLink.Send(cmd.Set(kRCX_SelectProgramOp, p));
		if (RCX_ERROR(result)) return result;
	
		result = gLink.Send(cmd.MakeDeleteSubs());
		if (RCX_ERROR(result)) return result;

		result = gLink.Send(cmd.MakeDeleteTasks());
		if (RCX_ERROR(result)) return result;
	}

	result = gLink.Send(cmd.Set(kRCX_SetDatalogOp, 0, 0));
	return result;
}


RCX_Result DownloadFirmware(const char *filename)
{
	FILE *fp;
	SRecord srec;
	bool ok;
	RCX_Result result;
	ULong rom, ram;
	
	fp = fopen(filename, "r");
	if (!fp)
	{
		fprintf(STDERR, "Error: could not open file \'%s\'\n", filename);
		return kQuietError;
	}
	
	ok = srec.Read(fp, kMaxFirmware);
	fclose(fp);
	
	if (!ok)
	{
		fprintf(STDERR, "Error: \'%s\' is not a valid S-Record file\n", filename);
		return kQuietError;
	}

	result = gLink.Open();
	if (RCX_ERROR(result)) return result;
	
	fprintf(STDERR, "Downloading firmware:");
	fflush(STDERR);
	result = gLink.SendFirmware(srec.GetData(), srec.GetLength());
	fputc('\n', STDERR);
	if (RCX_ERROR(result)) return result;
	
	result = gLink.GetVersion(rom, ram);
	if (RCX_ERROR(result)) return result;

	fprintf(STDERR, "Current Version: %08lx/%08lx\n", rom, ram);

	return result;
}


RCX_Result SendRawCommand(const char *text)
{
	int length = (int)strlen(text);
	RCX_Cmd cmd;
	RCX_Result result;
	
	// we need an even number of chars in text
	if (length & 1) return kUsageError;
	
	// determine actual length of command
	length /= 2;
	cmd.SetLength(length);
	
	for(int i=0; i<length; i++)
	{
		int byte = SRecord::ReadHexByte(text);
		if (byte == -1) return kUsageError;
		cmd[i] = (UByte) byte;
		text+=2;
	}
	
	result = gLink.Send(&cmd);

	if (result > 0)
	{
		for(int i=0; i<result; i++)
			printf("%02x ", gLink.GetReplyByte(i));
		printf("\n");
	}
	
	return result;
}


char *CreateFilename(const char *source, const char *oldExt, const char *newExt)
{
	char *filename;
	size_t n = strlen(source);
	size_t newExtLen = strlen(newExt);

	if (CheckExtension(source, oldExt))
		n -= strlen(oldExt);

	filename = new char[n + newExtLen + 1];
	memcpy(filename, source, n);
	strcpy(filename + n, newExt);
	
	return filename;
}


int CheckExtension(const char *s1, const char *ext)
{
	size_t slen = strlen(s1);
	size_t plen = strlen(ext);
	unsigned i;
	
	if (slen < plen) return 0;
	
	for(i=0; i<plen; i++)
		if (tolower(s1[slen-plen+i]) != tolower(ext[i])) return 0;
	
	return 1;
}


RCX_Result SetErrorFile(const char *filename)
{
	FILE *fp;
	
	if (*filename==0)
	{
		gErrorStream = stdout;
		return kRCX_OK;
	}
	
	fp = fopen(filename, "w");
	if (!fp)
	{
		fprintf(STDERR, "Error: could not open error file \'%s\'\n", filename);
		return kQuietError;
	}
	
	gErrorStream = fp;
	return kRCX_OK;
}


void DefineMacro(const char *text)
{
	const char *body;
	body = strchr(text, '=');
	if (body)
	{
		char *name;

		// create a copy of the symbol name
		int length = body - text;
		name = new char[length+1];
		memcpy(name, text, (size_t)length);
		name[length] = 0;
		
		Compiler::Get()->Define(name, body);
	}
	else	
	{
		Compiler::Get()->Define(text);
	}
}


int GetActionCode(const char *arg)
{
	for(int i=0; i< (int)(sizeof(sActionNames)/sizeof(const char *)); ++i)
		if (strcmp(sActionNames[i], arg)==0) return i+kFirstActionCode;

	return 0;
}


void PrintError(RCX_Result error, const char *filename)
{
	const char *targetName = gTargetCM ? "CyberMaster" : "RCX";
	if (!filename) filename = "?";
	
	if (error >= 0) return;
	
	switch(error)
	{
		case kRCX_OpenSerialError:
			fprintf(STDERR, "Could not open serial port\n");
			break;
		case kRCX_IREchoError:
			fprintf(STDERR, "Problem talking to IR device\n");
			break;
		case kRCX_ReplyError:
			fprintf(STDERR, "No reply from %s\n", targetName);
			break;
		case kRCX_MemFullError:
			fprintf(STDERR, "Not enough free memory in %s to download program\n", targetName);
			break;
		case kRCX_FileError:
			fprintf(STDERR, "Could not access file \'%s\'\n", filename);
			break;
		case kRCX_FormatError:
			fprintf(STDERR, "File \'%s\' is not a valid RCX image\n", filename);
			break;
		case kUsageError:
			PrintUsage();
			break;
		case kQuietError:
			break;
		default:
			fprintf(STDERR, "Error #%d\n", -error);
			break;
	}
}


void PrintUsage()
{
	fprintf(STDERR,"nqc version "VERSION" (built "__DATE__", " __TIME__")\n");
	fprintf(STDERR,"     Copyright (C) 1998,1999 David Baum.  All Rights Reserved.\n");
	fprintf(STDERR,"Usage: nqc [options] [actions] [ - | filename ] [actions]\n");
	fprintf(STDERR,"   - : read from stdin instead of a source_file\n");	
	fprintf(STDERR,"Options:\n");	
	fprintf(STDERR,"   -1: use NQC 1.x compatability mode\n");	
	fprintf(STDERR,"   -c: target CyberMaster instead of RCX\n");	
	fprintf(STDERR,"   -d: download program\n");
	fprintf(STDERR,"   -n: prevent the system file (rcx.nqh) from being included\n");
	fprintf(STDERR,"   -D<sym>[=<value>] : define macro <sym>\n");
	fprintf(STDERR,"   -E[<filename>] : write compiler errors to <filename> (or stdout)\n");
	fprintf(STDERR,"   -I<path>: search <path> for include files\n");
	fprintf(STDERR,"   -L[<filename>] : generate code listing to <filename> (or stdout)\n");
	fprintf(STDERR,"   -O<outfile>: specify output file\n");
	fprintf(STDERR,"   -S<portname>: specify serial port\n");
	fprintf(STDERR,"   -U<sym>: undefine macro <sym>\n");
	fprintf(STDERR,"Actions:\n");	
	fprintf(STDERR,"   -run: run current program\n");
	fprintf(STDERR,"   -pgm <number>: select program number\n");
	fprintf(STDERR,"   -datalog | -datalog_full: upload datalog\n");
	fprintf(STDERR,"   -near : set IR to near mode\n");
	fprintf(STDERR,"   -far : set IR to far mode\n");
	fprintf(STDERR,"   -watch <time> | now : set RCX time\n");
	fprintf(STDERR,"   -firmware <filename> : download firmware\n");
	fprintf(STDERR,"   -sleep <timeout> : set RCX sleep timeout\n");
	fprintf(STDERR,"   -msg <number> : send IR message to RCX\n");
	fprintf(STDERR,"   -raw <data> : format data as a packet and send to RCX\n");
	fprintf(STDERR,"   -clear : erase all programs and datalog in RCX\n");
}


RCX_Result AutoLink::Open()
{
	RCX_Result result;
	
	if (!fOpen)
	{
		result = RCX_Link::Open(fSerialPort);
		if (RCX_ERROR(result)) return result;
		
		fOpen = true;
	}
	return kRCX_OK;
}


void AutoLink::Close()
{
	if (fOpen)
	{
		RCX_Link::Close();
		fOpen = false;
	}
}


RCX_Result AutoLink::Send(const RCX_Cmd *cmd, bool retry)
{
	RCX_Result result;
	
	result = Open();
	if (RCX_ERROR(result)) return result;
	
	result = Sync();
	if (RCX_ERROR(result)) return result;
	
	result = RCX_Link::Send(cmd, retry);

	return retry ? result : kRCX_OK;
}


bool AutoLink::DownloadProgress(int /* soFar */, int /* total */)
{
	fputc('.', STDERR);
	fflush(STDERR);
	return true;
}



void MyCompiler::AddError(const Error &e, const LexLocation *loc)
{
	char msg[Error::kMaxErrorMsg];
	e.SPrint(msg);

	if (e.IsWarning())
	{
		fprintf(gErrorStream, "# Warning: ");
	}
	else
	{
		int errorCount = ErrorHandler::Get()->GetCount();
		// only print the first few errors
		if (errorCount > kMaxPrintedErrors)
		{
			if (errorCount == kMaxPrintedErrors+1)
				fprintf(gErrorStream, "Too many errors - only first %d reported\n", kMaxPrintedErrors);
			return;
		}

		fprintf(gErrorStream, "# Error: ");
	}

	fprintf(gErrorStream, "%s\n", msg);

	if (loc)
	{
		// get line number (and adjust for NL errors)
		int line = loc->fLine;
		if (loc->fCol == -1)
			line--;
	
		fprintf(gErrorStream, "File \"%s\" ; line %d\n", loc->fBuffer->GetName(), line);
	
		if ((loc->fCol != -1))
		{
			int i;
			const char *ptr = loc->fBuffer->GetLine(line);
			if (ptr)
			{
				fprintf(gErrorStream, "# ");
				
				for(; *ptr != '\n'; ptr++)
				{
					putc((*ptr == '\t') ? ' ' : *ptr, gErrorStream);
				}
				
				fprintf(gErrorStream,"\n# ");
				for(i=0; i<loc->fCol; i++)
					putc(' ', gErrorStream);
				for(i=0; i<loc->fLength; i++)
					putc('^', gErrorStream);
				fprintf(gErrorStream,"\n");
			}
		}
	}

	fprintf(gErrorStream, "#----------------------------------------------------------\n");
}


Buffer *MyCompiler::GetBuffer(const char *name)
{
	static char pathname[DirList::kMaxPathname];	

	if (!fDirs.Find(name, pathname))
		return nil;

	FILE *fp = fopen(name, "r");
	if (!fp) return 0;
	
	Buffer *buf = new Buffer();
	buf->Create(name, fp);
	fclose(fp);

	return buf;
}
