/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1999 David Baum.
 * All Rights Reserved.
 */

#include "Expr.h"
#include "RCX_Cmd.h"
#include "Bytecode.h"
#include "VarPool.h"
#include "Error.h"

Expr::~Expr()
{
}


RCX_Value Expr::EmitAny(Bytecode &b, const LexLocation *loc) const
{
	int v;
	
	if (Evaluate(v))
		return RCX_VALUE(kRCX_ConstantType, v);
	else
		return EmitAny_(b, loc);
}


bool Expr::EmitTo(Bytecode &b, int dst, const LexLocation *loc) const
{
	int v;
	
	if (Evaluate(v))
	{
		b.AddMove(dst, RCX_VALUE(kRCX_ConstantType, v));
		return true;
	}
	else
		return EmitTo_(b, dst, loc);
}


RCX_Value Expr::EmitConstrained(Bytecode &b, long mask, const struct LexLocation *loc) const
{
	RCX_Value ea = EmitAny(b, loc);
	if (ea == kIllegalEA) return ea;
	
	RCX_ValueType type = RCX_VALUE_TYPE(ea);
	if (((1L << type) & mask) == 0)
	{
		// if registers are not valid, then return illegal EA
		if (!((1 << kRCX_VariableType) & mask))
		{
			Error(kErr_BadExpression).Raise(loc);
			return kIllegalEA;
		}
		
		// move into temp register
		int dst = GetTempVar(b, loc);
		if (dst < 0) return kIllegalEA;

		b.AddMove(dst, ea);	
		ea = RCX_VALUE(kRCX_VariableType, dst);	
	}
	
	return ea;
}


bool Expr::EmitSide(Bytecode &b, const struct LexLocation *loc) const
{
	return EmitSide_(b, loc);
}


bool Expr::EmitTo_(Bytecode &b, int dst,  const LexLocation *loc) const
{
	RCX_Value ea = EmitAny_(b, loc);
	
	if (ea == kIllegalEA) return false;
	
	b.AddMove(dst, ea);
	b.ReleaseTempEA(ea);
	
	return true;
}


bool Expr::EmitSide_(Bytecode &, const LexLocation *) const
{
	return true;
}


int Expr::GetTempVar(Bytecode &b, const LexLocation *loc)
{
	int var = b.GetTempVar();
	
	if (var == kIllegalVar)
	{
		if (b.TempsAllowed())
			Error(kErr_NoMoreTemps).Raise(loc);
		else
			Error(kErr_TempsNotAllowed).Raise(loc);
	}
	
	return var;
}
