/**
 * @file table.c
 *
 * NJAMD table ADT.
 *
 * This data structure is not quite a strict array ADT.. It's a big block of 
 * memory you can request chunks off of, and index into or iterate over later 
 * if need be.
 *
 * Copyright (C) 2000 by Mike Perry.
 * Distributed WITHOUT WARRANTY under the GPL. See COPYING for details.
 */
#include <lib/table.h>
#include <lib/util.h>
#include <lib/portability.h>
#include <lib/output.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

static void table_resize(struct nj_table *table);

/**
 * Initializes a table
 * 
 * @param new_table The table to initialize
 * @param file If non-null, a file to use as the backing store for this memory (ie mmap file
 * and use it for our datasource). The paramater itself must be persistant.
 * @param tbl_size The initial memory size of the table
 * @param rezie If true, resize the table when it runs out of memory.
 * @param atomic If true, use the pthread mutex to lock the table.
 *
 * @NOTE: Non-atomic, non-file tables can use the nj_table_light data 
 * structure to save memory (ie for tables of tables).
 */
int __nj_table_bootstrap_init(struct nj_table *new_table, char *file, 
		size_t tbl_size, int resize, int atomic)
{
	int i, fd;

	if(file)
	{
		/* We need a NEW heapfd for this.. we can't just expand the old one
		 * Even though errno should be valid by now, we want to keep the format
		 * consistant */
		new_table->has_file = 1;
		if((fd = open(file, O_TRUNC|O_CREAT|O_RDWR, S_IRUSR|S_IWUSR)) == -1)
		{
			return -1;
		}

		/* What are we gonna do about initial memory state? */
		if((i = lseek(fd, tbl_size, SEEK_SET)) != tbl_size)
		{
			__nj_eprintf("Seeked %ld out of %ld\n", i, tbl_size);
			return -1;
		}

		/* Consider this an easter egg ;) */
		write(fd, "Mike Perry r0xx0rs", 1);

		lseek(fd, 0, SEEK_SET);

		if((new_table->data = mmap(NULL, tbl_size, PROT_READ|PROT_WRITE, MAP_SHARED, fd, 0)) == (void *)-1)
		{
			return -1;
		}

		close(fd);
	}
	else
	{
		new_table->has_file = 0;
		if((new_table->data= mmap(NULL, tbl_size, PROT_READ|PROT_WRITE, 
						MAP_PRIVATE|MAP_ANON, __nj_anonfd, 0)) == (void *)-1)
		{
			return -1;
		}
	}

	new_table->resize = resize;
	new_table->size = tbl_size;
	new_table->atomic = atomic;
	new_table->top = 0;

	if(file)
		new_table->file = file;
	return 0;
}

void __nj_table_user_init(struct nj_table *table, struct nj_prefs *prefs)
{
#ifdef _THREAD_SAFE
	if(table->atomic)
		pthread_mutex_init(&table->lock, NULL);
#endif

	table->persist = prefs->stat.persist;
}

/**
 * Closes a table.
 *
 * Called when the table is no longer needed. Unmaps its memory, and truncates 
 * it if it is file-based.
 * 
 * @param table the table
 */ 
void __nj_table_fini(struct nj_table *table)
{
#ifdef _THREAD_SAFE
	if(table->atomic)
		pthread_mutex_lock(&table->lock);
#endif

	/* Do not unmap, only sync */

	if(table->has_file)
	{
		if(table->persist)
		{
			msync(table->data, table->top, MS_SYNC);
			truncate(table->file, table->top);
		}
		else
		{
			unlink(table->file);
		}
	}

#ifdef _THREAD_SAFE
	if(table->atomic)
		pthread_mutex_unlock(&table->lock);

	pthread_mutex_destroy(&table->lock);
#endif
}

/**
 * Syncs the table to disk, synchronously.
 * 
 * @param table the table
 *
 * @sideeffects The syncing is synchronous, so it may take a while.
 */ 
void __nj_table_sync(struct nj_table *table)
{
#ifdef _THREAD_SAFE
	if(table->atomic)
		pthread_mutex_lock(&table->lock);
#endif
	
	if(table->has_file)
		msync(table->data, table->size, MS_SYNC);

#ifdef _THREAD_SAFE
	if(table->atomic)
		pthread_mutex_unlock(&table->lock);
#endif
}

/**
 * Trim the edge off a table
 *
 * @param table The table
 * @param new_size The new size
 */
void __nj_table_trim(struct nj_table *table, size_t new_size)
{
	nj_addr_t end;
#ifdef _THREAD_SAFE
	if(table->atomic)
		pthread_mutex_lock(&table->lock);
#endif

	end = (nj_addr_t)table->data + new_size;

	munmap((void *)end, table->size - new_size);

	table->size = new_size;
	
#ifdef _THREAD_SAFE
	if(table->atomic)
		pthread_mutex_unlock(&table->lock);
#endif

}

/**
 * Truncate the table 
 *
 * Releases the memory after the top of the table.
 *  
 * @param table The table to truncate
 */
void __nj_table_trunc(struct nj_table *table)
{
	__nj_table_trim(table, table->top);
}



/**
 * Request the top of the table for exclusive use
 *
 * This method will return a pointer to the top of the table, gauranteed 
 * to have max_use entires available. You MUST release the table with 
 * __nj_release_top when you know exactly how much memory you need.
 *
 * @param table A pointer to the table
 * @param max_use The maximum number of bytes you will use
 * @returns A u_int pointer to the memory region. Derefrencing this pointer i
 * will get you the index of the memory returned.
 * @sideffects Locks the table to protect your memory
 */
u_int *__nj_table_request_top(struct nj_table *table, size_t max_use)
{
	u_int *ret;
#ifdef _THREAD_SAFE
	if(table->atomic)
		pthread_mutex_lock(&table->lock);
#endif
	
	if(table->size < (table->top+max_use))
	{
		if(table->resize)
			table_resize(table);
		else
		{
#ifdef _THREAD_SAFE
			if(table->atomic)
				pthread_mutex_unlock(&table->lock);
#endif
			return NULL;
		}
	}

	ret = NJ_PTR_ADD_ADDR(table->data, table->top);
	*ret = table->top;

	/* Do NOT unlock */

	return ret;
}

/**
 * Release the table lock and increment the top
 *
 * Call this method to release the top of the table after you request it.
 * 
 * @param table A pointer to the table
 * @sideffects Unlocks the table, increments the top of the table
 */
void __nj_table_release_top(struct nj_table *table, size_t increment)
{
	table->top += increment;

#ifdef _THREAD_SAFE
	if(table->atomic)
		pthread_mutex_unlock(&table->lock);
#endif

}

/**
 * Get a chunk off the top of the table
 *
 * This method will return a pointer to the top of the table, gauranteed 
 * to have max_use entires available. You MUST release the table with 
 * __nj_release_top when you know exactly how much memory you need.
 *
 * @param table A pointer to the table
 * @param size The size of the chunk you want. Must be >= sizeof(int)
 * @returns A u_int pointer to the memory region. Derefrencing this pointer i
 * will get you the index of the memory returned.
 */
u_int *__nj_table_get_chunk(struct nj_table *table, size_t size)
{
	u_int *ret;

#ifdef _THREAD_SAFE
	if(table->atomic)
		pthread_mutex_lock(&table->lock);
#endif
	if(table->size < (table->top+size))
	{
		if(table->resize)
			table_resize(table);
		else
		{
#ifdef _THREAD_SAFE
			if(table->atomic)
				pthread_mutex_unlock(&table->lock);
#endif
			return NULL;
		}
	}

	ret = (u_int *)((nj_addr_t)table->data + table->top);

	*ret = table->top;
	
	table->top += size;

#ifdef _THREAD_SAFE
	if(table->atomic)
		pthread_mutex_unlock(&table->lock);
#endif

	return ret;
}

/**
 * Executes a function for each entry of size elem_size in a table.
 *
 * Continues to run until one of these functions returns non-null, at which
 * point execution ceases and that value is returned. The function is protected
 * inside a mutex lock, and should not attempt to reenter any table other 
 * functions unless the table is non-atomic.
 * 
 * @param table The table to iterate over
 * @param start_idx A pointer to the index to start at in the table.
 * 			It is updated to the index of the found item
 * @param elem_size The size of each element
 * @param iter The iterator function
 * @param ... Arguments to the iterator function
 * 
 * @returns The first non-null value from iter.
 * @see __nj_table_for_all_indicies
 */
void *__nj_table_for_all_entries(struct nj_table *table, u_int *start_idx, 
		size_t elem_size, nj_table_entry_iterator_t iter, ...)
{
	va_list ap;
	void *ret;
	int i;
	
	va_start(ap,iter);

#ifdef _THREAD_SAFE
	if(table->atomic)
		pthread_mutex_lock(&table->lock);
#endif

	for(i = (start_idx ? *start_idx*elem_size : 0); i < table->top; i+= elem_size)
	{
		if((ret = iter(table->data+i, ap)) != NULL)
		{
#ifdef _THREAD_SAFE
			if(table->atomic)
				pthread_mutex_unlock(&table->lock);
#endif
			va_end(ap);

			if(start_idx)
				*start_idx = i / elem_size;
			
			return ret;
		}
#ifdef NEED_VA_REWIND
		va_end(ap);
		va_start(ap, iter);
#endif
	}

	if(start_idx)
		*start_idx = table->top/elem_size;
	
#ifdef _THREAD_SAFE
	if(table->atomic)
		pthread_mutex_unlock(&table->lock);
#endif

	va_end(ap);
	return NULL;
}

/**
 * Executes a function for each entry in the table
 * 
 * The difference between this function and the for_all_entries is that the 
 * iterator is passed the table datastructure itself, as well as an index,
 * instead of just a void pointer, and iter continues until it returns a 
 * non-negative index.
 * 
 * @param table The table to iterate over
 * @param start_idx A pointer to the index to start at in the table.
 * 			It is updated to the index of the found item
 * @param elem_size The size of each element
 * @param iter The iterator function
 * @param ... Arguments to the iterator function
 * 
 * @returns The first non-negative index from iter
 * @see __nj_table_for_all_indicies
 */
int __nj_table_for_all_indicies(struct nj_table *table, u_int *start_idx, 
		size_t elem_size, nj_table_index_iterator_t iter, ...)
{
	va_list ap;
	int i;
	int ret;
	
	va_start(ap,iter);
#ifdef _THREAD_SAFE
	if(table->atomic)
		pthread_mutex_lock(&table->lock);
#endif
		
	for(i = (start_idx ? *start_idx : 0); i < table->top/elem_size; i++)
	{
		if((ret = iter(table, i, ap)) != -1)
		{
#ifdef _THREAD_SAFE
			if(table->atomic)
				pthread_mutex_unlock(&table->lock);
#endif
			va_end(ap);

			if(start_idx)
				*start_idx = i;

			return ret;
		}
#ifdef NEED_VA_REWIND
		va_end(ap);
		va_start(ap, iter);
#endif
	}
	
	if(start_idx)
		*start_idx = table->top/elem_size;

#ifdef _THREAD_SAFE
	if(table->atomic)
		pthread_mutex_unlock(&table->lock);
#endif

	va_end(ap);
	return -1;

}


/**
 * Resizes the table. 
 *
 * Resizes the table, calling mmap if needed. 
 *
 * @sideefects The table is now twice the size as before. 
 */ 
static void table_resize(struct nj_table *table)
{
	int fd, i;
	void *new_table;

	/* If we have a file, use it to  */
	if(table->has_file)
	{
		munmap(table->data, table->size);

		/* Double the size */	
		table->size <<= 1;

		if((fd = open(table->file, O_RDWR, S_IRUSR|S_IWUSR)) == -1)
		{
			__nj_critical_error(__FUNCTION__ ": Can't create heap file");
		}

		/* What are we gonna do about initial memory state? */
		if((i = lseek(fd, table->size, SEEK_SET)) != table->size)
		{
			__nj_eprintf("Seeked %ld out of %ld\n", i, table->size);
			__nj_critical_error(__FUNCTION__": lseek");
		}

		/* Consider this an easter egg ;) */
		write(fd, "Mike Perry r0xx0rs", 1);

		lseek(fd, 0, SEEK_SET);

		if((table->data = mmap(NULL, table->size, PROT_READ|PROT_WRITE, MAP_SHARED, fd, 0)) == (void *)-1)
		{
			__nj_critical_error(__FUNCTION__": mmap");
		}

		close(fd);
	}		
	else
	{
		/* Double the size */
		table->size <<= 1;

		/** @FIXME: no check for this yet.. */
#ifdef HAVE_LINUX_MREMAP 
		mremap(table->data, table->size >> 1, table->size, MREMAP_MAYMOVE);
#else
	
		if((new_table = mmap(NULL,
						table->size, PROT_READ|PROT_WRITE, 
						MAP_PRIVATE|MAP_ANON,
						__nj_anonfd, 0)) == (void *)-1)
		{
			__nj_critical_error(__FUNCTION__": mmap");
		}

		memcpy(new_table, table->data, table->size >> 1);
		munmap(table->data, table->size >> 1);
		table->data = new_table;
#endif
	}
}
// vim:ts=4
