/** 
 * @file block.c Block routines
 *
 * These functions work with blocks from the memory pools
 * 
 * Copyright (C) 2000 by Mike Perry.
 * Distributed WITHOUT WARRANTY under the GPL. See COPYING for details.
 * 
 */
#include <lib/portability.h>
#include <lib/allocator.h>
#include <lib/entry_pool.h>
#include <lib/user_chunk.h>
#include <lib/util.h>
#include <lib/block.h>
#include <lib/output.h>

/**
 * Return the size of a block
 *
 * Calculates the block size needed from the memory pool to support a user's
 * chunk.
 
 * @param len The length of the user's segment
 * @param alloc_type THe alloc type of the block
 * @param align THe alignment of the block
 *
 * @returns The size needed
 */ 
size_t __nj_block_calc_size(int len, int align, int alloc_type)
{
	switch(alloc_type)
	{
		case NJ_PROT_NONE:
			return NJ_PROT_NONE_BLOCK_SIZE(len, align);
			break;
		case NJ_PROT_OVER:
			return NJ_PROT_OVER_BLOCK_SIZE(len, align);
			break;

		case NJ_PROT_UNDER:
			return NJ_PROT_UNDER_BLOCK_SIZE(len, align);
			break;

		case NJ_PROT_SUNDER:
			return NJ_PROT_SUNDER_BLOCK_SIZE(len, align);
			break;
	}
	__nj_eprintf("Invalid alloc type %s\n", __nj_allocator_type_to_string(alloc_type));
	return 0;
}

/**
 * Gets the user chunk from the block
 * 
 * @param block The description of this block
 * @param addr The block to get a user chunk from
 * @param block_len The length of the block
 * @param user_len The length the user requested
 * @param prefs User prefrences
 *
 * @returns A user chunk from this block
 */
void __nj_block_init(struct nj_block *block, nj_addr_t addr, 
		size_t block_len, size_t user_len, struct nj_dynamic_prefs prefs)
{
	block->addr = addr;
	switch(prefs.alloc_type)
	{
		case NJ_PROT_NONE:
			/* Align the block to the requested alignment.
			 * If an index pointer will fit, return that
			 * Otherwise align the block plus an index poiner to align
			 */
			block->user_chunk = (NJ_ALIGN_UP(addr, prefs.align) - addr >= sizeof(nj_entry_index_t)) ?
				 NJ_ALIGN_UP(addr, prefs.align) :
				 NJ_ALIGN_UP(addr + sizeof(nj_entry_index_t), prefs.align);
			NJ_PROT_NONE_GET_FENCEPOST(block->user_chunk, user_len) = NJ_FENCEPOST;
			block->idx = (nj_entry_index_t *)NJ_ALIGN_DOWN(block->user_chunk - sizeof(nj_entry_index_t), NJ_INT_ALIGNMENT);
			break;
		case NJ_PROT_OVER:
			/* 
			 * The user's buffer must end on the page boundary.. 
			 * Making the length aligned to the proper alignment causes it's start
			 * to be aligned as well (since block is aligned, and PAGE_SIZE is aligned)
			 */
			block->user_chunk = (addr + ((block_len -  NJ_ALIGN_UP(user_len, prefs.align)) - NJ_PAGE_SIZE));
			NJ_PROT_OVER_GET_FENCEPOST(block->user_chunk, user_len) = NJ_FENCEPOST;
			block->idx = (nj_entry_index_t *)addr;
			break;
		case NJ_PROT_UNDER:
			/*
			 * The user's buffer starts one page into the block
			 * After the index pointer, but aligned to the requested alignment
			 */
			block->user_chunk = (addr + NJ_PAGE_SIZE + NJ_ALIGN_UP(sizeof(nj_entry_index_t), prefs.align));
			NJ_PROT_UNDER_GET_FENCEPOST(block->user_chunk, user_len) = NJ_FENCEPOST;
			block->idx = (nj_entry_index_t *)(addr + NJ_PAGE_SIZE);
			break;
		case NJ_PROT_SUNDER:
			/* THe user's buffer starts right after the protected page */
			block->user_chunk = (addr + NJ_PAGE_SIZE);
			NJ_PROT_SUNDER_GET_FENCEPOST(block->user_chunk, user_len) = NJ_FENCEPOST;
			block->idx = (nj_entry_index_t *)addr;
			break;
		default:
			__nj_eprintf("NJAMD: Error, unknwown alloc type?\n");
			break;
	}

}

/**
 * Gets the user chunk from the block
 * 
 * @param block The description of this block
 * @param addr The block to get a user chunk from
 * @param block_len The length of the block
 * @param user_len The length the user requested
 * @param move_data How to move the data (memcpy or memmove)
 * @param old_chunk The old user chunk
 * @param old_user_len The old user length
 * @param prefs User prefrences
 *
 * @returns A user chunk from this block
 */
void __nj_block_renew(struct nj_block *block, nj_addr_t addr, 
		size_t block_len, size_t user_len, 
		void *(*move_data)(void *, const void *, size_t),
		nj_addr_t old_chunk, size_t old_user_len,
		struct nj_dynamic_prefs prefs)
{

	block->addr = addr;
	switch(prefs.alloc_type)
	{
		case NJ_PROT_NONE:
			/* Align the block to the requested alignment.
			 * If an index pointer will fit, return that
			 * Otherwise align the block plus an index poiner to align
			 */
			block->user_chunk = (NJ_ALIGN_UP(addr, prefs.align) - addr >= sizeof(nj_entry_index_t)) ?
				 NJ_ALIGN_UP(addr, prefs.align) :
				 NJ_ALIGN_UP(addr + sizeof(nj_entry_index_t), prefs.align);
			move_data((void*)block->user_chunk, (void*)old_chunk, NJ_MIN(user_len, old_user_len));
			NJ_PROT_NONE_GET_FENCEPOST(block->user_chunk, user_len) = NJ_FENCEPOST;
			block->idx = (nj_entry_index_t *)NJ_ALIGN_DOWN(block->user_chunk - sizeof(nj_entry_index_t), NJ_INT_ALIGNMENT);
			break;
		case NJ_PROT_OVER:
			/* 
			 * The user's buffer must end on the page boundary.. 
			 * Making the length aligned to the proper alignment causes it's start
			 * to be aligned as well (since block is aligned, and PAGE_SIZE is aligned)
			 */
			block->user_chunk = (addr + ((block_len -  NJ_ALIGN_UP(user_len, prefs.align)) - NJ_PAGE_SIZE));
			move_data((void *)block->user_chunk, (void*)old_chunk, NJ_MIN(user_len, old_user_len));
			NJ_PROT_OVER_GET_FENCEPOST(block->user_chunk, user_len) = NJ_FENCEPOST;
			block->idx = (nj_entry_index_t *)addr;
			break;
		case NJ_PROT_UNDER:
			/*
			 * The user's buffer starts one page into the block
			 * After the index pointer, but aligned to the requested alignment
			 */
			block->user_chunk = (addr + NJ_PAGE_SIZE + NJ_ALIGN_UP(sizeof(nj_entry_index_t), prefs.align));
			move_data((void*)block->user_chunk, (void*)old_chunk, NJ_MIN(user_len, old_user_len));
			NJ_PROT_UNDER_GET_FENCEPOST(block->user_chunk, user_len) = NJ_FENCEPOST;
			block->idx = (nj_entry_index_t *)(addr + NJ_PAGE_SIZE);
			break;
		case NJ_PROT_SUNDER:
			/* THe user's buffer starts right after the protected page */
			block->user_chunk = (addr + NJ_PAGE_SIZE);
			move_data((void*)block->user_chunk, (void*)old_chunk, NJ_MIN(user_len, old_user_len));
			NJ_PROT_SUNDER_GET_FENCEPOST(block->user_chunk, user_len) = NJ_FENCEPOST;
			block->idx = (nj_entry_index_t *)addr;
			break;
		default:
			__nj_eprintf("NJAMD: Error, unknwown alloc type?\n");
			break;
	}

}


// vim:ts=4
