/**
 * @file callstack.h
 *
 * Header file for the callstack index object
 *
 * Since the return address callstack needs to be accessed by NJAMDPM, we
 * can't simply use pointers. We can store the index however. We also need to
 * store the callstack length.
 * 
 * Copyright (C) 2001 by Mike Perry.
 * Distributed WITHOUT WARRANTY under the GPL. See COPYING for details. 
 */
#ifndef __NJ_LIB_CALLSTACK_H__
#define __NJ_LIB_CALLSTACK_H__

#include <lib/stack.h>
#include <config.h>

/**@{ @name Callstack size paramaters
 *
 * Each allocation uses two pages of address space, or 2^13 bytes. So
 * there can be 2^32/2^13 = 2^19 allocations. Figure each allocation has an
 * average callstack length of 8 for malloc, and 8 for free. That's 2^23
 * entries. We have 9 bits to spare to hold the allocation lengths. I'm going
 * to go out on a limb and say any callstack length >64 is gonna be pretty
 * much useless, and if we have too many we'll run out of space anyways.
 * 2^6=64, 32-6 = 26.
 */
#define NJ_CALLSTACK_LEN_BITS				6
/** 8 for malloc, 8 for free */
#define NJ_CALLSTACK_AVG_PER_ALLOC			16

/** Comes out to 63 for now */
#define NJ_CALLSTACK_MAX_LEN				((u_long)(2<<NJ_CALLSTACK_LEN_BITS)-1)
#define NJ_CALLSTACK_MIN_LEN				(sizeof(struct nj_stack_item))
/* This will be a page size already */
#define NJ_CALLSTACK_INIT_TBL_SIZE			(NJ_ALLOCS_IN_ADDRESS_SPACE*NJ_CALLSTACK_AVG_PER_ALLOC*sizeof(u_long))
/*@}*/

/**@{ @name Dummy return addresses indices */ 
#define NJ_CALLSTACK_INDEX_DESTRUCTOR					((u_int)-1 & ((1 << (sizeof(u_long)*8 - NJ_CALLSTACK_LEN_BITS)) -1))
#define NJ_CALLSTACK_INDEX_IGNORE						((u_int)-2 & ((1 << (sizeof(u_long)*8 - NJ_CALLSTACK_LEN_BITS)) -1))
#define NJ_CALLSTACK_INDEX_NONE							((u_int)-3 & ((1 << (sizeof(u_long)*8 - NJ_CALLSTACK_LEN_BITS)) -1))
#define NJ_CALLSTACK_INDEX_NOTFREE						((u_int)-4 & ((1 << (sizeof(u_long)*8 - NJ_CALLSTACK_LEN_BITS)) -1))
#define NJ_CALLSTACK_INDEX_FALLOW						((u_int)-5 & ((1 << (sizeof(u_long)*8 - NJ_CALLSTACK_LEN_BITS)) -1))
/*@}*/

/** The callstack index itself. Is stored in the heap entry, hence the bitfun */
struct nj_callstack
{
	u_int len : NJ_CALLSTACK_LEN_BITS;						 /**< Length of the trace */
	u_long index : sizeof(u_long)*8 - NJ_CALLSTACK_LEN_BITS; /**< Index into table */
};


#endif /* callstack.h */
// vim:ts=4
