/**
 * @file block.h
 *
 * Header file for the memory pool block
 *
 * Copyright (C) 2000 by Mike Perry.
 * Distributed WITHOUT WARRANTY under the GPL. See COPYING for details. 
 */
#ifndef __NJ_LIB_BLOCK_H__
#define __NJ_LIB_BLOCK_H__
#include <config.h>
#include <lib/entry_pool.h>

struct nj_block
{
	nj_addr_t			addr;
	nj_addr_t			user_chunk;
	nj_entry_index_t	*idx;	
	size_t 				size;
};

/**@{ @name Block size macros */
/** Align the length to int alignment for the fencepost afterwards.
 * Add the fencepost size
 * add the whichever is larger:
 *  the overhead due to the user's requested alignment as compared
 *  to malloc's alignment
 * or
 *  the aligned index marker
 */
#define NJ_PROT_NONE_BLOCK_SIZE(len, align) \
			(NJ_ALIGN_UP((len), NJ_INT_ALIGNMENT) + sizeof(NJ_FENCEPOST) + \
				NJ_MAX((align > LIBC_MALLOC_ALIGNMENT ? \
							align - LIBC_MALLOC_ALIGNMENT : 0), \
						NJ_ALIGN_UP(sizeof(nj_entry_index_t), align)))

/** 
 * Add space for aligned length (this will account for aligned base
 *   because we do offset = PAGE - len)
 * add space for the heap entry before the buffer
 * add space for the fencepost before the buffer (which must start 
 *  on an int aligned boundary)
 * Round the whole thing to a 2-page block 
 */
#define NJ_PROT_OVER_BLOCK_SIZE(len, align)	\
			(NJ_ROUND_TO_BLOCK(NJ_ALIGN_UP(NJ_ALIGN_UP(len, align) \
					+ sizeof(nj_entry_index_t), NJ_INT_ALIGNMENT) \
					+ sizeof(NJ_FENCEPOST)))

/** Add space for length and entry, aligned for the fencepost
 * Add the fencepost
 * Round to 2-page block 
 */
#define NJ_PROT_UNDER_BLOCK_SIZE(len, align) \
	(NJ_ROUND_TO_BLOCK(NJ_ALIGN_UP(len + NJ_ALIGN_UP(sizeof(nj_entry_index_t), align), NJ_INT_ALIGNMENT) + sizeof(NJ_FENCEPOST)))

/**
 * Add space for the length, aligned to fencepost
 * add space for the fencepost
 * Round the whole thing to 2-page block 
 * Heap entry is in the proected page.. it will fit with no prob :) 
 */
#define NJ_PROT_SUNDER_BLOCK_SIZE(len, align) \
	(NJ_ROUND_TO_BLOCK(NJ_ALIGN_UP(len, NJ_INT_ALIGNMENT) + sizeof(NJ_FENCEPOST)))
/*@}*/

size_t __nj_block_calc_size(int, int, int);
void __nj_block_init(struct nj_block *block, nj_addr_t addr, 
		size_t block_len, size_t user_len, struct nj_dynamic_prefs prefs);
void __nj_block_renew(struct nj_block *block, nj_addr_t addr, 
		size_t block_len, size_t user_len, 
		void *(*move_data)(void *, const void *, size_t),
		nj_addr_t old_chunk, size_t old_user_len,
		struct nj_dynamic_prefs prefs);


#endif /* block.h */
// vim:ts=4
