/**
 * @file allocator.h
 *
 * Header file for the allocator object
 *
 * Copyright (C) 2000 by Mike Perry.
 * Distributed WITHOUT WARRANTY under the GPL. See COPYING for details. 
 */
#ifndef __NJ_LIB_ALLOCATOR_H__
#define __NJ_LIB_ALLOCATOR_H__

#include <lib/memory_pool.h>
#include <lib/entry_pool.h>
#include <lib/util.h>
#include <config.h>

/**@{ @name Allocator bootstrap types 
 *
 * Used before the allocator enters main and we know what type of allocation
 * the user wants.
 * 
 * Do NOT change these defaults.. We need them set this way
 * to determine when the user's environment prefs took over in the 
 * allocator.
 */
#define NJ_ALLOCATOR_BOOTSTRAP_ALLOC_TYPE	NJ_PROT_OVER
#define NJ_ALLOCATOR_BOOTSTRAP_FREE_TYPE	NJ_CHK_FREE_SEGV
/*@}*/

/**
 * Determine if chunk is between the first system allocation and the last one
 *
 * If yes, we used the bootstrap allocator
 */
#define NJ_ALLOCATOR_CHUNK_FROM_BOOTSTRAP(allocator, chunk)	\
	((allocator)->first_sys_alloc <= (chunk) && (chunk) < (allocator)->last_sys_alloc)

/** allocator object */
struct nj_allocator
{
	struct nj_memory_pool	 	mem_pool;	/**< The memory source  */
	struct nj_entry_pool		entry_pool;	/**< The heap entry pool */
	int							fixed_alloc;/**< Are all allocs of the same type?*/
	nj_addr_t					first_sys_alloc; /**< The first system allocation, used to keep track of the bootstrap alloc */
	nj_addr_t					last_sys_alloc; /**< The last allocation before main */
	void *(*libc_realloc)(void *,size_t);	/**< Symbol for the libc reallocator */
};

/**@{ @name Error types during discovery */
#define NJ_ALLOCATOR_ERROR_NONE					(1<<0)
#define NJ_ALLOCATOR_ERROR_WEIRD				(1<<1)
#define NJ_ALLOCATOR_ERROR_OVERFLOW				(1<<2)
#define NJ_ALLOCATOR_ERROR_UNDERFLOW			(1<<3)
#define NJ_ALLOCATOR_ERROR_BAD_PTR				(1<<4)
#define NJ_ALLOCATOR_ERROR_DOUBLE_FREE			(1<<5)
/*@}*/

#define NJ_ALLOCATOR_BLOCK_UNKNOWN		((nj_addr_t)-1)


void __nj_allocator_bootstrap_init(struct nj_allocator *);
void __nj_allocator_user_init(struct nj_allocator *, struct nj_libc_syms *, struct nj_prefs *);
void __nj_allocator_fini(struct nj_allocator *);
const char *__nj_allocator_type_to_string(int);
nj_addr_t __nj_allocator_request_user_chunk(struct nj_allocator *, size_t, struct nj_dynamic_prefs);
nj_addr_t __nj_allocator_resize_user_chunk(struct nj_allocator *, nj_addr_t, size_t, struct nj_dynamic_prefs);
void __nj_allocator_release_user_chunk(struct nj_allocator *, nj_addr_t, struct nj_dynamic_prefs);

#endif /* allocator.h */
// vim:ts=4
