/* Manage the menu bar in the main window.
 */

#include "ip.h"

/* We keep a list of menu configurations. Each of these contains a list of
 * toolkits whose menu should be shown when we are in that configuration.
 */

/* A menu configuration.
 */
typedef struct {
	char *name;		/* Configuration name */
	List *kits;		/* List of kits to show */
	Widget item;		/* Menu item for this kit */
} MenuConfig;

/* All the menu configs we know about. 
 */
static List *menu_configs = NULL;

/* Select an item on the menu menu.
 */
/*ARGSUSED*/
static void
select_menu_cb( Widget wid, XtPointer client, XtPointer call ) 
{
	decl( MenuConfig *, conf, client );
	decl( XmToggleButtonCallbackStruct *, cb, call );

	if( cb->reason == XmCR_VALUE_CHANGED && cb->set ) {
		/* Select it.
		 */
		(void) change_menu_config( conf->name );
	}
}

/* Fill the menu pane widget.
 */
static void *
add_menu_name( MenuConfig *conf )
{
	Arg args[ MAX_ARGS ];
	int n;

	/* Do we have the MenuPane yet? Do nothing if we do not.
	if( !popup_menu_pane )
		return( NULL );
	 */

	/* Make Menu sub-menu.
	n = 0;
	conf->item = add_popup_tog( popup_menu_pane, 
		conf->name, args, n, select_menu_cb, (XtPointer) conf ); 
	 */

	return( NULL );
}

/* Fill the menu pane widget.
 */
void
fill_menu_pane( void )
{
	im_list_map( menu_configs,
		(im_list_map_fn) add_menu_name, NULL, NULL );
}

/* Free a config.
 */
static void
free_config( MenuConfig *conf )
{
	im_list_remove( &menu_configs, conf );
	if( conf->item ) {
		XtDestroyWidget( conf->item );
		conf->item = NULL;
	}
	FREE( conf->name );
	if( conf->kits )
		im_list_free( &conf->kits, NULL, NULL, NULL );
	FREE( conf );
}

/* Make a new config.
 */
static MenuConfig *
new_config( char *name )
{
	MenuConfig *conf = IM_NEW( NULL, MenuConfig );

	conf->name = NULL;
	conf->kits = NULL;
	conf->item = NULL;
	im_list_add( &menu_configs, conf );
	SETSTR( conf->name, name );

	(void) add_menu_name( conf );

	return( conf );
}

/* Test for config has name.
 */
static void *
test_config_name( MenuConfig *conf, char *name )
{
	if( strcmp( conf->name, name ) == 0 )
		return( conf );
	else
		return( NULL );
}

/* Find a config.
 */
static MenuConfig *
find_config( char *name )
{
	return( (MenuConfig *) im_list_map( menu_configs,
		(im_list_map_fn) test_config_name, name, NULL ) );
}

/* Is this kit hidden? And should it be?
 */
static void *
update_menu_state( Toolkit *kit, MenuConfig *conf )
{
	/* Should be shown?
	 */
	toolkit_set_visibility( kit, 
		im_list_map( conf->kits, 
			(im_list_map_fn) im_list_eq, kit, NULL ) != NULL );

	return( NULL );
}

/* Change the menu configuration.
 */
Bool
change_menu_config( char *name )
{
	MenuConfig *conf = find_config( name );

	if( !conf ) {
		errors( "No such config \"%s\"", name );
		return( False );
	}

	/* Change state of all kits.
	 */
	toolkit_map( (toolkit_map_fn) update_menu_state, conf, NULL );

	/* Make sure the menu knows about it too.
	 */
	XmToggleButtonSetState( conf->item, True, True );

	return( True );
}

/* Remove a kit from a config.
 */
static void *
remove_kit( MenuConfig *conf, Toolkit *kit )
{
	/* Present?
	 */
	if( im_list_map( conf->kits, 
		(im_list_map_fn) im_list_eq, kit, NULL ) ) 
		/* Yes!
		 */
		im_list_remove( &conf->kits, kit );
	
	return( NULL );
}

/* Add a kit to a config.
 */
static void *
add_kit( MenuConfig *conf, char *kname )
{
	Toolkit *kit = toolkit_find( main_toolkitgroup, kname );

	/* Present?
	 */
	if( kit && !im_list_map( conf->kits, 
		(im_list_map_fn) im_list_eq, kit, NULL ) )
		/* No!
		 */
		im_list_add( &conf->kits, kit );

	return( NULL );
}

/* A kit has been removed --- make sure we remove any dangling pointers.
 */
void
menu_delete_kit( Toolkit *kit )
{
	/* Just loop for all our kits.
	 */
	im_list_map( menu_configs, 
		(im_list_map_fn) remove_kit, kit, NULL );
}

/* Structure we hold widgets in.
 */
typedef struct {
	DialogInfo *dia;		/* Handle for dialog */

	Widget configs;			/* ScrolledList for current configs */
	Widget cname;			/* Name widget */
	Widget kits;			/* ScrolledList for toolkits */
} EditmenuWidgets;

/* Add a kit name to the kit list.
 */
static void *
add_kit_name( Toolkit *kit, EditmenuWidgets *em )
{
	XmString str;

	/* Add the item.
	 */
	str = XmStringCreateLtoR( MODEL( kit )->name, charset );
	XmListAddItemUnselected( em->kits, str, 0 );
	XmStringFree( str );

	return( NULL );
}

/* Add a config name to the list.
 */
static void *
add_config_name( MenuConfig *conf, EditmenuWidgets *em )
{
	XmString str;

	/* Add the item.
	 */
	str = XmStringCreateLtoR( conf->name, charset );
	XmListAddItemUnselected( em->configs, str, 0 );
	XmStringFree( str );

	return( NULL );
}

/* Refill the kit list. If conf is non-NULL, then select items for that kit.
 */
static void *
set_kit_list( EditmenuWidgets *em )
{
	/* Clear and rebuild.
	 */
	XmListDeleteAllItems( em->kits );
	toolkit_map( (toolkit_map_fn) add_kit_name, em, NULL );

	return( NULL );
}

/* Given a kit, make sure its selection is set correctly.
 */
static void *
set_select( Toolkit *kit, EditmenuWidgets *em, MenuConfig *conf )
{
	XmString str;
	int pos;

	/* Make a string for this kit name, so we can access the item.
	 */
	str = XmStringCreateLtoR( MODEL( kit )->name, charset );
	if( !(pos = XmListItemPos( em->kits, str )) ) {
		/* Kit not on list! Strange.
		 */
		XmStringFree( str );
		return( NULL );
	}
	XmStringFree( str );

	/* Set select state.
	 */
	if( !kit->visible && XmListPosSelected( em->kits, pos ) )
		XmListDeselectPos( em->kits, pos );
	else if( kit->visible && !XmListPosSelected( em->kits, pos ) )
		XmListSelectPos( em->kits, pos, False );

	return( NULL );
}

/* Make sure the IM_RECT_RIGHT elements in the kit list are selected.
 */
static void
update_kit_selections( EditmenuWidgets *em, MenuConfig *conf )
{
	toolkit_map( (toolkit_map_fn) set_select, em, conf );
}

/* Refill the config list. 
 */
static void *
set_config_list( EditmenuWidgets *em )
{
	/* Clear and rebuild.
	 */
	XmListDeleteAllItems( em->configs );
	im_list_map( menu_configs, 
		(im_list_map_fn) add_config_name, em, NULL );

	return( NULL );
}

/* Set initial values for all widgets. This is only called during build, so we
 * can be careless about overwriting things.
 */
static void
fill_widgets( EditmenuWidgets *em )
{
	/* Set lists.
	 */
	set_kit_list( em );
	set_config_list( em );
}

/* A new config has been selected.
 */
static void
config_sel_cb( Widget wid, XtPointer client, XtPointer call ) 
{	
	decl( EditmenuWidgets *, em, client );
	decl( XmListCallbackStruct *, cb, call );

	if( cb->reason == XmCR_SINGLE_SELECT ) {
		char *cname;
		MenuConfig *conf;

		/* Get selected item and set.
		 */
		XmStringGetLtoR( cb->item, charset, &cname );
		set_texts( em->cname, cname );
		conf = find_config( cname );
		FREE( cname );

		/* New menus.
		 */
		change_menu_config( conf->name );

		/* Refresh kit list for new config.
		 */
		update_kit_selections( em, conf );
	}
}

/* Click on the Delete button.
 */
/*ARGSUSED*/
static void
delete_cb( Widget wid, XtPointer client, XtPointer call ) 
{	
	decl( EditmenuWidgets *, em, client );
	char *cname;
	MenuConfig *conf;

	/* Read current name.
	 */
	cname = XmTextGetString( em->cname );

	/* Find config.
	 */
	if( !(conf = find_config( cname )) )
		return;

	/* Free config and refresh screen.
	 */
	free_config( conf );
	set_config_list( em );
	set_texts( em->cname, "" );
}

/* Click on the Bind button.
 */
/*ARGSUSED*/
static void
bind_cb( Widget wid, XtPointer client, XtPointer call ) 
{	
	decl( EditmenuWidgets *, em, client );
	char *cname;
	MenuConfig *conf;
	XmString *selected;
	int nselected;
	int i;
	Arg args[ MAX_ARGS ];
	int n;

	/* Read current name.
	 */
	cname = XmTextGetString( em->cname );

	/* Empty?
	 */
	if( strcmp( cname, "" ) == 0 ) {
		SETSTR( cname, "untitled" );
		set_texts( em->cname, cname );
	}

	/* Exists? If not, make.
	 */
	if( !(conf = find_config( cname )) ) {
		if( !(conf = new_config( cname )) ) {
			box_alert( dia_shell( em->dia ) );
			FREE( cname );
			return;
		}

		/* Refresh screen for this new config.
		 */
		set_config_list( em );

		FREE( cname );
	}

	/* Clear show list for this config.
	 */
	im_list_free( &conf->kits, NULL, NULL, NULL );

	/* Find selected items.
	 */
	n = 0;
	XtSetArg( args[ n ], XmNselectedItemCount, &nselected ); n++;
	XtSetArg( args[ n ], XmNselectedItems, &selected ); n++;
	XtGetValues( em->kits, args, n );

	/* Rebuild show list for this config.
	 */
	for( i = 0; i < nselected; i++ ) {
		char *kname;
		Toolkit *kit;

		/* Get selected item and set.
		 */
		XmStringGetLtoR( selected[i], charset, &kname );
		if(( kit = toolkit_find( main_toolkitgroup, kname )) )
			im_list_add( &conf->kits, kit );
		FREE( kname );
	}

	/* Select the menu list.
	 */
	change_menu_config( conf->name );
}

/* Help button hit.
 */
/*ARGSUSED*/
static void
menu_help_cb( DialogInfo *dia, void *client, notify_fn nfn, void *sys )
{	
	/* Pop help.
	 */
	box_help( dia_shell( dia ), "edithelp" );

	nfn( sys, DIALOG_OK );
}

/* Build the inside of a menu edit window.
 */
static void
build_menu( DialogInfo *dia, Widget main_form, EditmenuWidgets *em )
{	
	Widget text, form, label, label2;
	Widget button, list;
	Arg args[ MAX_ARGS ];
	int n;

	/* Make form for bind + delete buttons.
	 */
	n = 0;
	XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[ n ], XmNrightAttachment, XmATTACH_POSITION ); n++;
	XtSetArg( args[ n ], XmNrightPosition, 50 ); n++;
	XtSetArg( args[ n ], XmNtopAttachment, XmATTACH_NONE ); n++;
	XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_FORM ); n++;
	form = XmCreateForm( main_form, "alkfjh", args, n );
	XtManageChild( form );

	n = 0;
	XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[ n ], XmNrightAttachment, XmATTACH_POSITION ); n++;
	XtSetArg( args[ n ], XmNrightPosition, 50 ); n++;
	XtSetArg( args[ n ], XmNtopAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_FORM ); n++;
	button = XmCreatePushButtonGadget( form, "bind", args, n );
	XtAddCallback( button, XmNactivateCallback, 
		(XtCallbackProc) bind_cb, em );
	XtManageChild( button );

	n = 0;
	XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_POSITION ); n++;
	XtSetArg( args[ n ], XmNleftPosition, 50 ); n++;
	XtSetArg( args[ n ], XmNrightAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[ n ], XmNtopAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_FORM ); n++;
	button = XmCreatePushButtonGadget( form, "delete", args, n );
	XtAddCallback( button, XmNactivateCallback, 
		(XtCallbackProc) delete_cb, em );
	XtManageChild( button );

	/* Make text widget for current config name.
	 */
	n = 0;
	XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[ n ], XmNrightAttachment, XmATTACH_POSITION ); n++;
	XtSetArg( args[ n ], XmNrightPosition, 50 ); n++;
	XtSetArg( args[ n ], XmNtopAttachment, XmATTACH_NONE ); n++;
	XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_WIDGET ); n++;
	XtSetArg( args[ n ], XmNbottomWidget, form ); n++;
	XtSetArg( args[ n ], XmNeditMode, XmSINGLE_LINE_EDIT ); n++;
	XtSetArg( args[ n ], XmNcolumns, 20 ); n++;
	em->cname = text = XmCreateText( main_form, "cname", args, n );
	XtManageChild( text );

	/* Current config label.
	 */
	n = 0;
	XtSetArg( args[ n ], XmNalignment, XmALIGNMENT_BEGINNING ); n++;
	XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[ n ], XmNrightAttachment, XmATTACH_POSITION ); n++;
	XtSetArg( args[ n ], XmNrightPosition, 50 ); n++;
	XtSetArg( args[ n ], XmNtopAttachment, XmATTACH_NONE ); n++;
	XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_WIDGET ); n++;
	XtSetArg( args[ n ], XmNbottomWidget, text ); n++;
	label = XmCreateLabelGadget( main_form, "current_config_name", args, n );
	XtManageChild( label );

	/* Configs.
	 */
	n = 0;
	XtSetArg( args[ n ], XmNalignment, XmALIGNMENT_BEGINNING ); n++;
	XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[ n ], XmNrightAttachment, XmATTACH_POSITION ); n++;
	XtSetArg( args[ n ], XmNrightPosition, 50 ); n++;
	XtSetArg( args[ n ], XmNtopAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_NONE ); n++;
	label2 = XmCreateLabelGadget( main_form, "config_names", args, n );
	XtManageChild( label2 );

	/* Make configs scrolledList.
	 */
	n = 0;
	XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[ n ], XmNrightAttachment, XmATTACH_POSITION ); n++;
	XtSetArg( args[ n ], XmNrightPosition, 50 ); n++;
	XtSetArg( args[ n ], XmNtopAttachment, XmATTACH_WIDGET ); n++;
	XtSetArg( args[ n ], XmNtopWidget, label2 ); n++;
	XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_WIDGET ); n++;
	XtSetArg( args[ n ], XmNbottomWidget, label ); n++;
	XtSetArg( args[ n ], XmNselectionPolicy, XmSINGLE_SELECT ); n++;
	XtSetArg( args[ n ], XmNvisibleItemCount, 10 ); n++;
	em->configs = list = XmCreateScrolledList( main_form, "configs", args, n );
	XtManageChild( list );
	XtAddCallback( list, XmNsingleSelectionCallback, config_sel_cb, em );

	/* Kits.
	 */
	n = 0;
	XtSetArg( args[ n ], XmNalignment, XmALIGNMENT_BEGINNING ); n++;
	XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_WIDGET ); n++;
	XtSetArg( args[ n ], XmNleftWidget, text ); n++;
	XtSetArg( args[ n ], XmNleftOffset, 4 ); n++;
	XtSetArg( args[ n ], XmNrightAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[ n ], XmNtopAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_NONE ); n++;
	label = XmCreateLabelGadget( main_form, "kits_label", args, n );
	XtManageChild( label );

	/* Make show kits scrolledList.
	 */
	n = 0;
	XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_WIDGET ); n++;
	XtSetArg( args[ n ], XmNleftWidget, text ); n++;
	XtSetArg( args[ n ], XmNleftOffset, 4 ); n++;
	XtSetArg( args[ n ], XmNrightAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[ n ], XmNtopAttachment, XmATTACH_WIDGET ); n++;
	XtSetArg( args[ n ], XmNtopWidget, label ); n++;
	XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[ n ], XmNselectionPolicy, XmMULTIPLE_SELECT ); n++;
	em->kits = list = XmCreateScrolledList( main_form, "tools", args, n );
	XtManageChild( list );

	/* Fill widgets up.
	 */
	fill_widgets( em );
}

/* Display a help page named by client.
 */
/*ARGSUSED*/
static void
ok_menu_conf_cb( DialogInfo *dia, void *client, notify_fn nfn, void *sys )
{
	/* Write iprc again.
	 */
	if( !iprc_save() )
		box_alert( NULL );

	nfn( sys, DIALOG_OK );
}

/* Pop up a menu edit window.
 */
void
pop_menu_edit( Widget par )
{	
	EditmenuWidgets *em = IM_NEW( NULL, EditmenuWidgets );

	em->dia = run_dialog( par, our_resources.tit_menuedit, 
		False, False, DIALOG_MESSAGE,
		ok_menu_conf_cb, NULL, menu_help_cb, NULL, em, 
		(build_fn) build_menu, em, NULL, NULL,
		null_notify_fn, NULL );
}

/* Append a char* to another string, enclosing it in <> brackets. We have to
 * escape any ">" charcters in the string itself.
 */
void *
write_item( char *in, char *out )
{
	char *p, *q;

	q = out + strlen( out );
	*q++ = '<';

	for( p = in; *p; p++ ) {
		if( *p == '>' ) {
			*q++ = '\\';
			*q++ = '>';
		}
		else
			*q++ = *p;
	}
	*q++ = '>';
	*q++ = ' ';
	*q++ = '\0';

	return( NULL );
}

/* Append a <kit1> string to buf2.
 */
static void *
append_kname( Toolkit *kit, char *buf2 )
{
	(void) write_item( MODEL( kit )->name, buf2 );

	return( NULL );
}

/* Append a MENU line for this config.
 */
static void *
append_cname( MenuConfig *conf, char *buf )
{
	char buf2[4096];

	/* Make <config-name> <kit1> <kit2> etc. string.
	 */
	im_snprintf( buf2, 4096, "<%s> ", conf->name );
	im_list_map( conf->kits,
		(im_list_map_fn) append_kname, buf2, NULL );

	/* Is this the first entry in buf? If yes, suppress the MENU key.
	 */
	if( *buf )
		strcat( buf, "MENU " );

	/* Append buf2 to buf.
	 */
	strcat( buf, buf2 );
	strcat( buf, "\n" );

	return( NULL );
}

/* Return a string which iprc can save.
 */
char *
menu_save( void )
{
	char buf[ 4096 ];

	strcpy( buf, "" );
	if( menu_configs )
		im_list_map( menu_configs,
			(im_list_map_fn) append_cname, buf, NULL );
	else
		/* No menu configs set --- just add a dummy config.
		 */
		strcat( buf, "<None> \n" );
	
	return( im_strdup( NULL, buf ) );
}

/* Read a string from a string, enclosed in "<" and ">" brackets. Return
 * an updated pointer, or NULL on error or end-of-string. Look for "\>" in
 * string, and turn back to ">".
 */
char *
read_item( char *in, char *out )
{
	/* Skip leading whitespace.
	 */
	while( isspace( *in ) )
		in++;
	if( *in == '\0' ) 
		return( NULL );
	else if( *in != '<' ) {
		errors( "\"<\" not found" );
		return( NULL );
	}
	in++;

	/* Copy to out until we hit ">". Look out for escaped embedded ">"
	 * characters.
	 */
	for( ; *in != '>'; in++ ) 
		switch( *in ) {
		case '\\':
			if( in[1] == '>' ) {
				*out++ = '>';
				in++;
			}
			else
				*out++ = '\\';
			break;

		case '\0':
			errors( "Closing \">\" not found" );
			return( NULL );

		default:
			*out++ = *in;
			break;
		}

	/* Add trailing '\0'.
	 */
	*out = '\0';

	return( in + 1 );
}

/* Load a menu config. Of the form <config-name> <kit-1> <kit-2> ...
 */
Bool
menu_load( char *str )
{
	char cname[ 4096 ];
	char kname[ 4096 ];

	MenuConfig *conf;
	Toolkit *kit;

	/* Read config name.
	 */
	if( !(str = read_item( str, cname )) )
		return( False );

	/* Exists? If not, make.
	 */
	if( !(conf = find_config( cname )) && !(conf = new_config( cname )) )
		return( False );

	/* Clear show list for this config.
	 */
	im_list_free( &conf->kits, NULL, NULL, NULL );

	/* Repeatedly add to show list for config.
	 */
	for(;;) {
		if( !(str = read_item( str, kname )) )
			break;
		if( (kit = toolkit_new( main_toolkitgroup, kname )) )
			im_list_add( &conf->kits, kit );
	}

	return( True );
}
