/*
 * ngetty-helper.c
 *
 * Copyright 2007,2008 Nikola Vladov
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#define _GNU_SOURCE

#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <time.h>
#include <alloca.h>
#include <errno.h>
#include <termios.h>
#include <sys/utsname.h>
#include <sys/ioctl.h>
#include <sys/param.h>
#include "lib.h"
#include "opts_defs.h"

#define outs(X) out_puts(X)

static struct utsname uts;
static struct stat st;
static char *tty, *o_opt, o_endchar; 
char **Oo = o;

static void tty_flush() { ioctl(0, TCFLSH, TCIFLUSH); }

void error(char *s,int i) {
  struct timespec ts = { 0, 600111222 };
  outs("\nngetty-helper: "); outs(s); out_char('\n'); out_flush();
  nanosleep(&ts, 0);
  tty_flush();
  _exit(i);
}

static int char_allow(unsigned char ch) {
  unsigned char c0, c1, *x = (unsigned char *)o[Oprint];
 again:
  c0 = x[0]; if (!c0) return -1;
  c1 = x[1]; if (!c1) return -1;
  if (c0 <= ch && ch <= c1) return 0;
  if (x[2] == ':') { x += 3; goto again; }
  return -1;
}

static void out_max(char *s, unsigned int len) {
  char c = 0;
  if (s == uts.nodename && !o[Olonghostname]) c = '.';
  while (len && *s && *s != c) {
    out_char(*s); ++s; --len;
  }
}

#define Out_MAX(X) x=X; len=sizeof(X); goto system

static void output_special_char(char *s, int len) {
  char *last = s + len;
  while (s < last) {
    char *x, c;
    if (*s != '\\') { out_char(*s); ++s; continue; }
    if (++s == last) return;

    c = *s++;
    switch (c) {
    case 's': Out_MAX(uts.sysname);
    case 'n': Out_MAX(uts.nodename);
    case 'r': Out_MAX(uts.release);
    case 'v': Out_MAX(uts.version);
    case 'm': Out_MAX(uts.machine);
    case 'o': Out_MAX(uts.domainname);
  system:
    out_max(x,len); break;
    case 'l':
      outs(tty); break;
#ifndef DONT_NEED_ESC_TIME
    case 't':
      x = "H:M:S"; goto do_time;
    case 'd':
      x = (o[Odatestring]) ? "a b d Y" : "Y-m-d";
    do_time:
      fmt_time(x); break;
#endif
#ifndef DONT_NEED_ESC_USERS
    case 'u':
    case 'U':
      {
	int fd=open(_PATH_UTMP, O_RDONLY);
	if (fd >= 0) {
	  struct utmp ut;
	  for (len=0; utmp_io(fd, &ut, F_RDLCK); )
	    if (ut.ut_type == USER_PROCESS) len++;
	  close(fd);
	  
	  x = (char *)&ut;
	  x += fmt_ulong(x, len);
	  if (c=='U') {
	    x += str_copynz(x, " users", 10);
	    if (len==1) --x;
	  }
	  *x = 0;
	  outs((char *)&uts);
	}
      }
      break;
#endif
    default:
      if (s+1 < last) {
	unsigned char uch = (unsigned char)c-'0';
	if (uch < 4) {	/* XXX \0xy is not tested */
	  c = (uch<<6) | ((s[0]-'0')<<3) | (s[1]-'0');
	  s += 2;
	}
      }
      out_char(c);
    }
  }
}

static void Out_sp_ch(char *s) { output_special_char(s, str_len(s)); }

static void do_prompt() {
  int fd,len;
  char *s;
  Out_sp_ch(o[Onewline]); 
  s = o[Oissuefile];
  if (s[0] && (fd=open(s, O_RDONLY)) >= 0) {
    if (GLOBAL_fstat_READ(fd,st,s, len,64000, 1)) _exit(1);
    close(fd);
    output_special_char(s, len);
  }
  Out_sp_ch(o[Ologinprompt]);
  out_flush();
}

static char *get_logname() {
  static char logname[40];
  char *c, *last=logname + sizeof(logname);
  alarm(x_atoi(o[Otimeout]));
  for (*logname=0; *logname==0; ) {
    for (c=logname;;) {
      if (read(0,c,1)<1) {
	if (errno==EINTR || errno==EIO || errno==ENOENT)
	  _exit(0);
	error("received strange error",9);
      }
      if (*c == '\n' || *c == '\r') {
	*c=0;
	break;
      }
      if (char_allow((unsigned char)*c))
	error("bad character in login name",10);
      if (++c == last)
	error("login name too long",11);
    } /* end read */
    if (*logname == '-')
      error("username may not start with a dash",13);
    if (*logname == 0)
      do_prompt();
  }
  alarm(0);
  return logname;
}

static char *opt_get(char *s, int len) {
  if (*o_opt == *s) {
    char *x = o_opt;
    if (!str_diffn(x,s,len) && x[len] == o_endchar) {
      if (o[Odebug] && s != tty) { 
	outs(x - 1);
	out_char('\n');
      }
      return x+len+1;
    }
  }
  return 0;
}

static char in_list(char *s, const char *user) {
  char ch;
  int len;
  if (s==0 || *s==0 || user==0) return 0;
  len = str_len(user);
  ch = *s++;
 again:
  if (!str_diffn(s,user,len))
    if (s[len]==ch || s[len]==0) return 1;
  while (*s && *s != ch) s++;
  if (*s++) goto again;
  return 0;
}

int main(int argc,char *argv[]) {
  int fd,len;
  char *s;
  if (argc < 2 || dup(0) != 1 || dup(0) != 2) _exit(100);

  o[Oautologinfirst] = o[Onoclearfirst] = o[Oautologinname] = "";
  o[Oissuefile] = "/etc/issue";
  o[Ologinprog] = LOGIN;
  o[Oprint] = "az:AZ:09:,.:__";
  o[Otimeout] = "180";
  o[Ologinprompt] = "\\n login: ";
  o[Onewline] = "\n";
  o[Oclear] = "\033c";
#ifndef DONT_NEED_ESC_TIME
  o[Omonths] = "JanFebMarAprMayJunJulAugSepOctNovDec";
  o[Odays] = "SunMonTueWedThuFriSat";
  o[Otz] = "/etc/localtime";
#endif

  tty = argv[1];
  if (!str_diffn(tty, "/dev/", 5)) tty += 5;

  if ((fd = open(NGETTY_HOME "/Conf", O_RDONLY)) >=0) {
    char **aa, *q;
    int tty_len = str_len(tty);
    /* 40 bytes for Months or Days */
    if (GLOBAL_fstat_READ(fd,st,s, len,64000, 40)) _exit(1);
    close(fd);
    if (st.st_uid || st.st_gid || (st.st_mode & 07177)) len = 0;
    s[len]=0;
    
    len = splitmem(0,s,'\n');
    aa=alloca((len+1) * sizeof(char*)); if (aa==0) _exit(1);
    splitmem(aa,s,'\n');

    for (; *aa; aa++) {
      s=aa[0];
      if (*s==0 || *s=='#' || *s=='\t' || *s==' ') continue;

      o_endchar = '=';
      if (*s=='=') o_opt = s+1;
      else {
	o_opt = s;
	if (!(o_opt = opt_get(tty, tty_len))) continue;
      }

      for (len=0, q=(char *)P_opt; len < MAX_options; len++) {
	int k = P_len[len];
	if (len >= MAX_equal) o_endchar = 0;
	if ((s = opt_get(q, k))) { o[len] = s; break; }
	q += k;
      }
    }
  }

  if ((s=o[Onice])) nice(x_atoi(s));
  uname(&uts);

  if ((s=o[Oenviron])) {
    char ch=*s++;
    if (ch) {
      len=splitmem(0,s,ch);
      environ=alloca((len+1) * sizeof(char*)); if (environ==0) _exit(1);
      splitmem(environ,s,ch);
    } else
      environ=0;
  }
#ifdef HAVE_SYSTEM_VHANGUP
  if (!o[Onohangup]) vhangup();
#endif
  if ((s=o[Ochroot])) chroot(s);
  if ((s=o[Ochdir])) chdir(s);

  if (argc>2 && argv[2][0] == 'l') {
    char *loginargv[]={ o[Ologinprog], "--", 0, 0, 0 };

    fork_and_exec(o[OshB]);

    s = o[Oautologinfirst];
    if (*s && !check_first(s,&st)) o[Oautologinname] = "";

    s = o[Oautologinname];
    if (*s) {
      loginargv[1] = "-f";
      loginargv[2] = s;
    } else {
      o[Oissuefile] = "";
      if (!o[Onousername])
	loginargv[2] = get_logname();
    }

    if ((s=o[Ologinargv])) {
      loginargv[1] = s;	/* change '--' or '-f' with login-argv */
      loginargv[3] = tty;
    }

    s = loginargv[2];
    if (in_list(o[Odeny],s) ||
	(o[Oallow] && in_list(o[Oallow],s)==0))
      error("login is disabled",14);

    out_flush();
    setsid();
    ioctl(0, TIOCSCTTY, (void *)1);
    tty_flush();

    utmp_do(1,tty);
    if (o[Oechooff]) {
      struct termios old;
      if (ioctl(0, TCGETS, &old)==0) {
	old.c_lflag &= ~ECHO;
	ioctl (0, TCSETS , &old);
      }
    }
    execve(*loginargv, loginargv, environ);
    _exit(127);
  } else {
    char *clear = o[Oclear];
    tty_flush();

    fork_and_exec(o[OshA]);
    if ((s=o[Odelay])) {
      struct timespec ts = { x_atoi(s), 100200300 };
      nanosleep(&ts, 0);
    }

    s = o[Onoclearfirst];
    if (*s && check_first(s,&st)) clear = "";

    Out_sp_ch(clear);
    utmp_do(0,tty);
    do_prompt();
  }
  _exit(0);
}
