/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/



// PKIX_Central.cpp: implementation of the PKIX_Central class.
//
//////////////////////////////////////////////////////////////////////

#include "PKIX_Central.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

PKIX_Central::PKIX_Central(const Config & Conf)
{
	ERR_clear_error();

	m_Engine = NULL;
	m_AuthModule = NULL;
	LocalServer = NULL;
	mString InitialDb(INITIAL_DB);


	m_Conf = Conf;


	//Load the authentication module if we have one
	if(m_Conf.get_AuthLib().size())
	{
		try
		{
			m_AuthModule = new AuthModule(m_Conf.get_AuthLib(), m_Conf.get_AuthCmd());
		}
		catch(ExceptionNewPKI e)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			Clear();
			throw ExceptionNewPKI();
		}
		if(!m_AuthModule)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_MALLOC);
			Clear();
			throw ExceptionNewPKI();
		}
	}

	//Loading engine is necessary
	if(!LoadEngine(m_Conf.get_Engine(), m_Conf.get_EngineCmd()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Clear();
		throw ExceptionNewPKI();
	}

	//We create the local server
	try
	{
		LocalServer = new Entity_SERVER(m_Conf, InitialDb, m_Engine, m_AuthModule, 0);
	}
	catch(ExceptionNewPKI e)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Clear();
		throw ExceptionNewPKI();
	}
	if(!LocalServer)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_MALLOC);
		Clear();
		throw ExceptionNewPKI();
	}

	bool Exists;
	if(!LocalServer->EntityExists(Exists))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Clear();
		throw ExceptionNewPKI();
	}
	//If entity exists we upgrade it
	if(!Exists)
	{
		//We create it
		if(!LocalServer->Create(EntityCreationDatas::EmptyInstance, AdminResponseBody::EmptyInstance))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			Clear();
			throw ExceptionNewPKI();
		}
	}
	//We load it
	if(!LocalServer->Load())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Clear();
		throw ExceptionNewPKI();
	}
}

PKIX_Central::~PKIX_Central()
{
	Clear();
}

void PKIX_Central::Clear()
{
	if(LocalServer)
		delete LocalServer;
	if(m_Engine)
		ENGINE_free(m_Engine);
	if(m_AuthModule)
		delete m_AuthModule;
}

bool PKIX_Central::LoadEngine(const mString & engine, const mString & EngineCmd)
{
	ERR_clear_error();
	vector<mString> cmds;
	size_t i;
	int pos;
	mString Name;
	mString Value;

	m_Engine = NULL;

	if(!engine.size())
		return true;


	if(engine == "auto")
	{
		ENGINE_register_all_complete();
		return true;
	}

	if((m_Engine = ENGINE_by_id(engine.c_str())) == NULL)
	{
		// On va essayer de la charger dynamiquement
		m_Engine = ENGINE_by_id("dynamic");
		if (!m_Engine)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}

		ERR_clear_error();
		if (!ENGINE_ctrl_cmd_string(m_Engine, "SO_PATH", engine.c_str(), 0))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}

		if (!ENGINE_ctrl_cmd_string(m_Engine, "LOAD", NULL, 0))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	ERR_clear_error();
    
	if(!ENGINE_set_default(m_Engine, ENGINE_METHOD_ALL))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// we execute the commands
	if(EngineCmd.size())
	{
		EngineCmd.Split(", ", cmds);
		for(i=0; i<cmds.size(); i++)
		{
			pos = cmds[i].find(":", 0);
			if(pos == -1)
			{
				if( !ENGINE_ctrl_cmd_string(m_Engine, cmds[i].c_str(), NULL, 0) )
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			else
			{
				Name = cmds[i].Left(pos);
				Value = cmds[i].Right(cmds[i].size() - pos - 1);
				if( !ENGINE_ctrl_cmd_string(m_Engine, Name.c_str(), Value.c_str(), 0) )
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
		}
	}

    return true;
}

void PKIX_Central::GetSSL_Cert(PKI_CERT & SslCert)
{
	LocalServer->GetEntityCertificate(SslCert);
}

bool PKIX_Central::OnADMIN_REQUEST(AdminResponse & response, const char * ClientIp, const PKI_CERT & ClientCert, const AdminRequest &request)
{
	return LocalServer->OnADMIN_REQUEST(response, ClientIp, ClientCert, request);
}

void PKIX_Central::OnConnectionClosed()
{
	LocalServer->OnConnectionClosed();
}

bool PKIX_Central::OnNewOCSP(const char * Ip, OCSP_REQUEST *request, OCSP_RESPONSE **response, const mString &EntityName, bool & ShouldStopServer)
{
	return LocalServer->OnNewOCSP(Ip, request, response, EntityName, ShouldStopServer);
}

void PKIX_Central::SetOcspServersHandler(OcspServers *OcspServersHandler)
{
	LocalServer->SetOcspServersHandler(OcspServersHandler);
}

void PKIX_Central::PrintInfo(FILE *out)
{
	LocalServer->PrintInfo(out);
}
