/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/

// Entity_RA.cpp: implementation of the Entity_RA class.
//
//////////////////////////////////////////////////////////////////////

#include "Entity_RA.h"
#include "svintl.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

Entity_RA::Entity_RA(ENTITY_CONSTRUCTOR_PARAMETERS):
			Entity(ENTITY_CONSTRUCTOR_PARAM_PASSTHRU, &myConf, &lStore,
				ASYNCHMSGS_TYPE_REQUESTER | ASYNCHMSGS_TYPE_RESPONDER | ASYNCHMSGS_RESPONDER_ASYNCH),
			lStore(EntityName, e)
{
	LdapSynch.SetEventHandler(&lStore);
	hThreadCheckExpire.Create(ThreadCheckExpire, this);
}

Entity_RA::~Entity_RA()
{
	hThreadCheckExpire.Stop();
	m_Jobs.StopAll();
}

bool Entity_RA::Create(const EntityCreationDatas & Params, AdminResponseBody & response)
{
	if(!Params)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}
	if(Params.get_type() != ENTITY_TYPE_RA)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	if(!Params.get_entityKey())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	//We create the database
	if(!Common_Create(Params.get_entityKey(), NULL))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!response.get_creEntity().set_type(ENTITY_TYPE_RA))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}
	if(!response.get_creEntity().set_entityPubKey(m_EntityKey.GetPublicKey()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_MALLOC);
		Destroy();
		return false;
	}

	return true;
}

bool Entity_RA::Load()
{
	if(!Common_Load())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!IsFullyInit())
	{
		return true;
	}

	LdapSynch.SetLogging(m_Logging);

	lStore.SetAclValidator(&AclValidator);

	return true;
}

bool Entity_RA::Init(const EntitySignatureResp & init_datas)
{
	if(IsFullyInit())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!init_datas)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}
	if(init_datas.get_body().get_type() != ENTITY_TYPE_RA)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}
	if(!Common_Init(init_datas.get_body().get_entitycert(), init_datas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool Entity_RA::LoginUser(UserHandle & hUser, int & UserType)
{
	if(!AclValidator.CanUserPerform(hUser.GetUserCert(), ACL_TYPE_AUTHENTICATE_ON_ENTITY))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	UserType = USER_TYPE_RA;
	return true;
}

void Entity_RA::LogoutUser(const UserHandle & hUser)
{
}

bool Entity_RA::ParseNewConf()
{
	mString Err;

	if(!m_Policies.From_POLICY_VALUE(myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_policies()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	lStore.set_Policies(m_Policies);

	if(!lStore.SetGroups(myConf.get_conf().get_groups()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	lStore.SetMinPasswdLen(myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_minpwdlen());
	
	if(ASN1_BIT_STRING_get_bit(myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_flags(), RA_ALLOW_LDAP_PROFILE_IMPORT))
	{
		if(!LdapClient.Connect(myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapServer(), 
				myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapPort(), 
				myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapUsername(), 
				myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapPassword(), 
				myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapBase(), 
				myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapAttrName(),
				myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapUtf8()))
		{
			ERR_to_mstring(Err);
			m_Logging->LogMessage(LOG_STATUS_TYPE_FAILURE, LOG_MESSAGE_TYPE_LDAP_CONNECTION, 0, myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapUsername().c_str(), LOG_NO_OBJECTID, myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapServer().c_str(), Err.c_str());
		}
		else
		{
			m_Logging->LogMessage(LOG_STATUS_TYPE_SUCCESS, LOG_MESSAGE_TYPE_LDAP_CONNECTION, 0, myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapUsername().c_str(), LOG_NO_OBJECTID, myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapServer().c_str());
		}

		if(ASN1_BIT_STRING_get_bit(myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_flags(), RA_ALLOW_LDAP_AUTO_SYNCH))
		{
			if(!LdapSynch.SetOptions(
				(ASN1_BIT_STRING_get_bit(myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_flags(), RA_REMOVE_UNKNOWN_DN_FIELDS) == 1),
				myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_dnspecs(),
				m_Policies,
				myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapProfilesGroup(),
				myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapFilters()))
			{
				ERR_to_mstring(Err);
				m_Logging->LogMessage(LOG_STATUS_TYPE_FAILURE, LOG_MESSAGE_TYPE_LDAP_AUTO_SYNCH_START, 0, myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapUsername().c_str(), LOG_NO_OBJECTID, myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapServer().c_str(), Err.c_str());
			}
			else
			{
				if(!LdapSynch.Start(myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapServer(), 
						myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapPort(), 
						myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapUsername(), 
						myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapPassword(), 
						myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapBase(), 
						myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapAttrName(),
						myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapUtf8()))
				{
					ERR_to_mstring(Err);
					m_Logging->LogMessage(LOG_STATUS_TYPE_FAILURE, LOG_MESSAGE_TYPE_LDAP_AUTO_SYNCH_START, 0, myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapUsername().c_str(), LOG_NO_OBJECTID, myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapServer().c_str(), Err.c_str());
				}
				else
				{
					m_Logging->LogMessage(LOG_STATUS_TYPE_SUCCESS, LOG_MESSAGE_TYPE_LDAP_AUTO_SYNCH_START, 0, myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapUsername().c_str(), LOG_NO_OBJECTID, myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapServer().c_str());
				}
			}
		}
		else
		{
			LdapSynch.Stop();
		}
	}
	else
	{
		LdapSynch.Stop();
	}


	if(ASN1_BIT_STRING_get_bit(myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_flags(), RA_WARN_PROFILE_OWNER_EXPIRING))
	{
		if(!hThreadCheckExpire.IsRunning() && !hThreadCheckExpire.Start())
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
	}
	else
	{
		hThreadCheckExpire.Stop();
	}
	return true;
}

bool Entity_RA::Upgrade(const char * Version)
{
	LocalRaConfBeta4 lconf;
	Asn1EncryptSign encrypt;
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);

	if(!Entity::Common_Upgrade(Version))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	while(strcmp(Version, NEWPKI_VERSION) != 0)
	{
		if(strcmp(Version, "2.0.0-beta4") == 0)
		{
			if(!CreateResponder(m_DbConn))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			if(!Load_Conf(&encrypt))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			if(!lconf.from_SignEncrypt(encrypt, m_EntityKey.GetRsaKey(), m_EntityKey.GetRsaKey()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			if(!Upgrade_EntityConf_From_Beta4_To_2_0_0(
					lconf.get_conf(), myConf.get_conf()) )
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			if(!myConf.set_cas(lconf.get_cas()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}

			// Upgrade the profiles store
			if(!sql.Execute("ALTER TABLE `profiles` ADD `ee_id` TEXT NOT NULL AFTER `profile_id`;"))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			if(!sql.Execute("ALTER TABLE `profiles` ADD `ee_tid` VARCHAR(41) NOT NULL AFTER `ee_id`;"))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			if(!sql.Execute("ALTER TABLE `profiles` ADD INDEX (ee_tid);"))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}

			if(!sql.Execute("ALTER TABLE `certs` ADD `ee_tid` VARCHAR(41) NOT NULL AFTER `profile_id`;"))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			if(!sql.Execute("ALTER TABLE `certs` ADD INDEX (ee_tid);"))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			Version = "2.0.0-rc1";
		}
		NewpkiDebug(LOG_LEVEL_DEBUG, m_EntityName.c_str(), _sv("Upgraded to version %s"), Version);
	}

	if(!WritePersonnalConf())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}


void Entity_RA::GetACL_List(mVector<unsigned long> & acl_list)
{
	int i;
	static ACL_TYPE list_acls[] =
	{
		ACL_TYPE_VIEW_LOGS,
		ACL_TYPE_SEND_ADMIN_MAIL,
		ACL_TYPE_RA_MANAGE_PROFILES,
		ACL_TYPE_RA_DELEGATE_PROFILE_OWNERSHIP,
		ACL_TYPE_RA_REQUEST_CERT,
		ACL_TYPE_RA_REVOKE_CERT,
		(ACL_TYPE)0
	};
	for(i=0; list_acls[i]; i++)
	{
		acl_list.push_back(list_acls[i]);
	}	
}


bool Entity_RA::ParseAdminCommand(AdminResponseBody & response, const PKI_CERT & ClientCert, const AdminRequest & AdminRequest)
{
	//We only accept SSLv3 connection	
	if(!ClientCert)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ERR_to_ADMIN_RESPONSE(response);
		return false;
	}
	return Private_ParseAdminCommand(true, this, LogsType, response, ClientCert, AdminRequest, GetUserHandle());
}

bool Entity_RA::Private_ParseAdminCommand(bool ExecuteCmd, Entity * me_this, mVector<unsigned long> &  mLogsType, AdminResponseBody & response, const PKI_CERT & ClientCert, const AdminRequest & AdminRequest, UserHandle & hUser)
{
	PARSER_COMMAND_BEGIN(Entity_RA, response, 0, AdminRequest, ClientCert, hUser, ExecuteCmd, me_this, mLogsType)
		PARSER_ADD_LOG_ENTRY(LOG_MESSAGE_TYPE_ENTITY_GET_MY_CONF)
		PARSER_ADD_LOG_ENTRY(LOG_MESSAGE_TYPE_SEND_ADMIN_MAIL)
		PARSER_ADD_LOG_ENTRY(LOG_MESSAGE_TYPE_SEND_MAIL)
		PARSER_ADD_LOG_ENTRY(LOG_MESSAGE_TYPE_CERT_SIG_RESP)
		PARSER_ADD_LOG_ENTRY(LOG_MESSAGE_TYPE_CERT_REV_RESP)
		PARSER_ADD_LOG_ENTRY(LOG_MESSAGE_TYPE_CERT_RNW_RESP)
		PARSER_ADD_LOG_ENTRY(LOG_MESSAGE_TYPE_LDAP_AUTO_SYNCH_START)
		PARSER_ADD_LOG_ENTRY(LOG_MESSAGE_TYPE_LDAP_AUTO_SYNCH_IMPRT)
		PARSER_ADD_LOG_ENTRY(LOG_MESSAGE_TYPE_LDAP_CONNECTION)
		PARSER_ADD_LOG_ENTRY(LOG_MESSAGE_TYPE_EE_DN_PUBLISH)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_LOGIN,					Entity_RA::UserLogin,				LOG_MESSAGE_TYPE_USER_LOGIN, (ClientCert)?(char*)ClientCert.GetStringName():NULL, LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_ENUM_LOGS,				Entity_RA::EnumLogs)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_GET_LOGS_COUNT,			Entity_RA::GetLogsCount)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_GET_MY_ACL,				Entity_RA::GetMyACL)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_GET_LOGS_TYPE,			Entity_RA::GetLogsType)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_SEND_ADMIN_MAIL,			Entity_RA::AdminSendMail,			LOG_MESSAGE_TYPE_ADD_ADMIN_MAIL_QUEUE,	_sv("none"), LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_CHECK_LOGS,				Entity_RA::CheckLogsIntegrity)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_GET_LOCAL_CONF,			Entity_RA::GetLocalConf,			LOG_MESSAGE_TYPE_GET_LOCAL_CONF,	_sv("none"), LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_GET_GROUPS,				Entity_RA::GetUsersGroups)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_IMPORT_PROFILE,			Entity_RA::ImportProfile,			LOG_MESSAGE_TYPE_IMPORT_PROFILE,	_sv("none"), LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_ENUM_PROFILES,			Entity_RA::EnumProfiles)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_REQUEST_CERT,			Entity_RA::RequestCert,				LOG_MESSAGE_TYPE_CERT_REQUEST,	"", body.get_certRequest().get_profileId())
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_REVOKE_CERT,				Entity_RA::RevokeCert,				LOG_MESSAGE_TYPE_CERT_REVOCATION,	"", body.get_serial())
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_DELETE_PKCS12,			Entity_RA::DeletePkcs12,			LOG_MESSAGE_TYPE_DELETE_PKCS12,	"", body.get_serial())
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_SEARCH_LDAP,				Entity_RA::SearchLdap)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_CHANGE_PROFILE_UID,		Entity_RA::ChangeProfileUID,		LOG_MESSAGE_TYPE_PROFILE_CHANGE_UID, "", body.get_profileUid().get_profileId())
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_CHANGE_PROFILE_OWNER,	Entity_RA::ChangeProfileOwner,		LOG_MESSAGE_TYPE_PROFILE_CHANGE_OWNER, "", body.get_profileOwner().get_profileId())
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_DELETE_PROFILE,			Entity_RA::DeleteProfile,			LOG_MESSAGE_TYPE_DELETE_PROFILE, "", body.get_profileId())
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_CHANGE_PROFILE_DN,		Entity_RA::ChangeProfileDN,			LOG_MESSAGE_TYPE_PROFILE_CHANGE_DN, "", body.get_profileDn().get_profileId())
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_DELETE_REQUEST,			Entity_RA::DeleteRequest,			LOG_MESSAGE_TYPE_DELETE_PROFILE, "", body.get_requestId())
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_ACCEPT_PROFILE,			Entity_RA::AcceptProfile,			LOG_MESSAGE_TYPE_PROFILE_ACCEPT, "", body.get_profileId())
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_REJECT_PROFILE,			Entity_RA::RejectProfile,			LOG_MESSAGE_TYPE_PROFILE_REJECT, "", body.get_rejectProfile().get_profileId())
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_SUSPEND_CERT,			Entity_RA::SuspendCert,				LOG_MESSAGE_TYPE_CERT_SUSPENSION,	"", body.get_serial())
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_UNSUSPEND_CERT,			Entity_RA::UnsuspendCert,			LOG_MESSAGE_TYPE_CERT_UNSUSPENSION,	"", body.get_serial())
	PARSER_COMMAND_END(Entity_RA)
}

void Entity_RA::LogsTypeGet(mVector<unsigned long> &  cLogsType)
{
	Private_ParseAdminCommand(false, NULL, cLogsType, AdminResponseBody::EmptyInstance, PKI_CERT::EmptyInstance, AdminRequest::EmptyInstance, UserHandle::EmptyInstance);
}


bool Entity_RA::GetLocalConf(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(AclValidator.ValidateCert(UserCert) != INTERNAL_CA_TYPE_USER)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_LOCAL_ENTITY_CONF))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockRead();
	if( !response.set_localEntityConf(myConf.get_conf()) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		return false;
	}
	ConfAccessLock.UnlockRead();

	// We don't send the LDAP info, those are very private datas
	response.get_localEntityConf().get_body().get_raConf().RA_CONF_PTR.set_ldapServer("");
	response.get_localEntityConf().get_body().get_raConf().RA_CONF_PTR.set_ldapUsername("");
	response.get_localEntityConf().get_body().get_raConf().RA_CONF_PTR.set_ldapPassword("");
	response.get_localEntityConf().get_body().get_raConf().RA_CONF_PTR.set_ldapBase("");
	response.get_localEntityConf().get_body().get_raConf().RA_CONF_PTR.set_ldapPort(0);
	response.get_localEntityConf().get_body().get_raConf().RA_CONF_PTR.set_ldapProfilesGroup(0);
	response.get_localEntityConf().get_body().get_raConf().RA_CONF_PTR.set_ldapFilters("");
	response.get_localEntityConf().get_body().get_raConf().RA_CONF_PTR.set_ldapAttrName("");

	return true;
}

bool Entity_RA::GetUsersGroups(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(AclValidator.ValidateCert(UserCert) != INTERNAL_CA_TYPE_USER)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_GROUPS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockRead();
	if(!response.set_groups(myConf.get_conf().get_groups()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		return false;
	}
	ConfAccessLock.UnlockRead();
	return true;
}


bool Entity_RA::ImportProfile(COMMAND_PARAMETERS)
{
	NewpkiRequest Request;
	long ProfileId;
	const X509_PUBKEY * recipient=NULL;


	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_RA_MANAGE_PROFILES))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	// Generate the request
	if(!Request.set_type(NEWPKI_REQUEST_TYPE_EE_VAL))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}


	// Did the user delegate ownership of profile,
	// and is he allowed to ?
	if(body.get_profile().get_ownerGroupSerial() &&  
		!AclValidator.CanUserPerform(UserCert, ACL_TYPE_RA_DELEGATE_PROFILE_OWNERSHIP))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ERR_add_error_data(1, _sv("Delegate profile's ownership"));
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_PROFILE_ID))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	

	// TODO: Reorder the DN !

	if(!lStore.InsertProfile(UserCert, body.get_profile(), ProfileId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// If we require user validation on this profile
	// then we must send the profile to the EE
	if(body.get_profile().get_eeValidation())
	{
		ConfAccessLock.LockRead();
		if(!myConf.get_conf().get_ees().get_list().size())
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
			ERR_add_error_data(1, _sv("You haven't configured an EE for this RA"));
			ConfAccessLock.UnlockRead();
			return false;
		}
		recipient = myConf.get_conf().get_ees().get_list()[0].get_eessl().GetX509_PUBKEY();
		if(!recipient)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			ConfAccessLock.UnlockRead();
			return false;
		}
		Request.get_eeValRequest().set_raId(ProfileId);
		if(!Request.get_eeValRequest().set_dn(body.get_profile().get_dn()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			lStore.DeleteProfile(PKI_CERT::EmptyInstance, ProfileId, false);
			ConfAccessLock.UnlockRead();
			return false;
		}
		if(!InsertRequest(ProfileId, Request, recipient))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			lStore.DeleteProfile(PKI_CERT::EmptyInstance, ProfileId, false);
			ConfAccessLock.UnlockRead();
			return false;
		}
		ConfAccessLock.UnlockRead();
	}
	response.set_id(ProfileId);
	return true;
}


bool Entity_RA::EnumProfiles(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_RA_MANAGE_PROFILES))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_PROFILES))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!lStore.EnumProfiles(UserCert, response.get_profiles(), 
							body.get_enumObjects().get_index(), 
							body.get_enumObjects().get_num(), 
							body.get_enumObjects().get_state(), 
							body.get_enumObjects().get_filter()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}


bool Entity_RA::SearchLdap(COMMAND_PARAMETERS)
{
	mString Filters;
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_RA_MANAGE_PROFILES))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!LdapClient)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_LDAP_NOT_CONFIGURED);
		return false;
	}

	ConfAccessLock.LockRead();
	if(myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapFilters().size())
	{
		if(Filters.sprintf("(&%s%s)", 
							myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_ldapFilters().c_str(), 
							body.get_ldapSearch().c_str()) <= 0)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
		}
	}
	else
	{
		Filters = body.get_ldapSearch();
	}
	ConfAccessLock.UnlockRead();

	if(!response.set_type(ADMIN_RESP_TYPE_LDAP_RESULTS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!LdapClient.Search(Filters, response.get_ldapResults()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool Entity_RA::RequestCertForEE(const Asn1OctetString & transactionId, const mString & eename, const NewpkiEeRequestCert & cert_request)
{
	NewpkiRequest Request;
	size_t i;
	unsigned long CertReqId;
	const EVP_PKEY * pubkey = NULL;
	X509_PUBKEY * recipient=NULL;

	// Is the type set ?
	if(!cert_request.get_type())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	ConfAccessLock.LockRead();
	// Are we allowed to request certs from this CA ?
	for(i=0; i < myConf.get_conf().get_cas().get_list().size(); i++)
	{
		if(myConf.get_conf().get_cas().get_list()[i].get_name() == cert_request.get_caName())
		{
			pubkey = myConf.get_conf().get_cas().get_list()[i].get_cassl().GetPublicKey();
			if(!pubkey)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockRead();
				return false;
			}
			break;
		}
	}
	// Did we find the CA ?
	if(i == myConf.get_conf().get_cas().get_list().size())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ConfAccessLock.UnlockRead();
		return false;
	}


	// We now verify if the request respects the policy constraints
	if(!ValidateRequestConstraints(ENTITY_TYPE_RA, cert_request.get_type(),
									cert_request.get_request(), 
									myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_minpwdlen(),
									myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_minkeylen(),
									myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_flags()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		return false;
	}
	// Duplicate public key
	if(X509_PUBKEY_set(&recipient, (EVP_PKEY*)pubkey) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		return false;
	}
	ConfAccessLock.UnlockRead();


	// Generate the CA requests
	if(!Request.set_type(NEWPKI_REQUEST_TYPE_CERT))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		X509_PUBKEY_free(recipient);
		return false;
	}
	// Generate the CSR
	if(!lStore.RequestCertificateForEE(transactionId, eename, cert_request, Request.get_certRequest().get_request(), CertReqId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		X509_PUBKEY_free(recipient);
		return false;
	}
	// We set the LDAP UID
	Request.get_certRequest().set_ldapUid("");
	// We set the request id
	Request.get_certRequest().set_id(CertReqId);
	ConfAccessLock.LockRead();
	// We set the validity len
	Request.get_certRequest().set_validity(myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_defaultValidity());
	ConfAccessLock.UnlockRead();
	
	if(!InsertRequest(CertReqId, Request, recipient))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		X509_PUBKEY_free(recipient);
		lStore.DeleteCertReq(CertReqId);
		return false;
	}
	X509_PUBKEY_free(recipient);
	return true;
}

bool Entity_RA::RevokeCertForEE(const Asn1OctetString & transactionId, const mString & eename, unsigned long cert_id)
{
	NewpkiRequest Request;
	int i;
	NewpkiProfileDatas Profile;

	// Make sure the owner profile comes from this EE

	if(!lStore.GetProfileByCertId(cert_id, Profile))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	i = Profile.get_eeId().find(":", 0);
	if(i == -1)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	if( ! (Profile.get_eeId().Right(Profile.get_eeId().size() - i - 1) == eename) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	// Set the associated TID
	if(!lStore.SetCertReqTransactionId(cert_id, transactionId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!Private_RevokeCert(PKI_CERT::EmptyInstance, false, cert_id))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		lStore.SetCertReqTransactionId(cert_id, Asn1OctetString::EmptyInstance);
		return false;
	}
	return true;
}

bool Entity_RA::RequestCert(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_RA_REQUEST_CERT))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	mString ldap_uid;
	NewpkiRequest Request;
	size_t i;
	unsigned long CertReqId;
	const EVP_PKEY * pubkey=NULL;
	X509_PUBKEY * recipient=NULL;

	// Is the type set ?
	if(!body.get_certRequest().get_type())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	ConfAccessLock.LockRead();
	// Are we allowed to request certs from this CA ?
	for(i=0; i < myConf.get_conf().get_cas().get_list().size(); i++)
	{
		if(myConf.get_conf().get_cas().get_list()[i].get_name() == body.get_certRequest().get_caName())
		{
			pubkey = myConf.get_conf().get_cas().get_list()[i].get_cassl().GetPublicKey();
			if(!pubkey)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockRead();
				return false;
			}
			break;
		}
	}
	// Did we find the CA ?
	if(i == myConf.get_conf().get_cas().get_list().size())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ConfAccessLock.UnlockRead();
		return false;
	}


	// We now verify if the request respects the policy constraints
	if(!ValidateRequestConstraints(ENTITY_TYPE_RA, body.get_certRequest().get_type(),
									body.get_certRequest().get_request(), 
									myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_minpwdlen(),
									myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_minkeylen(),
									myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_flags()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		return false;
	}
	// We duplicate the public key
	if(X509_PUBKEY_set(&recipient, (EVP_PKEY*)pubkey) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		return false;
	}
	ConfAccessLock.UnlockRead();


	// Generate the requests
	if(!Request.set_type(NEWPKI_REQUEST_TYPE_CERT))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		X509_PUBKEY_free(recipient);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_REQ_ID))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		X509_PUBKEY_free(recipient);
		return false;
	}

	// Generate the CSR
	if(!lStore.RequestCertificate(UserCert, body.get_certRequest(), Request.get_certRequest().get_request(), ldap_uid, CertReqId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		X509_PUBKEY_free(recipient);
		return false;
	}
	response.set_id(CertReqId);



	// We set the LDAP UID
	Request.get_certRequest().set_ldapUid(ldap_uid);
	// We set the request id
	Request.get_certRequest().set_id(CertReqId);
	// We set the validity len
	Request.get_certRequest().set_validity(body.get_certRequest().get_validity());
	
	
	if(!InsertRequest(CertReqId, Request, recipient))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		lStore.DeleteCertReq(CertReqId);
		X509_PUBKEY_free(recipient);
		return false;
	}	
	X509_PUBKEY_free(recipient);

	return true;
}


bool Entity_RA::Requester_OnNewResponse(const Asn1OctetString & transactionID, const X509_PUBKEY * sender, const NewpkiResponse & Response)
{
	size_t i;
	mString sender_name;
	unsigned long priv_attr;
	Asn1OctetString respTID;
	NewpkiProfileDatasCert cert;
	int status;
	PKI_CRL lastCrl;

	if(!GetPrivAttr(transactionID, priv_attr))
	{
		return false;
	}

	if(Response.get_type() == NEWPKI_RESPONSE_TYPE_EE_VAL || 
		Response.get_type() == NEWPKI_RESPONSE_TYPE_EE_DN_REMOVE ||
		Response.get_type() == NEWPKI_RESPONSE_TYPE_EE_UPDATE_CERT_STATUS ||
		Response.get_type() == NEWPKI_RESPONSE_TYPE_EE_CERT_PUBLISH)
	{
		ConfAccessLock.LockRead();

		// Now search the EE, to make sure it is the real signer !
		for(i=0; i < myConf.get_conf().get_ees().get_list().size(); i++)
		{
			// Are the public keys the same ?
			if(myConf.get_conf().get_ees().get_list()[i].get_eessl() == sender)
			{
				sender_name = myConf.get_conf().get_ees().get_list()[i].get_name();
				break;
			}
			
		}
		// We didn't find the EE
		if(i == myConf.get_conf().get_ees().get_list().size())
		{
			ConfAccessLock.UnlockRead();
			return true;
		}
		ConfAccessLock.UnlockRead();
	}
	else
	{
		if(!lStore.GetCertReq(priv_attr, cert))
		{
			return false;
		}
		
		ConfAccessLock.LockRead();

		// Now search the CA, to make sure it is the real signer !
		for(i=0; i < myConf.get_conf().get_cas().get_list().size(); i++)
		{
			// Is it the CA ?
			if( !(myConf.get_conf().get_cas().get_list()[i].get_name() == cert.get_caName()) )
				continue;

			// Are the public keys the same ?
			if(myConf.get_conf().get_cas().get_list()[i].get_cassl() == sender)
			{
				sender_name = myConf.get_conf().get_cas().get_list()[i].get_name();
				break;
			}	
		}
		cert.Clear();

		// We didn't find the CA
		if(i == myConf.get_conf().get_cas().get_list().size())
		{
			ConfAccessLock.UnlockRead();
			return true;
		}
		ConfAccessLock.UnlockRead();
	}
	if(!lStore.InsertResponse(sender_name, priv_attr, Response))
		return false;

	// If we have a response for a certificate, we need
	// to make sure the certificate is not for an EE
	// it is the case we send the response as it is to the EE
	if(Response.get_type() == NEWPKI_RESPONSE_TYPE_CERT ||
		Response.get_type() == NEWPKI_RESPONSE_TYPE_REV ||
		Response.get_type() == NEWPKI_RESPONSE_TYPE_SUSP ||
		Response.get_type() == NEWPKI_RESPONSE_TYPE_UNSUSP)
	{
		ConfAccessLock.LockRead();
		if(!myConf.get_conf().get_ees().get_list().size())
		{
			ConfAccessLock.UnlockRead();
			return true;
		}
		ConfAccessLock.UnlockRead();

		// Try the get the transaction ID
		if(!lStore.GetCertReqTransactionId(priv_attr, respTID) || !respTID.get_BufferLen())
		{
			if(Response.get_type() == NEWPKI_RESPONSE_TYPE_CERT)
			{
				// I personnaly (the RA) sent the request.
				// If the certificate's owner profile belongs
				// to an EE I need to let the EE know about the
				// new certificate
				// We forward it to the EE
				Forward_CertToEE(priv_attr, Response);
			}
			else
			{
				// I personnaly (the RA) sent the request.
				// If the certificate's owner profile belongs
				// to an EE I need to let the EE know about the
				// change of status of its certificate
				// We forward the new status to the EE
				status = -1;
				if(Response.get_type() == NEWPKI_RESPONSE_TYPE_REV)
				{
					if(Response.get_revResponse().get_status() == REV_RESPONSE_OK)
						status = NEWPKI_PROFILE_CERT_STATE_REVOKED;
					else if(Response.get_revResponse().get_status() == REV_RESPONSE_ERROR)
						status = NewPKIStore::CaStatus2StoreStatus(Response.get_revResponse().get_certStatus(), 
														NEWPKI_PROFILE_CERT_STATE_ACTIVE);

					if(Response.get_revResponse().get_lastCrl())
						lastCrl = Response.get_revResponse().get_lastCrl();
				}
				else if(Response.get_type() == NEWPKI_RESPONSE_TYPE_SUSP)
				{
					if(Response.get_suspResponse().get_status() == SUSP_RESPONSE_OK)
						status = NEWPKI_PROFILE_CERT_STATE_SUSPENDED;
					else if(Response.get_suspResponse().get_status() == SUSP_RESPONSE_ERROR)
						status = NewPKIStore::CaStatus2StoreStatus(Response.get_suspResponse().get_certStatus(), 
														NEWPKI_PROFILE_CERT_STATE_ACTIVE);

					if(Response.get_suspResponse().get_lastCrl())
						lastCrl = Response.get_suspResponse().get_lastCrl();
				}
				else if(Response.get_type() == NEWPKI_RESPONSE_TYPE_UNSUSP)
				{
					if(Response.get_unsuspResponse().get_status() == UNSUSP_RESPONSE_OK)
						status = NEWPKI_PROFILE_CERT_STATE_ACTIVE;
					else if(Response.get_unsuspResponse().get_status() == UNSUSP_RESPONSE_ERROR)
						status = NewPKIStore::CaStatus2StoreStatus(Response.get_unsuspResponse().get_certStatus(), 
														NEWPKI_PROFILE_CERT_STATE_SUSPENDED);

					if(Response.get_unsuspResponse().get_lastCrl())
						lastCrl = Response.get_unsuspResponse().get_lastCrl();
				}
				if(status != -1)
				{
					Forward_CertStatusToEE(priv_attr, status, lastCrl);
				}
			}
			return true;
		}
		else
		{
			// The request was sent on behalf of an EE, we must
			// send it back the response from the CA

			// Erase the transaction ID
			lStore.SetCertReqTransactionId(priv_attr, Asn1OctetString::EmptyInstance);

			if(Response.get_type() == NEWPKI_RESPONSE_TYPE_CERT)
			{
				Forward_CertResponseToEE(priv_attr, Response, respTID);
			}
			else if(Response.get_type() == NEWPKI_RESPONSE_TYPE_REV)
			{
				Forward_RevResponseToEE(priv_attr, Response, respTID);
			}
		}
		return true;
	}

	return true;
}

bool Entity_RA::Responder_ValidateRequest(const NewpkiRequest & Request, const X509_PUBKEY * Requester, mString & SenderName)
{
	ConfAccessLock.LockRead();

	// We need to search that the EE has the right 
	// to access us
	if(myConf.get_conf().get_ees().get_list().size() != 1)
	{
		//This RA is not allowed to access us
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ConfAccessLock.UnlockRead();
		return false;
	}

	//Is it the same public key
	if(!(myConf.get_conf().get_ees().get_list()[0].get_eessl() ==
		Requester))
	{
		//This RA is not allowed to access us
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ConfAccessLock.UnlockRead();
		return false;
	}

	SenderName = myConf.get_conf().get_ees().get_list()[0].get_name();
	ConfAccessLock.UnlockRead();

	// Is it the right type of request ?
	if(Request.get_type() != NEWPKI_REQUEST_TYPE_EE_DN_PUBLISH &&
		Request.get_type() != NEWPKI_REQUEST_TYPE_EE_CERT &&
		Request.get_type() != NEWPKI_REQUEST_TYPE_EE_REV)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}		

	return true;
}

bool Entity_RA::Responder_TreatRequestAsynch(const NewpkiRequest & Request, const Asn1OctetString & transactionId, const mString & SenderName)
{
	mString StrErr;
	mString ObjectName;
	LOG_MESSAGE_TYPE LogType;
	
	StrErr = "";
	ObjectName = _sv("Unknown");

	if(ProceedWithRequest(Request, transactionId, SenderName, LogType, ObjectName, StrErr))
	{
		if(StrErr.size())
			m_Logging->LogMessage(LOG_STATUS_TYPE_FAILURE, LogType, 0, SenderName.c_str(), LOG_NO_OBJECTID, ObjectName.c_str(), StrErr.c_str());
		else
			m_Logging->LogMessage(LOG_STATUS_TYPE_SUCCESS, LogType, 0, SenderName.c_str(), LOG_NO_OBJECTID, ObjectName.c_str());

		return true;
	}
	else
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

}


bool Entity_RA::ProceedWithRequest(const NewpkiRequest & req, 
									const Asn1OctetString & transactionId, 
									const mString & SenderName, 
									LOG_MESSAGE_TYPE & LogType, 
									mString & ObjectName, 
									mString & Err)
{
	char * dn_line;
	long ProfileId;
	unsigned long groupid;
	bool admin_validation;
	NewpkiResponse resp;
	mString eeid;

	//Proceed with the request
	switch(req.get_type())
	{
		case NEWPKI_REQUEST_TYPE_EE_DN_PUBLISH:
			LogType = LOG_MESSAGE_TYPE_EE_DN_PUBLISH;

			dn_line = X509_NAME_oneline((X509_NAME*)req.get_eeDnPublish().get_dn(), NULL, 0);
			if(dn_line)
			{
				if(ObjectName.sprintf("DN: %s", dn_line) <= 0)
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
					ERR_to_mstring(Err);
					return false;
				}
				free(dn_line);
			}
			else
			{
				if(ObjectName.sprintf("EE ID: %ld", req.get_eeDnPublish().get_eeId()) <= 0)
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
					ERR_to_mstring(Err);
					return false;
				}
			}

			ConfAccessLock.LockRead();
			admin_validation = (ASN1_BIT_STRING_get_bit(
								myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_flags(), 
								RA_EE_PROFILES_REQUIRE_VALIDATION) == 1);
			groupid = myConf.get_conf().get_body().get_raConf().RA_CONF_PTR.get_eeProfilesGroup();
			ConfAccessLock.UnlockRead();

			if(!groupid)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
				ERR_add_error_data(1, _sv("You must configure a group to delegate the ownership of the EE profiles"));
				ERR_to_mstring(Err);
				return false;
			}

			if(eeid.sprintf("%ld:%s", req.get_eeDnPublish().get_eeId(), SenderName.c_str()) <= 0)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				ERR_to_mstring(Err);
				return false;
			}

			if(!lStore.InsertProfileFromEE(req.get_eeDnPublish().get_dn(), 
											eeid, 
											transactionId, 
											admin_validation, 
											groupid, 
											ProfileId))
			{
				ERR_to_mstring(Err);
				return false;
			}

			// No admin validation is required we can send the response 
			// right away
			if(!admin_validation)
			{
				if(!resp.set_type(NEWPKI_RESPONSE_TYPE_EE_DN_PUBLISH))
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					ERR_to_mstring(Err);
					return false;
				}

				ConfAccessLock.LockRead();
				if(!InsertResponse(transactionId, resp, 
					myConf.get_conf().get_ees().get_list()[0].get_eessl().GetX509_PUBKEY()))
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					ERR_to_mstring(Err);
					ConfAccessLock.UnlockRead();
					return false;
				}
				ConfAccessLock.UnlockRead();
			}
			break;
		case NEWPKI_REQUEST_TYPE_EE_CERT:
			LogType = LOG_MESSAGE_TYPE_CERT_REQUEST;
			if(ObjectName.sprintf("EE ID: %ld:%s", req.get_eeCertRequest().get_profileId(), SenderName.c_str()) <= 0)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				ERR_to_mstring(Err);
				return false;
			}
			if(!RequestCertForEE(transactionId, SenderName, req.get_eeCertRequest()))
			{
				ERR_to_mstring(Err);
				return false;
			}
			break;
		case NEWPKI_REQUEST_TYPE_EE_REV:
			LogType = LOG_MESSAGE_TYPE_CERT_REQUEST;
			if(ObjectName.sprintf("EE ID: %ld:%s", req.get_eeCertRequest().get_profileId(), SenderName.c_str()) <= 0)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				ERR_to_mstring(Err);
				return false;
			}
			if(!RevokeCertForEE(transactionId, SenderName, req.get_eeCertRevoke().get_certId()))
			{
				ERR_to_mstring(Err);
				return false;
			}
			break;
		default:
			LogType = LOG_MESSAGE_TYPE_EE_DN_PUBLISH;
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			ERR_to_mstring(Err);
			return false;
			break;
	}
	return true;
}

bool Entity_RA::Private_RevokeCert(const PKI_CERT & UserCert, bool CheckOwner, unsigned long id)
{
	mString ttDatas;
	PKI_CSR req;
	PKI_RSA privKey;
	NewpkiRequest Request;
	size_t i;
	HashTable_Dn Dn;
	const EVP_PKEY * pubkey = NULL;
	X509_PUBKEY * recipient=NULL;

	unsigned long cert_serial;
	mString ca_name;
	mString ldap_uid;


	// Mark the cert and get usefull info
	if(!lStore.MarkCertificateForRevocation(UserCert, CheckOwner, id, cert_serial, ca_name, ldap_uid))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockRead();
	// Are we allowed to request certs from this CA ?
	for(i=0; i < myConf.get_conf().get_cas().get_list().size(); i++)
	{
		if(myConf.get_conf().get_cas().get_list()[i].get_name() == ca_name)
		{
			pubkey = myConf.get_conf().get_cas().get_list()[i].get_cassl().GetPublicKey();
			if(!pubkey)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockRead();
				lStore.UnmarkCertificateForRevocation(id);
				return false;
			}
			break;
		}
	}
	// Did we find the CA ?
	if(i == myConf.get_conf().get_cas().get_list().size())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ConfAccessLock.UnlockRead();
		lStore.UnmarkCertificateForRevocation(id);
		return false;
	}
	// Duplicate public key
	if(X509_PUBKEY_set(&recipient, (EVP_PKEY*)pubkey) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		lStore.UnmarkCertificateForRevocation(id);
		return false;
	}
	ConfAccessLock.UnlockRead();
	
	// Generate the request
	if(!Request.set_type(NEWPKI_REQUEST_TYPE_REV))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		lStore.UnmarkCertificateForRevocation(id);
		X509_PUBKEY_free(recipient);
		return false;
	}

	Request.get_revRequest().set_ldapUid(ldap_uid);
	Request.get_revRequest().set_id(id);
	Request.get_revRequest().set_reason(0);
	Request.get_revRequest().set_serial(cert_serial);


	if(!InsertRequest(id, Request, recipient))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		lStore.UnmarkCertificateForRevocation(id);
		X509_PUBKEY_free(recipient);
		return false;
	}
	X509_PUBKEY_free(recipient);
	return true;
}


bool Entity_RA::RevokeCert(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_RA_REVOKE_CERT))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}



	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!Private_RevokeCert(UserCert, true, body.get_serial()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool Entity_RA::UnsuspendCert(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_RA_REVOKE_CERT))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	mString ttDatas;
	PKI_CSR req;
	PKI_RSA privKey;
	NewpkiRequest Request;
	size_t i;
	HashTable_Dn Dn;
	const EVP_PKEY * pubkey = NULL;
	X509_PUBKEY * recipient=NULL;

	unsigned long cert_serial;
	mString ca_name;
	mString ldap_uid;


	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// Suspend the cert and get usefull info
	if(!lStore.MarkCertificateForUnsuspension(UserCert, body.get_serial(), cert_serial, 
											ca_name, ldap_uid))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockRead();
	// Are we allowed to request certs from this CA ?
	for(i=0; i < myConf.get_conf().get_cas().get_list().size(); i++)
	{
		if(myConf.get_conf().get_cas().get_list()[i].get_name() == ca_name)
		{
			pubkey = myConf.get_conf().get_cas().get_list()[i].get_cassl().GetPublicKey();
			if(!pubkey)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockRead();
				lStore.UnmarkCertificateForUnsuspension(body.get_serial());
				return false;
			}
			break;
		}
	}
	// Did we find the CA ?
	if(i == myConf.get_conf().get_cas().get_list().size())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ConfAccessLock.UnlockRead();
		lStore.UnmarkCertificateForUnsuspension(body.get_serial());
		return false;
	}
	// Duplicate public key
	if(X509_PUBKEY_set(&recipient, (EVP_PKEY*)pubkey) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		lStore.UnmarkCertificateForUnsuspension(body.get_serial());
		return false;
	}
	ConfAccessLock.UnlockRead();
	
	// Generate the request
	if(!Request.set_type(NEWPKI_REQUEST_TYPE_UNSUSP))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		lStore.UnmarkCertificateForUnsuspension(body.get_serial());
		X509_PUBKEY_free(recipient);
		return false;
	}

	Request.get_unsuspRequest().set_ldapUid(ldap_uid);
	Request.get_unsuspRequest().set_id(body.get_serial());
	Request.get_unsuspRequest().set_serial(cert_serial);


	if(!InsertRequest(body.get_serial(), Request, recipient))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		lStore.UnmarkCertificateForUnsuspension(body.get_serial());
		X509_PUBKEY_free(recipient);
		return false;
	}
	X509_PUBKEY_free(recipient);

	return true;
}

bool Entity_RA::SuspendCert(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_RA_REVOKE_CERT))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	mString ttDatas;
	PKI_CSR req;
	PKI_RSA privKey;
	NewpkiRequest Request;
	size_t i;
	HashTable_Dn Dn;
	const EVP_PKEY * pubkey = NULL;
	X509_PUBKEY * recipient=NULL;

	unsigned long cert_serial;
	mString ca_name;
	mString ldap_uid;


	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// Suspend the cert and get usefull info
	if(!lStore.MarkCertificateForSuspension(UserCert, body.get_serial(), cert_serial, 
											ca_name, ldap_uid))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockRead();
	// Are we allowed to request certs from this CA ?
	for(i=0; i < myConf.get_conf().get_cas().get_list().size(); i++)
	{
		if(myConf.get_conf().get_cas().get_list()[i].get_name() == ca_name)
		{
			pubkey = myConf.get_conf().get_cas().get_list()[i].get_cassl().GetPublicKey();
			if(!pubkey)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				ConfAccessLock.UnlockRead();
				lStore.UnmarkCertificateForSuspension(body.get_serial());
				return false;
			}
			break;
		}
	}
	// Did we find the CA ?
	if(i == myConf.get_conf().get_cas().get_list().size())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ConfAccessLock.UnlockRead();
		lStore.UnmarkCertificateForSuspension(body.get_serial());
		return false;
	}
	// Duplicate public key
	if(X509_PUBKEY_set(&recipient, (EVP_PKEY*)pubkey) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		lStore.UnmarkCertificateForSuspension(body.get_serial());
		return false;
	}
	ConfAccessLock.UnlockRead();
	
	// Generate the request
	if(!Request.set_type(NEWPKI_REQUEST_TYPE_SUSP))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		lStore.UnmarkCertificateForSuspension(body.get_serial());
		X509_PUBKEY_free(recipient);
		return false;
	}

	Request.get_suspRequest().set_ldapUid(ldap_uid);
	Request.get_suspRequest().set_id(body.get_serial());
	Request.get_suspRequest().set_serial(cert_serial);


	if(!InsertRequest(body.get_serial(), Request, recipient))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		lStore.UnmarkCertificateForSuspension(body.get_serial());
		X509_PUBKEY_free(recipient);
		return false;
	}
	X509_PUBKEY_free(recipient);

	return true;
}

bool Entity_RA::DeletePkcs12(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_RA_REQUEST_CERT))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!lStore.DeletePKCS12(UserCert, body.get_serial()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}


bool Entity_RA::ChangeProfileUID(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_RA_MANAGE_PROFILES))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!lStore.ChangeProfileUID(UserCert, body.get_profileUid().get_profileId(), body.get_profileUid().get_ldapUid()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool Entity_RA::ChangeProfileOwner(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_RA_MANAGE_PROFILES))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_RA_DELEGATE_PROFILE_OWNERSHIP))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!lStore.ChangeProfileOwner(UserCert, body.get_profileOwner().get_profileId(), body.get_profileOwner().get_ownerGroupSerial()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool Entity_RA::ChangeProfileDN(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_RA_MANAGE_PROFILES))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!lStore.ChangeProfileDN(UserCert, body.get_profileDn().get_profileId(), body.get_profileDn().get_dn()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool Entity_RA::AcceptProfile(COMMAND_PARAMETERS)
{
	NewpkiResponse resp;
	Asn1OctetString transactionId;

	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_RA_MANAGE_PROFILES))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// Get the profile to have the ee_tid
	if(!lStore.GetProfileTransactionId(UserCert, body.get_profileId(), transactionId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!lStore.AcceptProfile(UserCert, body.get_profileId()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!resp.set_type(NEWPKI_RESPONSE_TYPE_EE_DN_PUBLISH))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockRead();
	if(!InsertResponse(transactionId, resp, 
		myConf.get_conf().get_ees().get_list()[0].get_eessl().GetX509_PUBKEY()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		return false;
	}
	ConfAccessLock.UnlockRead();

	return true;
}

bool Entity_RA::RejectProfile(COMMAND_PARAMETERS)
{
	NewpkiResponse resp;
	Asn1OctetString transactionId;

	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_RA_MANAGE_PROFILES))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// Get the profile to have the ee_tid
	if(!lStore.GetProfileTransactionId(UserCert, body.get_rejectProfile().get_profileId(), transactionId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!lStore.DeleteProfile(UserCert, body.get_rejectProfile().get_profileId(), true))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!resp.set_type(NEWPKI_RESPONSE_TYPE_EE_DN_PUBLISH))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// Convert the reason given by the admin
	// to an error entry
	ERR_clear_error();
	NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
	ERR_add_error_data(1, body.get_rejectProfile().get_reason().c_str());
	ERR_to_ERROR_ENTRIES(resp.get_eeDnPublishResponse().get_errors());
	ERR_clear_error();


	ConfAccessLock.LockRead();
	if(!InsertResponse(transactionId, resp, 
		myConf.get_conf().get_ees().get_list()[0].get_eessl().GetX509_PUBKEY()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		return false;
	}
	ConfAccessLock.UnlockRead();

	return true;
}

bool Entity_RA::DeleteProfile(COMMAND_PARAMETERS)
{
	NewpkiProfileDatas profile;
	NewpkiRequest Request;
	unsigned long ee_id;
	const X509_PUBKEY * recipient=NULL;
	int pos;

	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_RA_MANAGE_PROFILES))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// Get the profile to have the value of ee_id
	if(!lStore.GetProfile(UserCert, body.get_profileId(), profile))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!lStore.DeleteProfile(UserCert, body.get_profileId(), true))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(profile.get_eeId().size())
	{
		pos = profile.get_eeId().find(":", 0);
		if(pos == -1)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			ERR_add_error_data(1, _sv("The EE Id has an invalid format"));
			return false;
		}
		ee_id = profile.get_eeId().Left(pos).c_ulng();

		// Generate the request
		if(!Request.set_type(NEWPKI_REQUEST_TYPE_EE_DN_REMOVE))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		ConfAccessLock.LockRead();
		if(!myConf.get_conf().get_ees().get_list().size())
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
			ERR_add_error_data(1, _sv("You haven't configured an EE for this RA"));
			ConfAccessLock.UnlockRead();
			return false;
		}
		recipient = myConf.get_conf().get_ees().get_list()[0].get_eessl().GetX509_PUBKEY();
		if(!recipient)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			ConfAccessLock.UnlockRead();
			return false;
		}
		Request.get_eeDnRemove().set_eeId(ee_id);
		if(!InsertRequest(body.get_profileId(), Request, recipient))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			ConfAccessLock.UnlockRead();
			return false;
		}
		ConfAccessLock.UnlockRead();
	}
	return true;
}

bool Entity_RA::DeleteRequest(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_RA_REQUEST_CERT))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!lStore.DeleteRequest(UserCert, body.get_requestId()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}


bool Entity_RA::PrepareConfToWrite()
{
	return true;
}

void Entity_RA::PrintInfo(FILE *out)
{

}

void Entity_RA::WarnCertOwnerOfExpiry(const NewpkiProfileDatasCert & cert, const NewpkiProfileDatas & Profile, int Days)
{
	MailInfo mail;
	mString profileDN;
	mString body;
	char * tmpSubject;

	if(!cert.get_adminMail().size())
		return;

	tmpSubject = X509_NAME_oneline((X509_NAME*)Profile.get_profile().get_dn(), NULL, 0);
	if(!tmpSubject)
	{
		profileDN = _sv("Unknown");
	}
	else
	{
		profileDN = tmpSubject;
		free(tmpSubject);
	}

	mail.set_MailTo(cert.get_adminMail());
	mail.set_SignMail(true);
	mail.set_Subject(_sv("A certificate is about to expiry"));
	if(body.sprintf(_sv("The following certificate will expiry in %d day(s)\n\n"
								"Profile ID: %ld\n"
								"Profile DN: %s\n"
								"Certificate ID: %ld\n"
								"Certificate DN: %s\n"
								"Signed by: %s"),
								Days, 
								Profile.get_id(), 
								profileDN.c_str(),
								cert.get_id(),
								cert.get_cert().GetStringName(),
								cert.get_caName().c_str()) <= 0)
	{
		return;
	}
	mail.set_Body(body);
	m_Jobs.SendMail(m_EntityCert.GetStringName(), mail, true);
}

void Entity_RA::ThreadCheckExpire(const NewpkiThread * Thread, void *param)
{
	Entity_RA * me_this = (Entity_RA*)param;
	mVector<NewpkiProfileDatas> Profiles;
	int index;
	time_t currTime;
	time_t lastTime;
	time_t expTime;
	size_t i, j;

	// We make sure the mailer is up
	while(!Thread->ShouldStop() && !me_this->m_Jobs.MailerIsUp())
	{
		NewpkiThread::Sleep(500);
	}

	time(&lastTime);
	while(!Thread->ShouldStop())
	{
		// We enumerate all the profiles
		index = 0;
		while(me_this->lStore.EnumProfiles(PKI_CERT::EmptyInstance, 
											Profiles, index, 20, 0, "", true) && Profiles.size())
		{
			for(i=0; i<Profiles.size(); i++)
			{
				// We check all the certificates of the current profile
				for(j=0; j<Profiles[i].get_certs().size(); j++)
				{
					time(&currTime);

					// We ignore this certificate if it's not active
					// or if it's already expired
					if(Profiles[i].get_certs()[j].get_state() != 
						NEWPKI_PROFILE_CERT_STATE_ACTIVE || 
						currTime >= 
						Profiles[i].get_certs()[j].get_cert().GetEndDate())
						continue;

					expTime = Profiles[i].get_certs()[j].get_cert().GetEndDate() - currTime;


					// If certificate expires in 10 days when warn the owner(s)
					if(expTime >= 864000 && 
						expTime < 950400)
					{
						me_this->WarnCertOwnerOfExpiry(Profiles[i].get_certs()[j], Profiles[i], 10);
					}
					else if(expTime >= 432000 && 
						expTime < 518400)
					{
						me_this->WarnCertOwnerOfExpiry(Profiles[i].get_certs()[j], Profiles[i], 5);
					}
					else if(expTime >= 86400 && 
						expTime < 172800)
					{
						me_this->WarnCertOwnerOfExpiry(Profiles[i].get_certs()[j], Profiles[i], 1);
					}
				}
			}

			Profiles.clear();
			index += 20;
		}

		// We wait 24 hours before testing all the certificates
		if(!Thread->SleepInterrupt(86400))
			break;
		time(&lastTime);
	}
}

void Entity_RA::Forward_CertResponseToEE(unsigned long CertReqId, const NewpkiResponse &Response, const Asn1OctetString &respTID)
{
	NewpkiResponse resp;
	NewpkiProfileDatasCert cert;

	if(!resp.set_type(NEWPKI_RESPONSE_TYPE_EE_CERT))
		return;

	resp.get_eeCertResponse().set_status(Response.get_certResponse().get_status());
	resp.get_eeCertResponse().set_raId(Response.get_certResponse().get_id());

	if(Response.get_certResponse().get_lastCrl())
		resp.get_eeCertResponse().set_lastCrl(Response.get_certResponse().get_lastCrl());

	if(Response.get_certResponse().get_status() == CERT_RESPONSE_OK)
	{
		// Try the get the certificate
		if(!lStore.GetCertReq(CertReqId, cert))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			ERR_to_ERROR_ENTRIES(resp.get_eeCertResponse().get_errors());
		}
		else if(!cert.get_cert())
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
			ERR_to_ERROR_ENTRIES(resp.get_eeCertResponse().get_errors());
		}
		else if(!resp.get_eeCertResponse().set_certificate(cert.get_cert()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			ERR_to_ERROR_ENTRIES(resp.get_eeCertResponse().get_errors());
		}
		else if(cert.get_p12() && !resp.get_eeCertResponse().set_p12(cert.get_p12()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			ERR_to_ERROR_ENTRIES(resp.get_eeCertResponse().get_errors());
		}
		else if(cert.get_p7b() && !resp.get_eeCertResponse().set_p7b(cert.get_p7b()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			ERR_to_ERROR_ENTRIES(resp.get_eeCertResponse().get_errors());
		}
	}
	else
	{
		resp.get_eeCertResponse().set_errors(Response.get_certResponse().get_errors());
	}

	ConfAccessLock.LockRead();
	if(!InsertResponse(respTID, resp, 
		myConf.get_conf().get_ees().get_list()[0].get_eessl().GetX509_PUBKEY()))
	{
		ConfAccessLock.UnlockRead();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return;
	}
	ConfAccessLock.UnlockRead();
}

void Entity_RA::Forward_RevResponseToEE(unsigned long CertReqId, const NewpkiResponse &Response, const Asn1OctetString &respTID)
{
	NewpkiResponse resp;

	if(!resp.set_type(NEWPKI_RESPONSE_TYPE_EE_REV))
		return;

	resp.get_eeRevResponse().set_raId(CertReqId);
	resp.get_eeRevResponse().set_status(Response.get_revResponse().get_status());
	resp.get_eeRevResponse().set_certStatus(Response.get_revResponse().get_certStatus());
	resp.get_eeRevResponse().set_errors(Response.get_revResponse().get_errors());
	if(Response.get_revResponse().get_lastCrl())
		resp.get_eeRevResponse().set_lastCrl(Response.get_revResponse().get_lastCrl());

	ConfAccessLock.LockRead();
	if(!InsertResponse(respTID, resp, 
		myConf.get_conf().get_ees().get_list()[0].get_eessl().GetX509_PUBKEY()))
	{
		ConfAccessLock.UnlockRead();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return;
	}
	ConfAccessLock.UnlockRead();
}

void Entity_RA::Forward_CertStatusToEE(unsigned long CertReqId, int Status, const PKI_CRL & lastCrl)
{
	const X509_PUBKEY * eekey;
	NewpkiRequest Request;
	unsigned long ee_id;
	
	if(!Request.set_type(NEWPKI_REQUEST_TYPE_EE_UPDATE_CERT_STATUS))
		return;

	Request.get_eeUpdCertStatus().set_raId(CertReqId);
	Request.get_eeUpdCertStatus().set_status(Status);
	if(lastCrl)
		Request.get_eeUpdCertStatus().set_lastCrl(lastCrl);
	
	ConfAccessLock.LockRead();
	eekey = GetOwnerProfileEeKey(CertReqId, ee_id);
	if(!eekey)
	{
		ConfAccessLock.UnlockRead();
		return;
	}
	if(!InsertRequest(0, Request, eekey))
	{
		ConfAccessLock.UnlockRead();
		return;
	}
	ConfAccessLock.UnlockRead();
}

const X509_PUBKEY* Entity_RA::GetOwnerProfileEeKey(unsigned long CertReqId, unsigned long & ee_id)
{
	mString eename;
	NewpkiProfileDatas Profile;
	size_t i;
	int pos;

	if(!lStore.GetProfileByCertId(CertReqId, Profile))
	{
		return NULL;
	}
	pos = Profile.get_eeId().find(":", 0);
	if(pos == -1)
	{
		return  NULL;
	}
	eename = Profile.get_eeId().Right(Profile.get_eeId().size() - pos - 1);
	if(!eename.size())
	{
		return  NULL;
	}
	ee_id = Profile.get_eeId().Left(pos).c_ulng();
	if(!ee_id)
	{
		return  NULL;
	}
	// Now find the EE
	for(i=0; i<myConf.get_conf().get_ees().get_list().size(); i++)
	{
		if(myConf.get_conf().get_ees().get_list()[i].get_name() == 
			eename)
			return myConf.get_conf().get_ees().get_list()[i].get_eessl().GetX509_PUBKEY();
	}
	return NULL;
}

void Entity_RA::Forward_CertToEE(unsigned long CertReqId, const NewpkiResponse &Response)
{
	NewpkiRequest Request;
	NewpkiProfileDatasCert cert;
	const X509_PUBKEY * eekey;
	unsigned long ee_id;

	if(Response.get_certResponse().get_status() != CERT_RESPONSE_OK)
		return;

	if(!Request.set_type(NEWPKI_REQUEST_TYPE_EE_CERT_PUBLISH))
		return;

	// Try the get the certificate
	if(!lStore.GetCertReq(CertReqId, cert) || !cert.get_cert())
	{
		return;
	}
	if(cert.get_state() != NEWPKI_PROFILE_CERT_STATE_ACTIVE)
		return;

	if(!Request.get_eeCertPublish().set_certificate(cert.get_cert()))
	{
		return;
	}
	Request.get_eeCertPublish().set_raId(CertReqId);
	Request.get_eeCertPublish().set_type(cert.get_type());
	Request.get_eeCertPublish().set_caName(cert.get_caName());
	if(cert.get_p12() && !Request.get_eeCertPublish().set_p12(cert.get_p12()))
	{
		return;
	}
	if(cert.get_p7b() && !Request.get_eeCertPublish().set_p7b(cert.get_p7b()))
	{
		return;
	}
	if(Response.get_certResponse().get_lastCrl())
		Request.get_eeCertPublish().set_lastCrl(Response.get_certResponse().get_lastCrl());


	ConfAccessLock.LockRead();
	eekey = GetOwnerProfileEeKey(CertReqId, ee_id);
	if(!eekey)
	{
		ConfAccessLock.UnlockRead();
		return;
	}
	Request.get_eeCertPublish().set_profileId(ee_id);
	if(!InsertRequest(0, Request, eekey))
	{
		ConfAccessLock.UnlockRead();
		return;
	}
	ConfAccessLock.UnlockRead();
}

