/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// EntityLog.h: interface for the EntityLog class.
//
//////////////////////////////////////////////////////////////////////

#ifndef ENTITYLOG_H
#define ENTITYLOG_H





#include <ASN1/Asn1Resp.h>
#include "SQL/SQL_Conn.h"
#include "SQL/SQL.h"
#include <PKI_CERT.h>
#include <ASN1/Asn1Logs.h>
#include <ASN1/Asn1Req.h>
#include <ReadersWriter.h>

#include <mString.h>
#include <HashCorrelation.h>

typedef void (PROC_ON_AUDIT_CALLBACK) (void * Param, LOG_MESSAGE_STATUS Status, LOG_MESSAGE_TYPE Message, const mString & User, const mString & Object, time_t log_timegmt, const mString & Error);


/*!
  This class allows to log actions
 */
class EntityLog  
{
public:
	/*! \brief This is the constructor.
	 *  \param DbConn [IN] A valid database connection.
	 *  \param EntityCert [IN] A pointer to the entity certificate, it will be used to sign the logs.
	 *  \exception ExceptionNewPKI An error occured.
	 */
	EntityLog(const SQL_Connection * DbConn, const PKI_CERT * EntityCert);

	/*! \brief This is the destructor.
	 */
	~EntityLog();

	/*! \brief This function logs a request.
	 *  \param Status [IN] The status of the message.
	 *  \param Message [IN] The type of object concerned by the log.
	 *  \param UserId [IN] The UserId performing the action.
	 *  \param UserDN [IN] The User's DN performing the action.
	 *  \param ObjectId [IN] The Id of the object.
	 *  \param ObjectName [IN] The name of the object.
	 *  \param Error [IN] An error mString.
	 */
	void LogMessage(LOG_MESSAGE_STATUS Status, LOG_MESSAGE_TYPE Message, unsigned long UserId, const char * UserDN, unsigned long ObjectId=LOG_NO_OBJECTID, const char * ObjectName = NULL, const char * Error = NULL) const;

	/*! \brief This function returns the logs.
	 *  \param Logs [OUT] The list of logs.
	 *  \param Filters [IN] The log filters.
	 *  \return true on success, false on failure.
	 */
	bool GetLogs(mVector< LogEntry > & Logs, const AdminReqEnumLogs & Filters) const;

	/*! \brief This function returns the number of logs.
	 *  \param Count [OUT] The number of logs.
	 *  \param Filters [IN] The log filters.
	 *  \return true on success, false on failure.
	 */
	bool GetLogsCount(unsigned long & Count, const AdminReqEnumLogs & Filters) const;

	/*! \brief This function checks the logs integrity.
	 *  \return true on success, false on failure.
	 */
	bool CheckLogsIntegrity() const;
	
	/*! \brief This function set the audits.
	 *  \param Audit [IN] The audits.
	 *  \return true on success, false on failure.
	 */
	bool SetAudit(const mVector<EntityAuditEntry> & Audit);
	
	/*! \brief This function set the audit callback.
	 *  \param OnAudit [IN] The audit callback.
	 *  \param Param [IN] A value to be passed back to the callback.
	 */
	void SetAuditCalllBack(PROC_ON_AUDIT_CALLBACK * OnAudit, void * Param);

private:
	bool SqlToLOG_ENTRY(SQL &sql, long i, LogEntry & Value) const;
	HashCorrelation LogsHash;
	CriticalSection HashLock;
	PKI_CERT m_EntityCert;
	#define LOG_INSERT			"INSERT INTO logs (log_status, log_type, user, object_name, error, log_date, signature) VALUES (%d, %d, '%s', '%s', '%s', %ld, '%s');"
	#define LOG_SELECT			"SELECT * FROM logs %s ORDER BY log_date DESC LIMIT %d,%d;"
	#define LOG_SELECT_COUNT	"SELECT COUNT(log_id) as ctr FROM logs %s;"
	#define LOG_SELECT_HASH		"SELECT hash FROM logs_hash WHERE hash_id=1;"
	#define LOG_INSERT_HASH		"REPLACE INTO logs_hash (hash_id, hash) VALUES (1, '%s');"
	#define LOG_DELETE			"DELETE FROM logs log_id=%ld;"
	#define LOG_GET_LOGS_LIST	"SELECT * FROM logs LIMIT %d,%d;"

	SQL_Connection * DB_Conn;
	mString FormatFiltersSQL(const AdminReqEnumLogs & Filters) const;
	PROC_ON_AUDIT_CALLBACK * m_OnAudit;
	void * m_Param;
	mVector<EntityAuditEntry> m_Audit;
	ReadersWriter AuditLock;
};

#endif

