/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/

// mString.h: interface for the mString class.
//
//////////////////////////////////////////////////////////////////////

#ifndef MSTRING_H
#define MSTRING_H

#include <openssl/asn1.h>

#include <string>
#include <vector>
using namespace std;
#include <ctype.h>

/*!
	This class is a buffer.
*/
class mBuffer
{
public:
	/*! \brief This is the constructor.
	 */
	mBuffer();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	mBuffer(const mBuffer & other);
	
	/*! \brief This is the destructor.
	 */
	virtual ~mBuffer();
	
	/*! \brief This function copies datas.
	 *  \param buffer [IN] the datas.
	 *  \param bufferlen [IN] the length of buffer.
	 *  \return true on success, false on failure.
	 */
	bool Copy(const unsigned char * buffer, size_t bufferlen);
	
	/*! \brief This function assigns datas.
	 *  \param buffer [IN] the datas.
	 *  \param bufferlen [IN] the length of buffer.
	 *  \return true on success, false on failure.
	 */
	void Assign(const unsigned char * buffer, size_t bufferlen);
	
	/*! \brief This function assigns datas.
	 *  \param buffer [IN] the datas.
	 *  \param bufferlen [IN] the length of buffer.
	 *  \return true on success, false on failure.
	 */
	const unsigned char * get_Buffer() const;
	
	/*! \brief This function returns the datas.
	 *  \return The datas.
	 */
	unsigned char * get_Buffer();
	
	/*! \brief This function returns the length of datas.
	 *  \return The length of datas.
	 */
	size_t get_BufferLen() const;
	
	/*! \brief This function clears the datas.
	 */
	void Clear();
	
	/*! \brief This function returns the status.
	 *  \return true when datas are set, false if not.
	 */
	bool isOK() const;
	
	/*! \brief This operator returns the status.
	 *  \return 1 when datas are set, 0 if not.
	 */
	operator int() const;

	/*! \brief This operator copies of mBuffer to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const mBuffer & other);	
private:
	unsigned char * m_buffer;
	size_t m_bufferlen;
};

/*!
	This class represents a string.
*/
class mString
{
public:
	/*! \brief This is the constructor.
	 */
	mString();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	mString( const mString & other );

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	mString( const string & other );

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	mString( const char * other );

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	mString( const unsigned char * other );

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	mString( const ASN1_STRING * other );

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	mString( const int other );

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	mString( const unsigned int other );

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	mString( const long other );

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	mString( const unsigned long other );
	
	/*! \brief This is the destructor.
	 */
	virtual ~mString();

	/*! \brief This encodes a string.
	 *  \param from [IN] the name of the encoding to convert from.
	 *  \param to [IN] the name of the encoding to convert to.
	 *  \param src [IN] the string to convert from.
	 *  \param dst [OUT] the string to convert to.
	 */
	static void Encode(const char * from, const char * to, const mString & src, mString & dst);

	/*! \brief This function trims the string of the right side.
	 *  \param filters [IN] the characters to trim.
	 */
	void TrimRight(const char * filters);
	
	/*! \brief This function trims the string of the left side.
	 *  \param filters [IN] the characters to trim.
	 */
	void TrimLeft(const char * filters);
	
	/*! \brief This function returns n chars on left of the string.
	 *  \param count [IN] how many chars to return.
	 *  \return The new string.
	 */
	mString Left(int count) const;
	
	/*! \brief This function returns n chars of the string.
	 *  \param start [IN] the start.
	 *  \param count [IN] how many chars to return.
	 *  \return The new string.
	 */
	mString Mid(int start, int count) const;
	
	/*! \brief This function returns n chars on the right side of the string.
	 *  \param count [IN] how many chars to return.
	 *  \return The new string.
	 */
	mString Right(int count) const;
	
	/*! \brief This function splits the string.
	 *  \param sep [IN] the separator(s).
	 *  \param results [OUT] the splitted strings.
	 */
	void Split(const char * sep, vector<mString> & results) const;
	
	/*! \brief This function reserves n chars.
	 *  \param t [IN] the number of chars to reserve.
	 */
	void reserve(size_t t);

	/*! \brief This function loads the string from a DER, it generates PEM.
	 *  \param indatas [IN] the DER datas.
	 *  \param indataslen [IN] the length of indatas.
	 *  \return true on success, false on failure.
	 */
	bool FromDER(const unsigned char * indatas, const int indataslen);

	/*! \brief This function converts the string to DER.
	 *  \param outdatas [OUT] the DER datas (outdatas must be freed).
	 *  \param outdataslen [OUT] the length of outdatas.
	 *  \return true on success, false on failure.
	 */
	bool ToDER(unsigned char ** outdatas, int * outdataslen) const;
	
	/*! \brief This function converts the string to DER.
	 *  \param outdatas [OUT] the DER datas.
	 *  \return true on success, false on failure.
	 */
	bool ToDER(mBuffer & outdatas) const;

	/*! \brief This function writes formatted data to the string.
	 *  \param format [IN] the format-control string.
	 *  \param ... [IN] the list of parameters.
	 *  \return the number of bytes stored int the string.
	 */
	int sprintf(const char *format, ...);

	/*! \brief This function writes formatted data to the string.
	 *  \param format [IN] the format-control string.
	 *  \param args [IN] the list of parameters.
	 *  \return the number of bytes stored int the string.
	 */
	int vsprintf(const char *format, va_list args);

	/*! \brief This function returns the position of a substring.
	 *  \param str [IN] the substring to search.
	 *  \param pos [IN] the position to start the search from.
	 *  \return the position of the substring.
	 */
	int find(const char *str, int pos) const;

	/*! \brief This function replaces all occurences of substring in the string.
	 *  \param pos [IN] the position to start the search from.
	 *  \param num [IN] the number of occurences to replace.
	 *  \param datas [IN] the num string.
	 */
	void replace(int pos, int num, const char *datas);
	
	/*! \brief This function replaces all occurences of substring in the string.
	 *  \param from [IN] the string to search.
	 *  \param to [IN] the string to replace.
	 */
	void replaceall(const mString & from, const mString & to);

	/*! \brief This function ensures that size() henceforth returns size.
	 *  \param size [IN] the size.
	 */
	void resize(unsigned int size);
	
	/*! \brief This function returns the size of the buffer.
	 */
	unsigned int capacity() const;
	
	/*! \brief This function returns the size of the string.
	 */
	unsigned int size() const;
	
	/*! \brief This function compares the string on limited number of chars.
	 *  \param cmp [IN] how many chars to return.
	 *  \return true when strings are same, false when different.
	 */
	bool left_compare(const char * cmp) const;
	
	/*! \brief This function compares the string.
	 *  \param other [IN] the string to compare to.
	 *  \return same as strcmp.
	 */
	int compare(const char *other) const;
	
	/*! \brief This function compares the string.
	 *  \param other [IN] the string to compare to.
	 *  \return same as strcmp.
	 */
	int compare(unsigned char *other) const;
	
	/*! \brief This function compares the string.
	 *  \param other [IN] the string to compare to.
	 *  \return same as strcmp.
	 */
	int compare(mString & other) const;
	
	/*! \brief This function compares the string.
	 *  \param other [IN] the string to compare to.
	 *  \return same as strcmp.
	 */
	int compare(string & other) const;
	
	/*! \brief This function compares a portion of the string.
	 *  \param from [IN] the index to start comparaison to.
	 *  \param max [IN] the number of chars to compare to.
	 *  \param other [IN] the string to compare to.
	 *  \return same as strcmp.
	 */
	int compare(int from, int max, char *other) const;
	
	/*! \brief This function compares a portion of the string.
	 *  \param from [IN] the index to start comparaison to.
	 *  \param max [IN] the number of chars to compare to.
	 *  \param other [IN] the string to compare to.
	 *  \return same as strcmp.
	 */
	int compare(int from, int max, unsigned char *other) const;
	
	/*! \brief This function compares a portion of the string.
	 *  \param from [IN] the index to start comparaison to.
	 *  \param max [IN] the number of chars to compare to.
	 *  \param other [IN] the string to compare to.
	 *  \return same as strcmp.
	 */
	int compare(int from, int max, mString & other) const;
	
	/*! \brief This function compares a portion of the string.
	 *  \param from [IN] the index to start comparaison to.
	 *  \param max [IN] the number of chars to compare to.
	 *  \param other [IN] the string to compare to.
	 *  \return same as strcmp.
	 */
	int compare(int from, int max, string & other) const;
	
	/*! \brief This operator compares the string.
	 *  \param rhs [IN] the string to compare to.
	 *  \return true is the string if < to rhs.
	 */
	bool operator<(const mString& rhs) const;

	/*! \brief This operator returns a single char.
	 *  \param pos [IN] the index of the char to return.
	 *  \return a reference to the char.
	 */
	char & operator[](unsigned int pos);
	
	/*! \brief This operator returns a single char.
	 *  \param pos [IN] the index of the char to return.
	 *  \return the char.
	 */
	const char operator[](unsigned int pos) const;

	/*! \brief This operator returns the string.
	 *  \return the string.
	 */
	const char * c_str() const;

	/*! \brief This operator returns the int representation of the string.
	 *  \return the int representation of the string.
	 */
	int c_int() const;

	/*! \brief This operator returns the unsigned int representation of the string.
	 *  \return the unsigned int representation of the string.
	 */
	unsigned int c_uint() const;

	/*! \brief This operator returns the long representation of the string.
	 *  \return the long representation of the string.
	 */
	long c_lng() const;

	/*! \brief This operator returns the unsigned long representation of the string.
	 *  \return the unsigned long representation of the string.
	 */
	unsigned long c_ulng() const;

	/*! \brief This operator returns the string.
	 *  \return the string.
	 */
	const string & c_string() const;

	/*! \brief This operator returns the string.
	 *  \return the string.
	 */
	string & c_string();

	/*! \brief This function converts the string to an ASN1_INTEGER.
	 *  \param dest [OUT] the ASN1_INTEGER.
	 *  \return true on success, false on failure.
	 */
	bool c_ASN1_INTEGER(ASN1_INTEGER ** dest) const;

	/*! \brief This function converts the string to an ASN1_UTF8STRING.
	 *  \param dest [OUT] the ASN1_UTF8STRING.
	 *  \return true on success, false on failure.
	 */
	bool c_ASN1_UTF8STRING(ASN1_UTF8STRING ** dest) const;

	/*! \brief This function assigns the string.
	 *  \param other [IN] the string to assign.
	 *  \return the new string.
	 */
	mString & operator=( const mString & other );

	/*! \brief This function assigns the string.
	 *  \param other [IN] the string to assign.
	 *  \return the new string.
	 */
	mString & operator=( const string & other );

	/*! \brief This function assigns the string.
	 *  \param other [IN] the string to assign.
	 *  \return the new string.
	 */
	mString & operator=( const char * other );

	/*! \brief This function assigns the string.
	 *  \param other [IN] the string to assign.
	 *  \return the new string.
	 */
	mString & operator=( const unsigned char * other );

	/*! \brief This function assigns the string.
	 *  \param other [IN] the string to assign.
	 *  \return the new string.
	 */
	mString & operator=( const ASN1_STRING * other );

	/*! \brief This function assigns the string.
	 *  \param other [IN] the string to assign.
	 *  \return the new string.
	 */
	mString & operator=( const int other );

	/*! \brief This function assigns the string.
	 *  \param other [IN] the string to assign.
	 *  \return the new string.
	 */
	mString & operator=( const unsigned int other );

	/*! \brief This function assigns the string.
	 *  \param other [IN] the string to assign.
	 *  \return the new string.
	 */
	mString & operator=( const long other );

	/*! \brief This function assigns the string.
	 *  \param other [IN] the string to assign.
	 *  \return the new string.
	 */
	mString & operator=( const unsigned long other );


	/*! \brief This function concatenates a string.
	 *  \param other [IN] the string to concatenate.
	 *  \return the new string.
	 */
	mString & operator+=( const mString & other );

	/*! \brief This function concatenates a string.
	 *  \param other [IN] the string to concatenate.
	 *  \return the new string.
	 */
	mString & operator+=( const string & other );

	/*! \brief This function concatenates a string.
	 *  \param other [IN] the string to concatenate.
	 *  \return the new string.
	 */
	mString & operator+=( const char * other );

	/*! \brief This function concatenates a string.
	 *  \param other [IN] the string to concatenate.
	 *  \return the new string.
	 */
	mString & operator+=( const unsigned char * other );

	/*! \brief This function concatenates a string.
	 *  \param other [IN] the string to concatenate.
	 *  \return the new string.
	 */
	mString & operator+=( const ASN1_STRING * other );

	/*! \brief This function concatenates a string.
	 *  \param other [IN] the string to concatenate.
	 *  \return the new string.
	 */
	mString & operator+=( const int other );

	/*! \brief This function concatenates a string.
	 *  \param other [IN] the string to concatenate.
	 *  \return the new string.
	 */
	mString & operator+=( const unsigned int other );

	/*! \brief This function concatenates a string.
	 *  \param other [IN] the string to concatenate.
	 *  \return the new string.
	 */
	mString & operator+=( const long other );

	/*! \brief This function concatenates a string.
	 *  \param other [IN] the string to concatenate.
	 *  \return the new string.
	 */
	mString & operator+=( const unsigned long other );


	/*! \brief This function compares a string.
	 *  \param other [IN] the string to compare to.
	 *  \return true when strings are same, false when different.
	 */
	bool operator==( const mString & other ) const;

	/*! \brief This function compares a string.
	 *  \param other [IN] the string to compare to.
	 *  \return true when strings are same, false when different.
	 */
	bool operator==( const string & other ) const;

	/*! \brief This function compares a string.
	 *  \param other [IN] the string to compare to.
	 *  \return true when strings are same, false when different.
	 */
	bool operator==( const unsigned char * other ) const;

	/*! \brief This function compares a string.
	 *  \param other [IN] the string to compare to.
	 *  \return true when strings are same, false when different.
	 */
	bool operator==( const char * other ) const;

	/*! \brief This function compares a string.
	 *  \param other [IN] the string to compare to.
	 *  \return true when strings are same, false when different.
	 */
	bool operator==( const ASN1_STRING * other ) const;

	/*! \brief This function compares a string.
	 *  \param other [IN] the string to compare to.
	 *  \return true when strings are same, false when different.
	 */
	bool operator==( const int other ) const;

	/*! \brief This function compares a string.
	 *  \param other [IN] the string to compare to.
	 *  \return true when strings are same, false when different.
	 */
	bool operator==( const unsigned int other ) const;

	/*! \brief This function compares a string.
	 *  \param other [IN] the string to compare to.
	 *  \return true when strings are same, false when different.
	 */
	bool operator==( const long other ) const;

	/*! \brief This function compares a string.
	 *  \param other [IN] the string to compare to.
	 *  \return true when strings are same, false when different.
	 */
	bool operator==( const unsigned long other ) const;

	/*! \brief This function compares a string.
	 *  \param other [IN] the string to compare to.
	 *  \return true when strings are same, false when different.
	 */
	bool operator!=( const mString & other ) const;

	/*! \brief This function compares a string.
	 *  \param other [IN] the string to compare to.
	 *  \return true when strings are same, false when different.
	 */
	bool operator!=( const string & other ) const;

	/*! \brief This function compares a string.
	 *  \param other [IN] the string to compare to.
	 *  \return true when strings are same, false when different.
	 */
	bool operator!=( const unsigned char * other ) const;

	/*! \brief This function compares a string.
	 *  \param other [IN] the string to compare to.
	 *  \return true when strings are same, false when different.
	 */
	bool operator!=( const char * other ) const;

	/*! \brief This function compares a string.
	 *  \param other [IN] the string to compare to.
	 *  \return true when strings are same, false when different.
	 */
	bool operator!=( const int other ) const;

	/*! \brief This function compares a string.
	 *  \param other [IN] the string to compare to.
	 *  \return true when strings are same, false when different.
	 */
	bool operator!=( const unsigned int other ) const;

	/*! \brief This function compares a string.
	 *  \param other [IN] the string to compare to.
	 *  \return true when strings are same, false when different.
	 */
	bool operator!=( const long other ) const;

	/*! \brief This function compares a string.
	 *  \param other [IN] the string to compare to.
	 *  \return true when strings are same, false when different.
	 */
	bool operator!=( const unsigned long other ) const;


	/*! \brief This member is the empty instance.
	 */
	static mString EmptyInstance;
private:
	/* format read states */
	#define DP_S_DEFAULT    0
	#define DP_S_FLAGS      1
	#define DP_S_MIN        2
	#define DP_S_DOT        3
	#define DP_S_MAX        4
	#define DP_S_MOD        5
	#define DP_S_CONV       6
	#define DP_S_DONE       7

	/* format flags - Bits */
	#define DP_F_MINUS      (1 << 0)
	#define DP_F_PLUS       (1 << 1)
	#define DP_F_SPACE      (1 << 2)
	#define DP_F_NUM        (1 << 3)
	#define DP_F_ZERO       (1 << 4)
	#define DP_F_UP         (1 << 5)
	#define DP_F_UNSIGNED   (1 << 6)

	/* conversion flags */
	#define DP_C_SHORT      1
	#define DP_C_LONG       2
	#define DP_C_LDOUBLE    3
	#define DP_C_LLONG      4

	/* some handy macros */
	#define char_to_int(p) (p - '0')
	#define OSSL_MAX(p,q) ((p >= q) ? p : q)

	void fmtstr(const char *, int, int, int);
	void fmtint(long, int, int, int, int);
	void fmtfp(double, int, int, int);
	double abs_val(double value);
	double pow10(int exp);
	long roundv(double value);

	char buffer_lng[sizeof(long) * 5];
	string buffer;
	void Reset();
};

#endif
