/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#include "Asn1Err.h"
#include <PKI_ERR.h>
#include <openssl/asn1t.h>

ASN1_SEQUENCE(ERROR_ENTRY) = {
	ASN1_SIMPLE(ERROR_ENTRY, line, ASN1_INTEGER),
	ASN1_SIMPLE(ERROR_ENTRY, file, ASN1_UTF8STRING),
	ASN1_SIMPLE(ERROR_ENTRY, lib, ASN1_INTEGER),
	ASN1_SIMPLE(ERROR_ENTRY, function, ASN1_INTEGER),
	ASN1_SIMPLE(ERROR_ENTRY, code, ASN1_INTEGER),
	ASN1_SIMPLE(ERROR_ENTRY, data, ASN1_UTF8STRING),
}ASN1_SEQUENCE_END(ERROR_ENTRY)
ErrorEntry ErrorEntry::EmptyInstance;
bool ErrorEntry::set_code(unsigned long c_code)
{
	m_code = c_code;
	return true;
}

unsigned long ErrorEntry::get_code() const
{
	return m_code;
}

bool ErrorEntry::set_data(const mString & c_data)
{
	m_data = c_data;
	return true;
}

const mString & ErrorEntry::get_data() const
{
	return m_data;
}

mString & ErrorEntry::get_data()
{
	return m_data;
}

bool ErrorEntry::set_file(const mString & c_file)
{
	m_file = c_file;
	return true;
}

const mString & ErrorEntry::get_file() const
{
	return m_file;
}

mString & ErrorEntry::get_file()
{
	return m_file;
}

bool ErrorEntry::set_function(unsigned long c_function)
{
	m_function = c_function;
	return true;
}

unsigned long ErrorEntry::get_function() const
{
	return m_function;
}

bool ErrorEntry::set_lib(unsigned long c_lib)
{
	m_lib = c_lib;
	return true;
}

unsigned long ErrorEntry::get_lib() const
{
	return m_lib;
}

bool ErrorEntry::set_line(unsigned long c_line)
{
	m_line = c_line;
	return true;
}

unsigned long ErrorEntry::get_line() const
{
	return m_line;
}

ErrorEntry::ErrorEntry():NewPKIObject()
{
	resetAll();
}

ErrorEntry::ErrorEntry(const ErrorEntry & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

ErrorEntry::~ErrorEntry()
{
	Clear();
}

void ErrorEntry::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void ErrorEntry::freeAll()
{
}

void ErrorEntry::resetAll()
{
	m_code = 0;
	m_data = "";
	m_file = "";
	m_function = 0;
	m_lib = 0;
	m_line = 0;
}

bool ErrorEntry::load_Datas(const ERROR_ENTRY * Datas)
{
	Clear();
	if(Datas->code)
	{
		m_code = ASN1_INTEGER_GET(Datas->code);
	}
	if(Datas->data)
	{
		m_data = Datas->data;
	}
	if(Datas->file)
	{
		m_file = Datas->file;
	}
	if(Datas->function)
	{
		m_function = ASN1_INTEGER_GET(Datas->function);
	}
	if(Datas->lib)
	{
		m_lib = ASN1_INTEGER_GET(Datas->lib);
	}
	if(Datas->line)
	{
		m_line = ASN1_INTEGER_GET(Datas->line);
	}
	m_isOk=true;
	return true;
}

bool ErrorEntry::give_Datas(ERROR_ENTRY ** Datas) const
{
	if(!(*Datas) && !(*Datas = (ERROR_ENTRY*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->code && !((*Datas)->code = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->code, m_code) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->code);
		(*Datas)->code = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!(*Datas)->data && !((*Datas)->data = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_data.c_ASN1_UTF8STRING(&(*Datas)->data))
	{
		ASN1_UTF8STRING_free((*Datas)->data);
		(*Datas)->data = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->file && !((*Datas)->file = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_file.c_ASN1_UTF8STRING(&(*Datas)->file))
	{
		ASN1_UTF8STRING_free((*Datas)->file);
		(*Datas)->file = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->function && !((*Datas)->function = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->function, m_function) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->function);
		(*Datas)->function = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!(*Datas)->lib && !((*Datas)->lib = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->lib, m_lib) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->lib);
		(*Datas)->lib = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!(*Datas)->line && !((*Datas)->line = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->line, m_line) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->line);
		(*Datas)->line = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	return true;
}

bool ErrorEntry::operator=(const ErrorEntry & other)
{
	Clear();
	m_code = other.m_code;
	m_data = other.m_data;
	m_file = other.m_file;
	m_function = other.m_function;
	m_lib = other.m_lib;
	m_line = other.m_line;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * ErrorEntry::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(ERROR_ENTRY);
}



void ERR_to_ERROR_ENTRIES(mVector< ErrorEntry > & errors)
{
	unsigned long lastError;
	const char *file,*data;
	int line,flags;
	char * ret_ErrString;
	char * ret_ErrLib;
	char * FileName;
	ERR_STATE * es;
	int i;
	ErrorEntry newError;


	es = ERR_get_state();
	if(!es)
	{
		return;
	}


	data = NULL;

	//We do this not to erase the error from the queue
	for(i=es->bottom+1; i <= es->top; i++)
	{
		lastError = es->err_buffer[i];

		line = es->err_line[i];
		file = es->err_file[i];
		flags=es->err_data_flags[i];
		data=es->err_data[i];

		ret_ErrString = (char *)ERR_reason_error_string(lastError);
		if(!ret_ErrString)
		{
			if(data && *data && (flags & ERR_TXT_STRING) )
			{
				ret_ErrString = (char*)data;
				data = NULL;
			}
			else
				continue;
		}
		
		ret_ErrLib = (char*)ERR_lib_error_string(lastError);		
		if(!ret_ErrLib)
			ret_ErrLib = (char*)ERR_func_error_string(lastError);
		if(!ret_ErrLib)
			ret_ErrLib = "NewPKI";


		FileName = strrchr(file, PATH_SEPARATOR);
		if(!FileName)
			FileName = (char*)file;
		else
			FileName++;


		newError.set_line(line);
		newError.set_file(file);
		newError.set_lib(ERR_GET_LIB(lastError));
		newError.set_function(ERR_GET_FUNC(lastError));
		newError.set_code(ERR_GET_REASON(lastError));

		if(data && (flags & ERR_TXT_STRING))
			newError.set_data(data);
		data=NULL;

		errors.push_back(newError);
		newError.Clear();
	}
}

void ERROR_ENTRIES_to_string(const vector<ErrorEntry> & errors, mString & error, bool OnlyString)
{
	size_t i;
	ERR_clear_error();
	for(i=0; i < errors.size(); i++)
	{
		ERR_put_error(errors[i].get_lib(), errors[i].get_function(), errors[i].get_code(), errors[i].get_file().c_str(), errors[i].get_line());
		if(errors[i].get_data().size())
		{
			ERR_add_error_data(1, errors[i].get_data().c_str());
		}
	}

	ERR_to_mstring(error, OnlyString);
	ERR_clear_error();
}

