/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#ifndef ASN1BACKUP_H
#define ASN1BACKUP_H

#include <mVector.h>
#include <ASN1/NewPKIObject.h>
#include <openssl/asn1.h>
#include <openssl/x509.h>
#include <openssl/asn1.h>
#include <openssl/asn1t.h>
#include <openssl/x509v3.h>
#include <openssl/evp.h>
#include <openssl/pem.h>
#include <openssl/ossl_typ.h>
#include "Asn1Err.h"
#include "Asn1Conf.h"
#include "Asn1Cert.h"
#include "Asn1Resp.h"
#include <mVector.h>
#include <mString.h>



/*!
	This structure represents the backup of an entity cert.
*/
typedef struct st_NEWPKI_BACKUP_CERT
{
	X509 * cert;						//!< The certificate
	ASN1_OCTET_STRING * privkey;		//!< The private key
}NEWPKI_BACKUP_CERT;
DECLARE_ASN1_ITEM(NEWPKI_BACKUP_CERT)


/*!
	This structure represents the backup of an internal CA.
*/
typedef struct st_INTERNAL_CA_BACKUP
{
	ASN1_INTEGER * ca_type;				//!< The CA type.
	INTERNAL_CA * internal_Ca;			//!< The CA
}INTERNAL_CA_BACKUP;
DECLARE_ASN1_ITEM(INTERNAL_CA_BACKUP)




/*!
	This structure represents the backup of a RA entity.
*/
typedef struct st_NEWPKI_BACKUP_RA
{
	NEWPKI_BACKUP_CERT * entity_cert;				//!< The entity cert.
	STACK_OF(NEWPKI_PROFILE_DATAS) * profiles;		//!< The profiles.
	STACK_OF(CRYPTED_NEWPKI_REQUEST) * requests;	//!< The requests.
}NEWPKI_BACKUP_RA;
DECLARE_ASN1_ITEM(NEWPKI_BACKUP_RA)

/*!
	This structure represents the backup of a CA entity.
*/
typedef struct st_NEWPKI_BACKUP_CA
{
	NEWPKI_BACKUP_CERT * entity_cert;				//!< The entity cert.
	STACK_OF(INTERNAL_CA_CERT) * certs;				//!< The certificates.
	INTERNAL_CA_BACKUP * internal_Cas;				//!< The CA.
	STACK_OF(CRYPTED_NEWPKI_REQUEST) * requests;	//!< The requests.
}NEWPKI_BACKUP_CA;
DECLARE_ASN1_ITEM(NEWPKI_BACKUP_CA)

/*!
	This structure represents the backup of a PKI entity.
*/
typedef struct st_NEWPKI_BACKUP_PKI
{
	NEWPKI_BACKUP_CERT * entity_cert;				//!< The entity cert.
	STACK_OF(INTERNAL_CA_CERT) * certs;				//!< The certificates.
	STACK_OF(INTERNAL_CA_BACKUP) * internal_Cas;	//!< The CAs
}NEWPKI_BACKUP_PKI;
DECLARE_ASN1_ITEM(NEWPKI_BACKUP_PKI)

/*!
	This structure represents the backup of a Repository entity.
*/
typedef struct st_NEWPKI_BACKUP_REP
{
	NEWPKI_BACKUP_CERT * entity_cert;				//!< The entity cert.
}NEWPKI_BACKUP_REP;
DECLARE_ASN1_ITEM(NEWPKI_BACKUP_REP)

/*!
	This structure represents the backup of a KeyStore entity.
*/
typedef struct st_NEWPKI_BACKUP_KEY
{
	NEWPKI_BACKUP_CERT * entity_cert;				//!< The entity cert.
}NEWPKI_BACKUP_KEY;
DECLARE_ASN1_ITEM(NEWPKI_BACKUP_KEY)

/*!
	This structure represents the backup of an EE entity.
*/
typedef struct st_NEWPKI_BACKUP_EE
{
	NEWPKI_BACKUP_CERT * entity_cert;				//!< The entity cert.
}NEWPKI_BACKUP_EE;
DECLARE_ASN1_ITEM(NEWPKI_BACKUP_EE)

/*!
	This structure represents the backup of a Backup entity.
*/
typedef struct st_NEWPKI_BACKUP_BACK
{
	NEWPKI_BACKUP_CERT * entity_cert;				//!< The entity cert.
}NEWPKI_BACKUP_BACK;
DECLARE_ASN1_ITEM(NEWPKI_BACKUP_BACK)

/*!
	This structure represents the backup of a Publication entity.
*/
typedef struct st_NEWPKI_BACKUP_PUB
{
	NEWPKI_BACKUP_CERT * entity_cert;				//!< The entity cert.
}NEWPKI_BACKUP_PUB;
DECLARE_ASN1_ITEM(NEWPKI_BACKUP_PUB)


/*!
	This structure represents the backup of an entity.
*/
typedef struct st_NEWPKI_BACKUP
{
	int type;									//!< The type.
	union{
		NEWPKI_BACKUP_RA * ra_backup;			//!< The backup of RA entity.
		NEWPKI_BACKUP_CA * ca_backup;			//!< The backup of CA entity.
		NEWPKI_BACKUP_REP * rep_backup;			//!< The backup of Repository entity.
		NEWPKI_BACKUP_PUB * pub_backup;			//!< The backup of Publication entity.
		NEWPKI_BACKUP_KEY * key_backup;			//!< The backup of KeyStore entity.
		NEWPKI_BACKUP_PKI * pki_backup;			//!< The backup of PKI entity.
		NEWPKI_BACKUP_BACK * back_backup;		//!< The backup of Backup entity.
		NEWPKI_BACKUP_EE * ee_backup;			//!< The backup of EE entity.
	}d;
}NEWPKI_BACKUP;
DECLARE_ASN1_ITEM(NEWPKI_BACKUP)



/*!
  This class is the representation of NEWPKI_BACKUP_CERT
*/
class NewpkiBackupCert : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiBackupCert();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiBackupCert(const NewpkiBackupCert & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiBackupCert();


	/*! \brief This function converts a NEWPKI_BACKUP_CERT to a NewpkiBackupCert.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_BACKUP_CERT * Datas);

	/*! \brief This function converts a NewpkiBackupCert to a NEWPKI_BACKUP_CERT.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_BACKUP_CERT ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_BACKUP_CERT.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509::cert.
	 *  \param c_cert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_cert(const PKI_CERT & c_cert);

	/*! \brief This function returns X509::cert.
	 *  \return The value of X509::cert.
	 */
	const PKI_CERT & get_cert() const;

	/*! \brief This function returns X509::cert.
	 *  \return The value of X509::cert.
	 */
	PKI_CERT & get_cert();


	/*! \brief This function sets ASN1_OCTET_STRING::privkey.
	 *  \param c_privkey [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_privkey(const Asn1OctetString & c_privkey);

	/*! \brief This function returns ASN1_OCTET_STRING::privkey.
	 *  \return The value of ASN1_OCTET_STRING::privkey.
	 */
	const Asn1OctetString & get_privkey() const;

	/*! \brief This function returns ASN1_OCTET_STRING::privkey.
	 *  \return The value of ASN1_OCTET_STRING::privkey.
	 */
	Asn1OctetString & get_privkey();


	/*! \brief This function copies a NewpkiBackupCert to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiBackupCert & other);
	/*! \brief This member is an empty instance of NewpkiBackupCert.
	 */
	static NewpkiBackupCert EmptyInstance;

private:
	PKI_CERT m_cert;
	Asn1OctetString m_privkey;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of INTERNAL_CA_BACKUP
*/
class InternalCaBackup : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	InternalCaBackup();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	InternalCaBackup(const InternalCaBackup & other);

	/*! \brief This is the destructor.
	 */
	~InternalCaBackup();


	/*! \brief This function converts a INTERNAL_CA_BACKUP to a InternalCaBackup.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const INTERNAL_CA_BACKUP * Datas);

	/*! \brief This function converts a InternalCaBackup to a INTERNAL_CA_BACKUP.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(INTERNAL_CA_BACKUP ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of INTERNAL_CA_BACKUP.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::ca_type.
	 *  \param c_caType [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_caType(unsigned long c_caType);

	/*! \brief This function returns ASN1_INTEGER::ca_type.
	 *  \return The value of ASN1_INTEGER::ca_type.
	 */
	unsigned long get_caType() const;


	/*! \brief This function sets INTERNAL_CA::internal_Ca.
	 *  \param c_internalCa [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_internalCa(const InternalCa & c_internalCa);

	/*! \brief This function returns INTERNAL_CA::internal_Ca.
	 *  \return The value of INTERNAL_CA::internal_Ca.
	 */
	const InternalCa & get_internalCa() const;

	/*! \brief This function returns INTERNAL_CA::internal_Ca.
	 *  \return The value of INTERNAL_CA::internal_Ca.
	 */
	InternalCa & get_internalCa();


	/*! \brief This function copies a InternalCaBackup to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const InternalCaBackup & other);
	/*! \brief This member is an empty instance of InternalCaBackup.
	 */
	static InternalCaBackup EmptyInstance;

private:
	unsigned long m_caType;
	InternalCa m_internalCa;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_BACKUP_RA
*/
class NewpkiBackupRa : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiBackupRa();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiBackupRa(const NewpkiBackupRa & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiBackupRa();


	/*! \brief This function converts a NEWPKI_BACKUP_RA to a NewpkiBackupRa.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_BACKUP_RA * Datas);

	/*! \brief This function converts a NewpkiBackupRa to a NEWPKI_BACKUP_RA.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_BACKUP_RA ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_BACKUP_RA.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets NEWPKI_BACKUP_CERT::entity_cert.
	 *  \param c_entityCert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityCert(const NewpkiBackupCert & c_entityCert);

	/*! \brief This function returns NEWPKI_BACKUP_CERT::entity_cert.
	 *  \return The value of NEWPKI_BACKUP_CERT::entity_cert.
	 */
	const NewpkiBackupCert & get_entityCert() const;

	/*! \brief This function returns NEWPKI_BACKUP_CERT::entity_cert.
	 *  \return The value of NEWPKI_BACKUP_CERT::entity_cert.
	 */
	NewpkiBackupCert & get_entityCert();


	/*! \brief This function sets NEWPKI_PROFILE_DATAS::profiles.
	 *  \param c_profiles [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_profiles(const mVector< NewpkiProfileDatas > & c_profiles);

	/*! \brief This function returns NEWPKI_PROFILE_DATAS::profiles.
	 *  \return The value of NEWPKI_PROFILE_DATAS::profiles.
	 */
	const mVector< NewpkiProfileDatas > & get_profiles() const;

	/*! \brief This function returns NEWPKI_PROFILE_DATAS::profiles.
	 *  \return The value of NEWPKI_PROFILE_DATAS::profiles.
	 */
	mVector< NewpkiProfileDatas > & get_profiles();


	/*! \brief This function sets CRYPTED_NEWPKI_REQUEST::requests.
	 *  \param c_requests [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_requests(const mVector< CryptedNewpkiRequest > & c_requests);

	/*! \brief This function returns CRYPTED_NEWPKI_REQUEST::requests.
	 *  \return The value of CRYPTED_NEWPKI_REQUEST::requests.
	 */
	const mVector< CryptedNewpkiRequest > & get_requests() const;

	/*! \brief This function returns CRYPTED_NEWPKI_REQUEST::requests.
	 *  \return The value of CRYPTED_NEWPKI_REQUEST::requests.
	 */
	mVector< CryptedNewpkiRequest > & get_requests();


	/*! \brief This function copies a NewpkiBackupRa to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiBackupRa & other);
	/*! \brief This member is an empty instance of NewpkiBackupRa.
	 */
	static NewpkiBackupRa EmptyInstance;

private:
	NewpkiBackupCert m_entityCert;
	mVector< NewpkiProfileDatas > m_profiles;
	mVector< CryptedNewpkiRequest > m_requests;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_BACKUP_CA
*/
class NewpkiBackupCa : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiBackupCa();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiBackupCa(const NewpkiBackupCa & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiBackupCa();


	/*! \brief This function converts a NEWPKI_BACKUP_CA to a NewpkiBackupCa.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_BACKUP_CA * Datas);

	/*! \brief This function converts a NewpkiBackupCa to a NEWPKI_BACKUP_CA.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_BACKUP_CA ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_BACKUP_CA.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets INTERNAL_CA_CERT::certs.
	 *  \param c_certs [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_certs(const mVector< InternalCaCert > & c_certs);

	/*! \brief This function returns INTERNAL_CA_CERT::certs.
	 *  \return The value of INTERNAL_CA_CERT::certs.
	 */
	const mVector< InternalCaCert > & get_certs() const;

	/*! \brief This function returns INTERNAL_CA_CERT::certs.
	 *  \return The value of INTERNAL_CA_CERT::certs.
	 */
	mVector< InternalCaCert > & get_certs();


	/*! \brief This function sets NEWPKI_BACKUP_CERT::entity_cert.
	 *  \param c_entityCert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityCert(const NewpkiBackupCert & c_entityCert);

	/*! \brief This function returns NEWPKI_BACKUP_CERT::entity_cert.
	 *  \return The value of NEWPKI_BACKUP_CERT::entity_cert.
	 */
	const NewpkiBackupCert & get_entityCert() const;

	/*! \brief This function returns NEWPKI_BACKUP_CERT::entity_cert.
	 *  \return The value of NEWPKI_BACKUP_CERT::entity_cert.
	 */
	NewpkiBackupCert & get_entityCert();


	/*! \brief This function sets INTERNAL_CA_BACKUP::internal_Cas.
	 *  \param c_internalCas [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_internalCas(const InternalCaBackup & c_internalCas);

	/*! \brief This function returns INTERNAL_CA_BACKUP::internal_Cas.
	 *  \return The value of INTERNAL_CA_BACKUP::internal_Cas.
	 */
	const InternalCaBackup & get_internalCas() const;

	/*! \brief This function returns INTERNAL_CA_BACKUP::internal_Cas.
	 *  \return The value of INTERNAL_CA_BACKUP::internal_Cas.
	 */
	InternalCaBackup & get_internalCas();


	/*! \brief This function sets CRYPTED_NEWPKI_REQUEST::requests.
	 *  \param c_requests [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_requests(const mVector< CryptedNewpkiRequest > & c_requests);

	/*! \brief This function returns CRYPTED_NEWPKI_REQUEST::requests.
	 *  \return The value of CRYPTED_NEWPKI_REQUEST::requests.
	 */
	const mVector< CryptedNewpkiRequest > & get_requests() const;

	/*! \brief This function returns CRYPTED_NEWPKI_REQUEST::requests.
	 *  \return The value of CRYPTED_NEWPKI_REQUEST::requests.
	 */
	mVector< CryptedNewpkiRequest > & get_requests();


	/*! \brief This function copies a NewpkiBackupCa to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiBackupCa & other);
	/*! \brief This member is an empty instance of NewpkiBackupCa.
	 */
	static NewpkiBackupCa EmptyInstance;

private:
	mVector< InternalCaCert > m_certs;
	NewpkiBackupCert m_entityCert;
	InternalCaBackup m_internalCas;
	mVector< CryptedNewpkiRequest > m_requests;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_BACKUP_PKI
*/
class NewpkiBackupPki : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiBackupPki();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiBackupPki(const NewpkiBackupPki & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiBackupPki();


	/*! \brief This function converts a NEWPKI_BACKUP_PKI to a NewpkiBackupPki.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_BACKUP_PKI * Datas);

	/*! \brief This function converts a NewpkiBackupPki to a NEWPKI_BACKUP_PKI.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_BACKUP_PKI ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_BACKUP_PKI.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets INTERNAL_CA_CERT::certs.
	 *  \param c_certs [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_certs(const mVector< InternalCaCert > & c_certs);

	/*! \brief This function returns INTERNAL_CA_CERT::certs.
	 *  \return The value of INTERNAL_CA_CERT::certs.
	 */
	const mVector< InternalCaCert > & get_certs() const;

	/*! \brief This function returns INTERNAL_CA_CERT::certs.
	 *  \return The value of INTERNAL_CA_CERT::certs.
	 */
	mVector< InternalCaCert > & get_certs();


	/*! \brief This function sets NEWPKI_BACKUP_CERT::entity_cert.
	 *  \param c_entityCert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityCert(const NewpkiBackupCert & c_entityCert);

	/*! \brief This function returns NEWPKI_BACKUP_CERT::entity_cert.
	 *  \return The value of NEWPKI_BACKUP_CERT::entity_cert.
	 */
	const NewpkiBackupCert & get_entityCert() const;

	/*! \brief This function returns NEWPKI_BACKUP_CERT::entity_cert.
	 *  \return The value of NEWPKI_BACKUP_CERT::entity_cert.
	 */
	NewpkiBackupCert & get_entityCert();


	/*! \brief This function sets INTERNAL_CA_BACKUP::internal_Cas.
	 *  \param c_internalCas [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_internalCas(const mVector< InternalCaBackup > & c_internalCas);

	/*! \brief This function returns INTERNAL_CA_BACKUP::internal_Cas.
	 *  \return The value of INTERNAL_CA_BACKUP::internal_Cas.
	 */
	const mVector< InternalCaBackup > & get_internalCas() const;

	/*! \brief This function returns INTERNAL_CA_BACKUP::internal_Cas.
	 *  \return The value of INTERNAL_CA_BACKUP::internal_Cas.
	 */
	mVector< InternalCaBackup > & get_internalCas();


	/*! \brief This function copies a NewpkiBackupPki to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiBackupPki & other);
	/*! \brief This member is an empty instance of NewpkiBackupPki.
	 */
	static NewpkiBackupPki EmptyInstance;

private:
	mVector< InternalCaCert > m_certs;
	NewpkiBackupCert m_entityCert;
	mVector< InternalCaBackup > m_internalCas;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_BACKUP_REP
*/
class NewpkiBackupRep : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiBackupRep();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiBackupRep(const NewpkiBackupRep & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiBackupRep();


	/*! \brief This function converts a NEWPKI_BACKUP_REP to a NewpkiBackupRep.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_BACKUP_REP * Datas);

	/*! \brief This function converts a NewpkiBackupRep to a NEWPKI_BACKUP_REP.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_BACKUP_REP ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_BACKUP_REP.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets NEWPKI_BACKUP_CERT::entity_cert.
	 *  \param c_entityCert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityCert(const NewpkiBackupCert & c_entityCert);

	/*! \brief This function returns NEWPKI_BACKUP_CERT::entity_cert.
	 *  \return The value of NEWPKI_BACKUP_CERT::entity_cert.
	 */
	const NewpkiBackupCert & get_entityCert() const;

	/*! \brief This function returns NEWPKI_BACKUP_CERT::entity_cert.
	 *  \return The value of NEWPKI_BACKUP_CERT::entity_cert.
	 */
	NewpkiBackupCert & get_entityCert();


	/*! \brief This function copies a NewpkiBackupRep to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiBackupRep & other);
	/*! \brief This member is an empty instance of NewpkiBackupRep.
	 */
	static NewpkiBackupRep EmptyInstance;

private:
	NewpkiBackupCert m_entityCert;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_BACKUP_PUB
*/
class NewpkiBackupPub : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiBackupPub();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiBackupPub(const NewpkiBackupPub & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiBackupPub();


	/*! \brief This function converts a NEWPKI_BACKUP_PUB to a NewpkiBackupPub.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_BACKUP_PUB * Datas);

	/*! \brief This function converts a NewpkiBackupPub to a NEWPKI_BACKUP_PUB.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_BACKUP_PUB ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_BACKUP_PUB.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets NEWPKI_BACKUP_CERT::entity_cert.
	 *  \param c_entityCert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityCert(const NewpkiBackupCert & c_entityCert);

	/*! \brief This function returns NEWPKI_BACKUP_CERT::entity_cert.
	 *  \return The value of NEWPKI_BACKUP_CERT::entity_cert.
	 */
	const NewpkiBackupCert & get_entityCert() const;

	/*! \brief This function returns NEWPKI_BACKUP_CERT::entity_cert.
	 *  \return The value of NEWPKI_BACKUP_CERT::entity_cert.
	 */
	NewpkiBackupCert & get_entityCert();


	/*! \brief This function copies a NewpkiBackupPub to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiBackupPub & other);
	/*! \brief This member is an empty instance of NewpkiBackupPub.
	 */
	static NewpkiBackupPub EmptyInstance;

private:
	NewpkiBackupCert m_entityCert;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_BACKUP_KEY
*/
class NewpkiBackupKey : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiBackupKey();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiBackupKey(const NewpkiBackupKey & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiBackupKey();


	/*! \brief This function converts a NEWPKI_BACKUP_KEY to a NewpkiBackupKey.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_BACKUP_KEY * Datas);

	/*! \brief This function converts a NewpkiBackupKey to a NEWPKI_BACKUP_KEY.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_BACKUP_KEY ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_BACKUP_KEY.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets NEWPKI_BACKUP_CERT::entity_cert.
	 *  \param c_entityCert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityCert(const NewpkiBackupCert & c_entityCert);

	/*! \brief This function returns NEWPKI_BACKUP_CERT::entity_cert.
	 *  \return The value of NEWPKI_BACKUP_CERT::entity_cert.
	 */
	const NewpkiBackupCert & get_entityCert() const;

	/*! \brief This function returns NEWPKI_BACKUP_CERT::entity_cert.
	 *  \return The value of NEWPKI_BACKUP_CERT::entity_cert.
	 */
	NewpkiBackupCert & get_entityCert();


	/*! \brief This function copies a NewpkiBackupKey to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiBackupKey & other);
	/*! \brief This member is an empty instance of NewpkiBackupKey.
	 */
	static NewpkiBackupKey EmptyInstance;

private:
	NewpkiBackupCert m_entityCert;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_BACKUP_EE
*/
class NewpkiBackupEe : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiBackupEe();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiBackupEe(const NewpkiBackupEe & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiBackupEe();


	/*! \brief This function converts a NEWPKI_BACKUP_EE to a NewpkiBackupEe.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_BACKUP_EE * Datas);

	/*! \brief This function converts a NewpkiBackupEe to a NEWPKI_BACKUP_EE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_BACKUP_EE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_BACKUP_EE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets NEWPKI_BACKUP_CERT::entity_cert.
	 *  \param c_entityCert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityCert(const NewpkiBackupCert & c_entityCert);

	/*! \brief This function returns NEWPKI_BACKUP_CERT::entity_cert.
	 *  \return The value of NEWPKI_BACKUP_CERT::entity_cert.
	 */
	const NewpkiBackupCert & get_entityCert() const;

	/*! \brief This function returns NEWPKI_BACKUP_CERT::entity_cert.
	 *  \return The value of NEWPKI_BACKUP_CERT::entity_cert.
	 */
	NewpkiBackupCert & get_entityCert();


	/*! \brief This function copies a NewpkiBackupEe to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiBackupEe & other);
	/*! \brief This member is an empty instance of NewpkiBackupEe.
	 */
	static NewpkiBackupEe EmptyInstance;

private:
	NewpkiBackupCert m_entityCert;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_BACKUP_BACK
*/
class NewpkiBackupBack : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiBackupBack();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiBackupBack(const NewpkiBackupBack & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiBackupBack();


	/*! \brief This function converts a NEWPKI_BACKUP_BACK to a NewpkiBackupBack.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_BACKUP_BACK * Datas);

	/*! \brief This function converts a NewpkiBackupBack to a NEWPKI_BACKUP_BACK.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_BACKUP_BACK ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_BACKUP_BACK.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets NEWPKI_BACKUP_CERT::entity_cert.
	 *  \param c_entityCert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityCert(const NewpkiBackupCert & c_entityCert);

	/*! \brief This function returns NEWPKI_BACKUP_CERT::entity_cert.
	 *  \return The value of NEWPKI_BACKUP_CERT::entity_cert.
	 */
	const NewpkiBackupCert & get_entityCert() const;

	/*! \brief This function returns NEWPKI_BACKUP_CERT::entity_cert.
	 *  \return The value of NEWPKI_BACKUP_CERT::entity_cert.
	 */
	NewpkiBackupCert & get_entityCert();


	/*! \brief This function copies a NewpkiBackupBack to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiBackupBack & other);
	/*! \brief This member is an empty instance of NewpkiBackupBack.
	 */
	static NewpkiBackupBack EmptyInstance;

private:
	NewpkiBackupCert m_entityCert;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of NEWPKI_BACKUP
*/
class NewpkiBackup : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiBackup();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	NewpkiBackup(const NewpkiBackup & other);

	/*! \brief This is the destructor.
	 */
	~NewpkiBackup();


	/*! \brief This function converts a NEWPKI_BACKUP to a NewpkiBackup.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const NEWPKI_BACKUP * Datas);

	/*! \brief This function converts a NewpkiBackup to a NEWPKI_BACKUP.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(NEWPKI_BACKUP ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of NEWPKI_BACKUP.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets NEWPKI_BACKUP_BACK::d.back_backup.
	 *  \param c_backBackup [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_backBackup(const NewpkiBackupBack & c_backBackup);

	/*! \brief This function returns NEWPKI_BACKUP_BACK::d.back_backup.
	 *  \return The value of NEWPKI_BACKUP_BACK::d.back_backup.
	 */
	const NewpkiBackupBack & get_backBackup() const;

	/*! \brief This function returns NEWPKI_BACKUP_BACK::d.back_backup.
	 *  \return The value of NEWPKI_BACKUP_BACK::d.back_backup.
	 */
	NewpkiBackupBack & get_backBackup();


	/*! \brief This function sets NEWPKI_BACKUP_CA::d.ca_backup.
	 *  \param c_caBackup [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_caBackup(const NewpkiBackupCa & c_caBackup);

	/*! \brief This function returns NEWPKI_BACKUP_CA::d.ca_backup.
	 *  \return The value of NEWPKI_BACKUP_CA::d.ca_backup.
	 */
	const NewpkiBackupCa & get_caBackup() const;

	/*! \brief This function returns NEWPKI_BACKUP_CA::d.ca_backup.
	 *  \return The value of NEWPKI_BACKUP_CA::d.ca_backup.
	 */
	NewpkiBackupCa & get_caBackup();


	/*! \brief This function sets NEWPKI_BACKUP_EE::d.ee_backup.
	 *  \param c_eeBackup [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeBackup(const NewpkiBackupEe & c_eeBackup);

	/*! \brief This function returns NEWPKI_BACKUP_EE::d.ee_backup.
	 *  \return The value of NEWPKI_BACKUP_EE::d.ee_backup.
	 */
	const NewpkiBackupEe & get_eeBackup() const;

	/*! \brief This function returns NEWPKI_BACKUP_EE::d.ee_backup.
	 *  \return The value of NEWPKI_BACKUP_EE::d.ee_backup.
	 */
	NewpkiBackupEe & get_eeBackup();


	/*! \brief This function sets NEWPKI_BACKUP_KEY::d.key_backup.
	 *  \param c_keyBackup [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_keyBackup(const NewpkiBackupKey & c_keyBackup);

	/*! \brief This function returns NEWPKI_BACKUP_KEY::d.key_backup.
	 *  \return The value of NEWPKI_BACKUP_KEY::d.key_backup.
	 */
	const NewpkiBackupKey & get_keyBackup() const;

	/*! \brief This function returns NEWPKI_BACKUP_KEY::d.key_backup.
	 *  \return The value of NEWPKI_BACKUP_KEY::d.key_backup.
	 */
	NewpkiBackupKey & get_keyBackup();


	/*! \brief This function sets NEWPKI_BACKUP_PKI::d.pki_backup.
	 *  \param c_pkiBackup [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_pkiBackup(const NewpkiBackupPki & c_pkiBackup);

	/*! \brief This function returns NEWPKI_BACKUP_PKI::d.pki_backup.
	 *  \return The value of NEWPKI_BACKUP_PKI::d.pki_backup.
	 */
	const NewpkiBackupPki & get_pkiBackup() const;

	/*! \brief This function returns NEWPKI_BACKUP_PKI::d.pki_backup.
	 *  \return The value of NEWPKI_BACKUP_PKI::d.pki_backup.
	 */
	NewpkiBackupPki & get_pkiBackup();


	/*! \brief This function sets NEWPKI_BACKUP_PUB::d.pub_backup.
	 *  \param c_pubBackup [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_pubBackup(const NewpkiBackupPub & c_pubBackup);

	/*! \brief This function returns NEWPKI_BACKUP_PUB::d.pub_backup.
	 *  \return The value of NEWPKI_BACKUP_PUB::d.pub_backup.
	 */
	const NewpkiBackupPub & get_pubBackup() const;

	/*! \brief This function returns NEWPKI_BACKUP_PUB::d.pub_backup.
	 *  \return The value of NEWPKI_BACKUP_PUB::d.pub_backup.
	 */
	NewpkiBackupPub & get_pubBackup();


	/*! \brief This function sets NEWPKI_BACKUP_RA::d.ra_backup.
	 *  \param c_raBackup [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_raBackup(const NewpkiBackupRa & c_raBackup);

	/*! \brief This function returns NEWPKI_BACKUP_RA::d.ra_backup.
	 *  \return The value of NEWPKI_BACKUP_RA::d.ra_backup.
	 */
	const NewpkiBackupRa & get_raBackup() const;

	/*! \brief This function returns NEWPKI_BACKUP_RA::d.ra_backup.
	 *  \return The value of NEWPKI_BACKUP_RA::d.ra_backup.
	 */
	NewpkiBackupRa & get_raBackup();


	/*! \brief This function sets NEWPKI_BACKUP_REP::d.rep_backup.
	 *  \param c_repBackup [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_repBackup(const NewpkiBackupRep & c_repBackup);

	/*! \brief This function returns NEWPKI_BACKUP_REP::d.rep_backup.
	 *  \return The value of NEWPKI_BACKUP_REP::d.rep_backup.
	 */
	const NewpkiBackupRep & get_repBackup() const;

	/*! \brief This function returns NEWPKI_BACKUP_REP::d.rep_backup.
	 *  \return The value of NEWPKI_BACKUP_REP::d.rep_backup.
	 */
	NewpkiBackupRep & get_repBackup();


	/*! \brief This function copies a NewpkiBackup to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const NewpkiBackup & other);
	/*! \brief This member is an empty instance of NewpkiBackup.
	 */
	static NewpkiBackup EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	NewpkiBackupBack * m_backBackup;
	NewpkiBackupCa * m_caBackup;
	NewpkiBackupEe * m_eeBackup;
	NewpkiBackupKey * m_keyBackup;
	NewpkiBackupPki * m_pkiBackup;
	NewpkiBackupPub * m_pubBackup;
	NewpkiBackupRa * m_raBackup;
	NewpkiBackupRep * m_repBackup;
	void resetAll();
	void freeAll();
protected:
};


#endif // #ifndef ASN1BACKUP_H
