/*  Neutrino:  neutrino-util.c
 *
 *  Copyright (C) 2002 David A Knight <david@ritter.demon.co.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */
#include <config.h>

#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include <glib/gconvert.h>
#include <glib/gunicode.h>
#include <glib/gstrfuncs.h>
#include <glib/gstring.h>
#include <gtk/gtkwidget.h>

#ifndef HAVE_GNOME_VFS

#include <glib/gfileutils.h>
#include <glib/gdir.h>

#include <sys/stat.h>
#include <sys/types.h>

#else

#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-directory.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#include <libgnomevfs/gnome-vfs-ops.h>

#endif


#include "neutrino-util.h"

void neutrino_change_state( GtkWidget *widget )
{
	gtk_widget_set_sensitive( widget,
				  ! GTK_WIDGET_IS_SENSITIVE( widget ) );
}

gchar *neutrino_util_create_backup_filename( const gchar *uri )
{
#ifndef HAVE_GNOME_VFS
	GString *tempuri;
	gchar *ret;
	gboolean exist;

	tempuri = g_string_new( uri );

	do {
		g_string_append_c( tempuri, '~' );
		exist = g_file_test( tempuri->str, G_FILE_TEST_EXISTS );
	} while( exist );

	ret = tempuri->str;
	g_string_free( tempuri, FALSE );

	return ret;
#else
	GString *tempuri;
	gchar *ret;
	GnomeVFSURI *guri;
	gboolean exist;

	tempuri = g_string_new( uri );

	do {
		g_string_append_c( tempuri, '~' );
		guri = gnome_vfs_uri_new( tempuri->str );
		exist = gnome_vfs_uri_exists( guri );
		gnome_vfs_uri_unref( guri );
	} while( exist );

	ret = tempuri->str;
	g_string_free( tempuri, FALSE );

	return ret;
#endif
}

gboolean neutrino_util_concat_file_to_handle( void *handle,
					const gchar *file2,
					gsize offset,
					gsize *written )
{
#ifndef HAVE_GNOME_VFS
	FILE *fhandle;
	FILE *handle2;
	gboolean ok;
	gchar buffer[ 4096 ];
	size_t red;
	size_t wrote;
	
	*written = 0;
	
	fhandle = (FILE*)handle;
	handle2 = NULL;

	handle2 = fopen( file2, "rb" );
	ok = ( handle2 != NULL );
	if( ok ) {
		ok = ( fseek( handle2, offset, SEEK_SET ) == 0 );
	}
	while( ok ) {
		red = fread( buffer, 1, 4096, handle2 );
		if( red > 0 ) {
			wrote = fwrite( buffer, 1, red, fhandle );
			if( wrote > 0 ) {
				(*written) += wrote;
			} else {
				ok = FALSE;
			}
		} else if( ! feof( handle2 ) ) {
			ok = FALSE;
		} else {
			break;
		}
	}
	if( handle2 ) {
		fclose( handle2 );
	}

	return ok;
#else
	GnomeVFSHandle *vhandle;
	GnomeVFSHandle *handle2;
	GnomeVFSResult result;
	GnomeVFSFileSize got;
	GnomeVFSFileSize wrote;

	vhandle = (GnomeVFSHandle*)handle;
	handle2 = NULL;
	result = gnome_vfs_open( &handle2, file2,
				 GNOME_VFS_OPEN_READ );
	*written = 0;
	if( result == GNOME_VFS_OK ) {
		result = gnome_vfs_seek( handle2, GNOME_VFS_SEEK_START,
					 offset );
	}
	if( result == GNOME_VFS_OK ) {
		gchar buffer[ 4096 ];
		do {
			result = gnome_vfs_read( handle2,
						 buffer,
						 4096, &got );
			if( result == GNOME_VFS_OK ) {
				gnome_vfs_write( vhandle,
						 buffer,
						 got,
						 &wrote );
				(*written) += wrote;
			}
		} while( result == GNOME_VFS_OK );
		if( result == GNOME_VFS_ERROR_EOF ) {
			result = GNOME_VFS_OK;
		}
	}
	if( handle2 ) {
		gnome_vfs_close( handle2 );
	}

	return ( result == GNOME_VFS_OK );
#endif
}

gboolean neutrino_util_concat_handle_to_file( void *handle,
					const gchar *file2,
					gsize offset,
					gsize *written )
{
#ifndef HAVE_GNOME_VFS
	FILE *vhandle;
	FILE *handle2;
	gboolean ok;
	gchar buffer[ 4096 ];
	size_t red;
	size_t wrote;
	
	vhandle = (FILE*)handle;
	handle2 = fopen( file2, "a+b" );
	ok = ( handle2 != NULL );
	*written = 0;
	
	if( ok ) {
		ok = ( fseek( handle2, offset, SEEK_SET ) == 0 );
	}
	while( ok ) {
		red = fread( buffer, 1, 4096, vhandle );
		if( red > 0 ) {
			wrote = fwrite( buffer, 1, red, handle2 );
			if( wrote > 0 ) {
				(*written) += wrote;
			} else {
				ok = FALSE;
			}
		} else if( ! feof( vhandle ) ) {
			ok = FALSE;
		} else {
			break;
		}
	}
	if( handle2 ) {
		fclose( handle2 );
	}

	return ok;
#else
	GnomeVFSHandle *vhandle;
	GnomeVFSHandle *handle2;
	GnomeVFSResult result;

	vhandle = (GnomeVFSHandle*)handle;
	handle2 = NULL;
	*written = 0;
	result = gnome_vfs_open( &handle2, file2,
				 GNOME_VFS_OPEN_WRITE );
	if( result == GNOME_VFS_ERROR_NOT_FOUND ) {
		result = gnome_vfs_create( &handle2,
					   file2,
					   GNOME_VFS_OPEN_WRITE,
					   TRUE,
					   GNOME_VFS_PERM_USER_READ |
					   GNOME_VFS_PERM_USER_WRITE |
					   GNOME_VFS_PERM_GROUP_READ |
					   GNOME_VFS_PERM_OTHER_READ );
	}

	if( result == GNOME_VFS_OK ) {
		result = gnome_vfs_seek( handle2, GNOME_VFS_SEEK_START,
					 offset );
	}
	if( result == GNOME_VFS_OK ) {
		gchar buffer[ 4096 ];
		GnomeVFSFileSize got;
		GnomeVFSFileSize wrote;
		do {
			result = gnome_vfs_read( vhandle,
						 buffer,
						 4096, &got );
			if( result == GNOME_VFS_OK ) {
				gnome_vfs_write( handle2,
						 buffer,
						 got,
						 &wrote );
			}
		} while( result == GNOME_VFS_OK );
		if( result == GNOME_VFS_ERROR_EOF ) {
			result = GNOME_VFS_OK;
		}
	}
	if( handle2 ) {
		gnome_vfs_close( handle2 );
	}

	return ( result == GNOME_VFS_OK );
#endif
}

GList *neutrino_util_recurse_directory( const gchar *uri, GList *list )
{
#ifndef HAVE_GNOME_VFS
	GDir *dir;
	const gchar *name;
	gchar *nexturi;

	dir = g_dir_open( uri, 0, NULL );
	if( ! dir ) {
		list = g_list_prepend( list, g_strdup( uri ) );
	} else {
		name = g_dir_read_name( dir );
				
		if( uri[ strlen( uri ) - 1 ] != '/' ) {
			nexturi = g_strconcat( uri, "/", name, NULL );
		} else {
			nexturi = g_strconcat( uri, name, NULL );
		}
		list = neutrino_util_recurse_directory( nexturi, list );

		g_dir_close( dir );
	}

	return list;
#else
	GnomeVFSResult result;
	GnomeVFSDirectoryHandle *handle;
	GnomeVFSFileInfo *info;

	result = gnome_vfs_directory_open( &handle, uri,
					   GNOME_VFS_FILE_INFO_FOLLOW_LINKS );
	if( result == GNOME_VFS_ERROR_NOT_A_DIRECTORY ) {
		list = g_list_prepend( list, g_strdup( uri ) );
	} else if( result == GNOME_VFS_OK ) {
		info = gnome_vfs_file_info_new();
		
		do {
			result = gnome_vfs_directory_read_next( handle, info );
			
			if( result == GNOME_VFS_OK && *info->name != '.' ) {
				gchar *nexturi;
				
				if( uri[ strlen( uri ) - 1 ] != '/' ) {
					nexturi = g_strconcat( uri,
							       "/",
							       info->name,
							       NULL );
				} else {
					nexturi = g_strconcat( uri,
							       info->name,
							       NULL );
				}
				
				list = neutrino_util_recurse_directory(nexturi,
								       list );
				
				g_free( nexturi );
			}
		} while( result == GNOME_VFS_OK );

		gnome_vfs_directory_close( handle );
		gnome_vfs_file_info_unref( info );
	}
	return list;
#endif
}

GString *neutrino_util_load_file( const gchar *path )
{
#ifndef HAVE_GNOME_VFS
	GString *ret;
	gchar *temp;
	gsize len;

	ret = NULL;
	temp = NULL;

	if( g_file_get_contents( path, &temp, &len, NULL ) ) {
		ret = g_string_new( NULL );
		g_string_append_len( ret, temp, len );
	}
	g_free( temp );	

	return ret;
#else
	GnomeVFSResult result;
	GString *ret;
	GnomeVFSFileSize len;
	gchar *buf;

	ret = NULL;
	buf = NULL;

	result = gnome_vfs_read_entire_file( path, &len, &buf );
	if( result == GNOME_VFS_OK ) {
		ret = g_string_new( NULL );
		g_string_append_len( ret, buf, len );
	}
	g_free( buf );

	return ret;
#endif
}



gchar *neutrino_util_charset_convert( const gchar *data )
{
	const gchar *end;
	gchar *ret;
	guint len;
	
	ret = NULL;
	if( data && ! g_utf8_validate( data, strlen( data ), &end ) ) {
		gchar *temp = NULL;
		const gchar *charset;
		gboolean utf8;
		const char *encodings_to_try[ 2 ] = { 0, 0 };

		utf8 = g_get_charset( &charset );

		if( ! utf8 ) {
			encodings_to_try[ 0 ] = charset;
		}
		if( g_ascii_strcasecmp( charset, "ISO-8859-1" ) ) {
			encodings_to_try[ 1 ] = "ISO-8859-1";
		}

		if( encodings_to_try[ 0 ] ) {
			temp = g_convert( data, strlen( data ), "UTF-8", 
					  encodings_to_try[ 0 ], 
					  NULL, &len, NULL );
		}
		if( ! temp && encodings_to_try[ 1 ] ) {
			temp = g_convert( data, strlen( data ), "UTF-8", 
					  encodings_to_try[ 1 ], 
					  NULL, &len, NULL );
		}

		if( temp ) {
			data = temp;
		}
		ret = data;
	} else if( data ) {
		ret = g_strdup( data );
	}

	return ret;
}


