/* pnminterp - scale up portable anymap by interpolating between pixels.
 * Copyright (C) 1998,2000 Russell Marks.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "pnm.h"

enum an_edge_mode {
    EDGE_DROP,
        /* drop one (source) pixel at right/bottom edges. */
    EDGE_INTERP_TO_BLACK,
        /* interpolate right/bottom edge pixels to black. */
    EDGE_NON_INTERP
        /* don't interpolate right/bottom edge pixels 
           (default, and what zgv does). */
};


struct cmdline_info {
    /* All the information the user supplied in the command line,
       in a form easy for the program to use.
    */
    char *input_filespec;  /* Filespecs of input files */
    enum an_edge_mode edge_mode;
    int scale;
};



/* prototypes */
void usage(void);
static void interp(FILE *in, const int cols, const int rows, 
                   const xelval maxval, const int format,
                   const int output_format,
                   const int scale, const enum an_edge_mode edge_mode);
static void 
write_interp_line(const xel * const curline,
                  const xel * const nextline, 
                  xel * const outbuf,
                  const int cols,const xelval maxval, const int format,
                  const int scale, const int edge_mode);

static void
parse_command_line(int argc, char ** argv,
                   struct cmdline_info *cmdline_p) {
/*----------------------------------------------------------------------------
   Note that the file spec array we return is stored in the storage that
   was passed to us as the argv array.
-----------------------------------------------------------------------------*/
    optStruct *option_def = malloc(100*sizeof(optStruct));
        /* Instructions to OptParseOptions2 on how to parse our options.
         */
    optStruct2 opt;
    int blackedge;
    int dropedge;

    unsigned int option_def_index;

    option_def_index = 0;   /* incremented by OPTENTRY */
    OPTENTRY('b',   "blackedge",    OPT_FLAG,    &blackedge,           0);
    OPTENTRY('d',   "dropedge",     OPT_FLAG,    &dropedge,            0);

    /* Set the defaults */
    blackedge = FALSE;
    dropedge = FALSE;
    
    opt.opt_table = option_def;
    opt.short_allowed = TRUE;  /* We have some short (old-fashioned) options */
    opt.allowNegNum = TRUE;  /* We may have parms that are negative numbers */

    pm_optParseOptions2(&argc, argv, opt, 0);
        /* Uses and sets argc, argv, and some of *cmdline_p and others. */

    if (blackedge && dropedge) 
        pm_error("Can't specify both -blackedge and -dropedge options.");
    else if (blackedge)
        cmdline_p->edge_mode = EDGE_INTERP_TO_BLACK;
    else if (dropedge)
        cmdline_p->edge_mode = EDGE_DROP;
    else
        cmdline_p->edge_mode = EDGE_NON_INTERP;

    if ((argc-1) != 1 && (argc-1) != 2)
        pm_error("Wrong number of arguments.  "
                 "Must be 1 or 2 arguments: scale and optional filename");

    {
        char *endptr;   /* ptr to 1st invalid character in scale arg */

        cmdline_p->scale = strtol(argv[1], &endptr, 10);
        if (*argv[1] == '\0') 
            pm_error("Scale argument is a null string.  Must be a number.");
        else if (*endptr != '\0')
            pm_error("Scale argument contains non-numeric character '%c'.",
                     *endptr);
        else if (cmdline_p->scale < 2)
            pm_error("Scale argument must be at least 2.  You specified %d",
                     cmdline_p->scale);
    }
    if (argc-1 > 1) 
        cmdline_p->input_filespec = argv[2];
    else
        cmdline_p->input_filespec = "-";

}



int 
main(int argc,char *argv[]) {

    FILE *ifp;

    int rows, cols, format, output_format;
    xelval maxval;
    struct cmdline_info cmdline; 
    
    pnm_init(&argc, argv);

    parse_command_line(argc, argv, &cmdline);

    ifp = pm_openr(cmdline.input_filespec);

    pnm_readpnminit(ifp, &cols, &rows, &maxval, &format);

    if (cols < 2)
        pm_error("Image is too narrow.  Must be at least 2 columns.");
    if (rows < 2)
        pm_error("Image is too short.  Must be at least 2 lines.");

    if (PNM_FORMAT_TYPE(format) == PBM_TYPE) {
        output_format = PGM_TYPE;
        /* usual filter message when reading PBM but writing PGM: */
        pm_message("pnminterp: promoting from PBM to PGM");
    } else {
        output_format = PPM_TYPE;
    }
    {
        const int output_cols = 
            (cols - (cmdline.edge_mode == EDGE_DROP ? 1 : 0)) * cmdline.scale;
        const int output_rows = 
            (rows - (cmdline.edge_mode == EDGE_DROP ? 1 : 0)) * cmdline.scale;

        pnm_writepnminit(stdout, output_cols, output_rows,
                         maxval, output_format, 0);
    }

    interp(ifp,cols,rows,maxval,format,output_format,cmdline.scale,
           cmdline.edge_mode);

    pm_close(ifp);

    exit(0);
}



static void interp(FILE *in,const int cols,const int rows, 
                   const xelval maxval,const int format,
                   const int output_format,
                   const int scale,const enum an_edge_mode edge_mode) 
{
  xel *linebuf1, *linebuf2;  /* Input buffers for two ros at a time */
  xel *curline, *nextline;   /* Pointers to one of the two above buffers */
  xel *outbuf;   /* One-row output buffer */
  int row;

  linebuf1=pnm_allocrow(cols);
  linebuf2=pnm_allocrow(cols);
  outbuf=pnm_allocrow(cols*scale);

  curline=linebuf1;
  nextline=linebuf2;
  pnm_readpnmrow(in,curline,cols,maxval,format);

  for(row=1;row<=rows;row++)
  {
    if(row==rows)
      {
      /* last line is about to be output. there is no further `next line'.
       * if EDGE_DROP, we stop here, with output of rows-1 rows.
       * if EDGE_INTERP_TO_BLACK we make next line black.
       * if EDGE_NON_INTERP (default) we make it a copy of the current line.
       */
      if(edge_mode==EDGE_DROP)
        break;
      
      if(edge_mode==EDGE_INTERP_TO_BLACK) 
        {
        int col;
        for (col=0; col < cols; col++)
          PNM_ASSIGN1(nextline[col], 0);
        } 
      else 
        { /* EDGE_NON_INTERP */
        int col;
        for (col=0; col < cols; col++)
          nextline[col] = curline[col];
        }
      }
    else
      {
      pnm_readpnmrow(in,nextline,cols,maxval,format);
      }
    
    /* interpolate curline towards nextline into outbuf */
    write_interp_line(curline,nextline,outbuf,cols,maxval,format,
                      scale,edge_mode);
    {
      /* Advance "next" line to "current" line by switching line buffers */
      xel *tmp;
      tmp=curline;
      curline=nextline;
      nextline=tmp;
    }
  }
  pnm_freerow(outbuf);
  pnm_freerow(linebuf2);
  pnm_freerow(linebuf1);
}



void write_interp_line(const xel * const curline,
                       const xel * const nextline, 
                       xel * const outbuf,
                       const int cols,const xelval maxval,const int format,
                       const int scale,const int edge_mode)
{
int row;
int a1,a2,a3,a4;
int scaleincr=0,subpix_xpos,subpix_ypos,sxmulsiz,symulsiz,simulsiz=0;
int sisize=0,sis2=0;
int swidth=cols*scale;

if(edge_mode==EDGE_DROP) swidth-=scale;

sisize=0;
while(sisize<256) sisize+=scale;
scaleincr=sisize/scale;
simulsiz=scaleincr*sisize;
sis2=sisize*sisize;

for(row=0;row<scale;row++)
  {
  int col, outcol;

  col=0;
  
  subpix_ypos=(row%scale)*scaleincr;
  subpix_xpos=0;
  
  symulsiz=sisize*subpix_ypos;
  sxmulsiz=sisize*subpix_xpos;
  
  for(outcol=0;outcol<swidth;outcol++)
    {
    a3=symulsiz-(a4=subpix_xpos*subpix_ypos);
    a2=sxmulsiz-a4;
    a1=sis2-sxmulsiz-symulsiz+a4;

    if (col >= cols-1) 
      {
        /* We're at the edge -- nothing to the right of us. */
          if(edge_mode==EDGE_INTERP_TO_BLACK)
              /* assume column to the right of us is black. */
              PPM_ASSIGN(outbuf[outcol], 
                         (PPM_GETR(curline[col])*a1+
                          PPM_GETR(nextline[col])*a3)/sis2,
                         (PPM_GETG(curline[col])*a1+
                          PPM_GETG(nextline[col])*a3)/sis2,
                         (PPM_GETB(curline[col])*a1+
                          PPM_GETB(nextline[col])*a3)/sis2
                  );
          else	/* EDGE_NON_INTERP */
              /* assume column to the right of us is the same as this one */
              PPM_ASSIGN(outbuf[outcol], 
                         (PPM_GETR(curline[col])*(a1+a2)+
                          PPM_GETR(nextline[col])*(a3+a4))/sis2,
                         (PPM_GETG(curline[col])*(a1+a2)+
                          PPM_GETG(nextline[col])*(a3+a4))/sis2,
                         (PPM_GETB(curline[col])*(a1+a2)+
                          PPM_GETB(nextline[col])*(a3+a4))/sis2
                  );
      } else {
          /* Use the column to the right (col+1) in the interpolation */

          PPM_ASSIGN(outbuf[outcol], 
                     (PPM_GETR(curline[col])*a1+
                      PPM_GETR(curline[col+1])*a2+
                      PPM_GETR(nextline[col])*a3+
                      PPM_GETR(nextline[col+1])*a4)/sis2,
                     (PPM_GETG(curline[col])*a1+
                      PPM_GETG(curline[col+1])*a2+
                      PPM_GETG(nextline[col])*a3+
                      PPM_GETG(nextline[col+1])*a4)/sis2,
                     (PPM_GETB(curline[col])*a1+
                      PPM_GETB(curline[col+1])*a2+
                      PPM_GETB(nextline[col])*a3+
                      PPM_GETB(nextline[col+1])*a4)/sis2
              );
      }
    subpix_xpos+=scaleincr;
    sxmulsiz+=simulsiz;
    if(subpix_xpos>=sisize)
      {
      subpix_xpos=sxmulsiz=0;
      col++;
      }
    }
  pnm_writepnmrow(stdout,outbuf,swidth,maxval,format,0);
  }
}
