/* iface.c -- pseudo-interface handling.

   Copyright (C) 2008 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dnet.h>

#include "missing.h"
#include "usr.h"
#include "usr-priv.h"

static int
iface_handler(const struct intf_entry *entry, void *arg)
{
    struct iftable *iftable;
    struct if_entry *i;

    iftable = arg;

    iftable->nentries++;

    iftable->entries = realloc(iftable->entries,
			       iftable->nentries*sizeof(*iftable->entries) );
    if (iftable->entries == NULL)
	return -1;

    i = &iftable->entries[iftable->nentries - 1];

    /*
     * Copy libdnet interface structure into our own (USR) interface
     * structure.
     */
    strlcpy(i->intf_name, entry->intf_name, sizeof i->intf_name);
    i->intf_type = entry->intf_type;
    i->intf_flags = entry->intf_flags;
    i->intf_mtu = entry->intf_mtu;
    i->intf_addr = entry->intf_addr;
    i->intf_dst_addr = entry->intf_dst_addr;
    i->intf_link_addr = entry->intf_link_addr;

    return 0;
}

/*
 * Synchronizes an interface table with the system's interface table.
 */
static int
iftable_sync(struct iftable *iftable)
{
    intf_t *i;

    i = intf_open();
    if (i == NULL)
	return -1;

    /* Start fresh */
    iftable->nentries = 0;
    free(iftable->entries);
    iftable->entries = NULL;

    /*
     * Read system interface table. All work is done by the callback
     * function iface_handler()
     */
    if (intf_loop(i, iface_handler, iftable) == -1) {
	intf_close(i);
	return -1;
    }

    intf_close(i);

    return 0;
}

/*
 * Creates an interface table. "sync" indicates that the interface table
 * needs to be initially synchronized with the system interfaces.
 */
struct iftable *
iftable_new(int sync)
{
    struct iftable *iftable;

    iftable = (struct iftable *) malloc(sizeof(struct rtable) );
    if (iftable == NULL)
	return NULL;

    iftable->nentries = 0;
    iftable->entries = NULL;

    if (sync)
	if (iftable_sync(iftable) == -1) {
	    free(iftable);
	    return NULL;
	}

    return iftable;
}

/*
 * Destroys an interface table.
 */
void
iftable_destroy(struct iftable *iftable)
{
    free(iftable->entries);
    free(iftable);
}

/*
 * Prints an interface table. If the pointer to the interface table is NULL
 * then print the system interface table.
 */
void
iftable_print(struct iftable *iftable)
{
    int n;
    struct if_entry *i;
    char flags[sizeof "UP LOOPBACK POINTTOPOINT NOARP BROADCAST MULTICAST "];
    char *iftype;
    char addr[20]; /* Has to be 20 or addr_ntop() from libdnet will not work */

    if (iftable == NULL)
	iftable = default_iftable;

    for (n = 0; n < iftable->nentries; n++) {
	flags[0] = '\0';

	i = &iftable->entries[n];

	if (i->intf_flags & INTF_FLAG_UP)
	    strlcat(flags, "UP,", sizeof flags);

	if (i->intf_flags & INTF_FLAG_LOOPBACK)
	    strlcat(flags, "LOOPBACK,", sizeof flags);

	if (i->intf_flags & INTF_FLAG_POINTOPOINT)
	    strlcat(flags, "POINTTOPOINT,", sizeof flags);

	if (i->intf_flags & INTF_FLAG_NOARP)
	    strlcat(flags, "NOARP,", sizeof flags);

	if (i->intf_flags & INTF_FLAG_BROADCAST)
	    strlcat(flags, "BROADCAST,", sizeof flags);

	if (i->intf_flags & INTF_FLAG_MULTICAST)
	    strlcat(flags, "MULTICAST,", sizeof flags);

	if (flags[0] != '\0')
	    /* Remove extra character at the end */
	    flags[strlen(flags) - 1] = '\0';

	printf("%d: %s: <%s> mtu %d\n", n + 1, i->intf_name,
	       flags, i->intf_mtu);

	addr[0] = '\0';

	if (i->intf_type == INTF_TYPE_OTHER)
	    iftype = "other";
	else if (i->intf_type == INTF_TYPE_ETH) {
	    iftype = "ethernet";
	    addr_ntop(&i->intf_link_addr, addr, sizeof addr);
	} else if (i->intf_type == INTF_TYPE_LOOPBACK)
	    iftype = "loopback";
	else if (i->intf_type == INTF_TYPE_TUN)
	    iftype = "virtual/internal";
	else
	    /* Shouldn't happen */
	    iftype = "other";

	printf("    link/%s %s\n", iftype, addr);

	if (i->intf_addr.addr_type == ADDR_TYPE_IP) {
	    addr_ntop(&i->intf_addr, addr, sizeof addr);
	    printf("    inet %s\n", addr);
	}
    }
}

