/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#ifdef LINUX
#define __FAVOR_BSD
#endif
#include <sys/socket.h>
#include <arpa/inet.h>

#include <gtk/gtk.h>
#include <callbacks.h>
#include <netdude/nd.h>
#include <netdude/nd_globals.h>
#include <netdude/nd_dialog.h>
#include <netdude/nd_gui.h>
#include <netdude/nd_misc.h>
#include <netdude/nd_packet.h>
#include <netdude/nd_trace.h>
#include <netdude/nd_tcpdump.h>
#include <netdude/nd_protocol_registry.h>
#include <netdude/nd_protocol_inst.h>
#include <netdude/nd_raw_protocol.h>
#include <support.h>


ND_Packet  *
nd_packet_new(ND_Trace *trace)
{
  ND_Packet *p;

  p = (ND_Packet*) g_new0(ND_Packet, 1);

  D_ASSERT_PTR(p);
  if (!p)
    return NULL;

  p->trace = trace;

  return (p);
}


void
nd_packet_pcap_read_handler(u_char *data, const struct pcap_pkthdr *h,
			    const u_char *pdata)
{
  ND_Trace  *trace;
  ND_Packet *packet;


  trace = (ND_Trace *) data;

  packet = nd_packet_new(trace);
  packet->ph = *h;

  packet->data = g_malloc(h->caplen);
  memcpy(packet->data, pdata, h->caplen);

  if (trace->pl_end)
    {
      trace->pl_end->next = packet;
      packet->prev = trace->pl_end;
      trace->pl_end = packet;
    }
  else
    {
      trace->pl = trace->pl_end = packet;
    }

  trace->num_packets++;

  while (gtk_events_pending())
    gtk_main_iteration();
}


void
nd_packet_pcap_read_handler_gui(u_char *data, const struct pcap_pkthdr *h,
				const u_char *pdata)
{
  ND_Trace  *trace;
  char       line[MAXPATHLEN];
  char      *s[2];

  nd_packet_pcap_read_handler(data, h, pdata);
  trace = (ND_Trace *) data;

  /* Do not look up TCP context, as it'll be
   * correct during the first scan anyway!
   */
  nd_tcpdump_get_packet_line(trace->pl_end, line, FALSE);
  
  s[0] = line;
  s[1] = "";
  gtk_clist_append(GTK_CLIST(trace->list), s);

  if (nd_packet_is_complete(trace->pl_end))
    nd_gui_list_incomplete_row_set(trace->list, nd_trace_size(trace) - 1, FALSE);
  else
    nd_gui_list_incomplete_row_set(trace->list, nd_trace_size(trace) - 1, TRUE);

  while (gtk_events_pending())
    gtk_main_iteration();

  return;
  TOUCH(data);
}


static void
packet_free_proto_data(gpointer data, gpointer user_data)
{
  ND_ProtoData *pd = (ND_ProtoData *) data;

  if (!pd)
    return;
  
  nd_proto_data_free(pd);
  
  return;
  TOUCH(user_data);
}


static void
packet_clear_proto_flag(gpointer data, gpointer user_data)
{
  ND_ProtoData *pd   = (ND_ProtoData *) data;
  ND_Packet *packet  = (ND_Packet *) user_data;

  if (!pd || !packet)
    return;

  packet->protocols &= ~(pd->inst.proto->id);
}


void        
nd_packet_free(ND_Packet *p)
{
  if (!p)
    return;

  g_list_foreach(p->pd, packet_free_proto_data, p->trace);
  g_list_free(p->pd);
  g_free(p->data);
  g_free(p);
}


void
nd_packet_delete(ND_Packet *p, gboolean gui_update)
{
  if (!p)
    return;

  if (!p->trace)
    {
      nd_packet_free(p);
      return;
    }

  if (gui_update)
    nd_gui_list_remove_row(p->trace->list, nd_packet_get_index(p));
  
  if (p->next)
    {
      if (p->prev)
	{
	  /* It's a normal packet (not first or last) --> just cut it out */
	  p->prev->next = p->next;
	  p->next->prev = p->prev;
	}
      else
	{
	  /* It's got a next packet, but no previous one --> first packet */
	  if (p->trace)
	    p->trace->pl = p->next;

	  p->next->prev = NULL;
	}
    }
  else if (p->prev)
    {
      /* It's got no next one, but a previous one --> last packet */
      p->prev->next = NULL;

      if (p->trace)
	p->trace->pl_end = p->prev;
    }
  else if (p->trace)
    {
      /* It's the only packet in the trace */
      p->trace->pl = NULL;
      p->trace->pl_end = NULL;
    }


  /* This packet may be part of the selection, update accordingly: */
  
  if (p->sel_next || p->sel_prev)
    {
      if (p->trace)
	p->trace->sel.size--;

      p->trace->sel.last_valid = FALSE;

      /* Same pattern as above: */

      if (p->sel_next)
	{
	  if (p->sel_prev)
	    {
	      /* selected before and after -- just cut it out */
	      p->sel_prev->sel_next = p->sel_next;
	      p->sel_next->sel_prev = p->sel_prev;
	    }
	  else
	    {
	      /* No selections before this one -- new first selected */
	      if (p->trace)
		p->trace->sel.sel = p->sel_next;
	      
	      p->sel_next->sel_prev = NULL;
	    }
	}
      else if (p->sel_prev)
	{
	  p->sel_prev->sel_next = NULL;
	}
      else if (p->trace)
	{
	  p->trace->sel.sel = NULL;
	}
    }

  p->trace->num_packets--;

  if (p->trace)
    {
      nd_trace_set_dirty(p->trace, TRUE);
      
      if (p->trace->cur_packet == p)
	nd_trace_set_current_packet(p->trace, NULL);
    }
  
  nd_packet_free(p);
}


ND_Packet*
nd_packet_duplicate(ND_Packet *p)
{
  ND_Packet       *copy;
  ND_ProtoData    *pd_old, *pd_new;
  GList           *l;

  if (!p)
    return (NULL);

  copy = nd_packet_new(p->trace);

  copy->ph          = p->ph;
  copy->protocols   = p->protocols;
  copy->is_hidden   = 0;

  /* Avoid pointer havoc -- this packet does not at first
     belong to any packet lists! */
  copy->prev = copy->next = copy->sel_prev = copy->sel_next = NULL;

  /* Copy the packet data and hook it into the right place */
  
  if (! (copy->data = g_malloc(sizeof(guchar) * p->ph.caplen)))
    {
      g_free(copy);
      return NULL;
    }
  
  memcpy(copy->data, p->data, p->ph.caplen);

  /* Now make sure the offsets of the upper layers are correct. */
  for (l = p->pd; l; l = g_list_next(l))
    {
      pd_old = (ND_ProtoData*) l->data;
      pd_new =
	nd_proto_data_new(pd_old->inst.proto, pd_old->inst.nesting,
			  copy->data + (pd_old->data - p->data),
			  copy->data + (pd_old->data_end - p->data));
      
      copy->pd = g_list_append(copy->pd, pd_new);
    }
  
  return copy;
}


static void        
packet_init(ND_Packet *p, pcap_t * pcap)
{
  int type;
  ND_Protocol *proto;

  if (!p)
    return;

  /* Clear the index of contained protocols: */
  p->protocols = 0;

  /* Remove any existing offsets */

  if (p->pd)
    {
      g_list_foreach(p->pd, packet_free_proto_data, p->trace);
      g_list_free(p->pd);
      p->pd = NULL;
    }

  /* Check what we have at the link layer and handle things off
     to the according protocol. The rest is up to them.
  */
  type = pcap_datalink(pcap);

  switch (type)
    {      
    case DLT_NULL:
      /* We'll assume it's IP for now */
    case DLT_RAW:
      proto = nd_proto_registry_find(ND_PROTO_LAYER_NET, 0x0800);
      D(("Found protocol %s\n", proto->name));
      D_ASSERT_PTR(proto);
      if (proto)
	proto->init_packet(p, p->data, nd_packet_get_end(p));
      break;
      
    default:
      proto = nd_proto_registry_find(ND_PROTO_LAYER_LINK, type);
      D_ASSERT_PTR(proto);
      D(("Found protocol %s\n", proto->name));
      
      if (proto)
	proto->init_packet(p, p->data, nd_packet_get_end(p));
    }
}


void
nd_packet_init(ND_Packet *p)
{
  if (!p || !p->trace)
    return;

  packet_init(p, p->trace->pcap);
}


void
nd_packet_init_from_pcap(ND_Packet *p, pcap_t *pcap)
{
  if (!p || !pcap)
    return;

  packet_init(p, pcap);
}


void            
nd_packet_update(ND_Packet *packet, ND_Protocol *proto, guint nesting)
{
  GList            *l, *l2;
  ND_ProtoData     *pd;

  D_ENTER;

  if (!packet)
    D_RETURN;

  if (!proto)
    {
      nd_packet_init(packet);
      D_RETURN;
    }

  for (l = packet->pd; l; l = g_list_next(l))
    {
      pd = (ND_ProtoData *) l->data;
    
      if (pd->inst.proto == proto && pd->inst.nesting == nesting)
	break;
    }
  
  if (!l)
    {
      D(("WARNING -- header to update from not found in packet.\n"));
      D_RETURN;
    }

  /* Split off the rest of the previously recognized protocols: */

  if (l == packet->pd)
    {
      /* It is the first header. We can just as well reinitialize
	 the whole thing. */
      nd_packet_init(packet);
      D_RETURN;
    }

  pd = (ND_ProtoData *) l->data;

  /* If there's a previous item, make sure the list ends there */
  
  if ( (l2 = g_list_previous(l)))
    l2->next = NULL;
  
  /* Clear protocol flags of all protocols beyond that point */
  g_list_foreach(l, packet_clear_proto_flag, packet);

  /* Make sure the current data doesn't get
     deleted, and clean up the rest. */
  l->data = NULL;
  g_list_foreach(l, packet_free_proto_data, packet->trace);
  g_list_free(l);

  D(("Updating packet from %s/%i onward\n",
     pd->inst.proto->name, pd->inst.nesting));

  /* Re-initialize packet from this protocol on: */
  pd->inst.proto->init_packet(packet, pd->data, pd->data_end);

  /* Finally, clean up the single chunk we have excluded above */
  nd_proto_data_free(pd);

  D_RETURN;
}


void    
nd_packet_set_gui(const ND_Packet *packet)
{
  GtkWidget     *page;
  GtkNotebook   *notebook;
  ND_ProtoData  *pd;
  ND_ProtoInfo  *pinf;
  int            i;
  GList         *l;
  gboolean       has_sel_proto = FALSE;

  if (!packet)
    return;

  notebook = nd_trace_get_notebook(packet->trace);
  D_ASSERT_PTR(notebook);

  /* Show and hide the protocol tabs for this packet,
     and while doing this, also gray out the protocol menus
     that aren't applicable for this packet:
  */
  page = gtk_notebook_get_nth_page(notebook, 0);
  for (i = 0; gtk_notebook_get_nth_page(notebook, i); i++)
    {
      page = gtk_notebook_get_nth_page(notebook, i);
      D_ASSERT_PTR(page);

      pinf = gtk_object_get_data(GTK_OBJECT(page), "pinf");
      D_ASSERT_PTR(pinf);

      if (nd_packet_has_proto_nested(packet, pinf->inst.proto, pinf->inst.nesting))
	{
	  gtk_widget_show(page);
	  if (pinf->inst.proto->proto_menu_item)
	    gtk_widget_set_sensitive(pinf->inst.proto->proto_menu_item, TRUE);
	}
      else
	{
	  gtk_widget_hide(page);
	  if (pinf->inst.proto->proto_menu_item)
	    gtk_widget_set_sensitive(pinf->inst.proto->proto_menu_item, FALSE);
	}
    }

  if (packet->trace->cur_pi_sel)
    {
      has_sel_proto =
	nd_packet_has_proto_nested(packet,
				   packet->trace->cur_pi_sel->proto,
				   packet->trace->cur_pi_sel->nesting);
    }

  /* Now, update each of the packet's protocol header's contents
     in the GUI: */
  for (l = packet->pd, i = 0; l; l = g_list_next(l), i++)
    {
      pd = (ND_ProtoData *) l->data;
      
      D_ASSERT_PTR(packet);
      D_ASSERT_PTR(packet->trace);
      
      if (i == 0 && !packet->trace->cur_pi_sel)
	nd_trace_set_current_proto_selection(packet->trace, &pd->inst);
      
      /* only update the gui for the first protocol, and only if it
	 does not have the selected protocol (which we set below)
	 anyway. Kills a lot of flickering :)
      */
      nd_trace_set_current_proto(packet->trace, &pd->inst, (i == 0 && !has_sel_proto));
      
      pinf = nd_trace_get_proto_info(packet->trace, pd->inst.proto, pd->inst.nesting);
      D_ASSERT_PTR(pinf);

      nd_gui_proto_table_block_events(packet->trace, pinf);
      pd->inst.proto->set_gui(packet, pinf);
      nd_gui_proto_table_unblock_events(packet->trace, pinf);

      /* sort the tabs so that the GUI reflects the sequence in
	 the packet: */

      gtk_notebook_reorder_child(notebook, pinf->proto_tab, i);
    }

  if (packet->trace->cur_pi_sel)
    nd_trace_set_current_proto(packet->trace, packet->trace->cur_pi_sel, TRUE);

  nd_gui_list_update_packet(packet);    
}


guchar *
nd_packet_get_data(const ND_Packet *packet,
		   const ND_Protocol *proto,
		   guint nesting)
{
  ND_ProtoData *pd;
  GList *l;

  if (!packet || !proto || !nd_packet_has_proto(packet, proto))
    return NULL;

  for (l = packet->pd; l; l = g_list_next(l))
    {
      pd = (ND_ProtoData *) l->data;
      
      if (pd->inst.proto->id == proto->id &&
	  pd->inst.nesting == nesting)
	return pd->data;
    }

  return NULL;
}


guchar         *
nd_packet_get_data_end(const ND_Packet *packet,
		       const ND_Protocol *proto,
		       guint nesting)
{
  ND_ProtoData *pd;
  GList *l;

  if (!packet || !proto || !nd_packet_has_proto(packet, proto))
    return NULL;

  for (l = packet->pd; l; l = g_list_next(l))
    {
      pd = (ND_ProtoData*) l->data;

      if (pd->inst.proto->id == proto->id &&
	  pd->inst.nesting == nesting)
	return pd->data_end;
    }
  
  return NULL;
}


void
nd_packet_add_proto_data(ND_Packet *packet, ND_Protocol *proto,
			 guchar *data, guchar *data_end)
{
  guchar *real_end;
  ND_ProtoData *pd;
  guint         nesting = 0;
  GList        *l;

  if (!packet || !proto)
    return;

  for (l = packet->pd; l; l = g_list_next(l))
    {
      pd = (ND_ProtoData *) l->data;

      if (pd->inst.proto == proto)
	nesting++;
      
      if (g_list_next(l) == NULL)
	break;
    }

  /* l is now last list item */

  real_end = nd_packet_get_end(packet);

  if (real_end >= data_end)
    pd = nd_proto_data_new(proto, nesting, data, data_end);
  else
    pd = nd_proto_data_new(proto, nesting, data, real_end);

  D_ASSERT_PTR(pd);
  if (!pd)
    return;

  if (nesting > 0)
    {
      /* We are nesting this protocol, so we must make sure
	 that the trace's protocol notebook has enough copies
	 of this protocol's tab.
      */
      
      D(("Nesting protocol %s, level %i\n", proto->name, nesting));
      if (! nd_trace_get_proto_info(packet->trace, proto, nesting))
	nd_trace_add_proto_tab(packet->trace, proto, nesting);
    }

  packet->pd = g_list_append(packet->pd, pd);
  packet->protocols |= proto->id;

  D(("Added proto %s at offset %u, nesting is %i\n",
     proto->name, data - packet->data, pd->inst.nesting));
}


ND_ProtoData   *
nd_packet_get_proto_data(const ND_Packet *packet,
			 const ND_Protocol *proto,
			 guint nesting)
{
  GList *l;
  ND_ProtoData *pd;

  for (l = packet->pd; l; l = g_list_next(l))
    {
      pd = (ND_ProtoData *) l->data;

      if (pd->inst.proto == proto &&
	  pd->inst.nesting == nesting)

	return pd;
    }

  return NULL;
}


guchar         *
nd_packet_get_end(const ND_Packet *packet)
{
  if (!packet)
    return (NULL);

  return (packet->data + packet->ph.caplen);
}


gboolean    
nd_packet_has_proto(const ND_Packet *packet,
		    const ND_Protocol *proto)
{
  if (!packet || !proto)
    return FALSE;

  return ((packet->protocols & proto->id) > 0);
}


gboolean    
nd_packet_has_proto_nested(const ND_Packet *packet,
			   const ND_Protocol *proto,
			   guint nesting)
{
  ND_ProtoData *pd;
  GList *l;

  if (!packet || !proto)
    return FALSE;

  for (l = packet->pd; l; l = g_list_next(l))
    {
      pd = (ND_ProtoData *) l->data;
      
      if (pd->inst.proto == proto &&
	  pd->inst.nesting == nesting)
	return TRUE;
    }

  return FALSE;
}


gboolean
nd_packet_has_complete_header(const ND_Packet *p,
			      const ND_Protocol *proto,
			      guint nesting)
{
  if (!p || !proto)
    return (FALSE);

  if (!nd_packet_has_proto(p, proto))
    return (FALSE);

  return (proto->header_complete(p, nesting));
}


gboolean        
nd_packet_is_complete(const ND_Packet *packet)
{
  if (!packet)
    return FALSE;

  return (packet->ph.caplen == packet->ph.len);
}


void
nd_packet_update_proto_state(ND_Packet *packet, int index)
{
  GList *l;
  ND_ProtoData *pd;
  
  for (l = packet->pd; l; l = g_list_next(l))
    {
      pd = (ND_ProtoData *) l->data;

      if (pd->inst.proto->is_stateful)
	pd->inst.proto->update_state(packet, index);
    }
}


void            
nd_packet_foreach_proto(ND_Packet *packet,
			ND_PacketFunc callback,
			void *user_data)
{
  ND_ProtoData *pd;
  GList *l;

  if (!packet || !callback)
    return;

  for (l = packet->pd; l; l = g_list_next(l))
    {
      pd = (ND_ProtoData *) l->data;

      callback(packet, pd, user_data);
    }
}


void            
nd_packet_foreach_proto_backward(ND_Packet *packet,
				 ND_PacketFunc callback,
				 void *user_data)
{
  ND_ProtoData *pd;
  GList *l;

  if (!packet || !callback)
    return;

  for (l = g_list_last(packet->pd); l; l = g_list_previous(l))
    {
      pd = (ND_ProtoData *) l->data;

      callback(packet, pd, user_data);
    }
}


void            
nd_packet_modified(ND_Packet *packet)
{
  if (!packet)
    return;

  nd_gui_list_update_packet(packet);

  if (packet == nd_trace_get_current_packet(packet->trace))
    nd_packet_set_gui(packet);

  nd_trace_set_dirty(packet->trace, TRUE); 
}


void            
nd_packet_modified_at_index(ND_Packet *packet, int index)
{
  if (!packet)
    return;

  if (index < 0)
    {
      nd_packet_modified(packet);
      return;
    }

  nd_gui_list_update_packet_at_index(packet, index);

  if (packet == nd_trace_get_current_packet(packet->trace))
    nd_packet_set_gui(packet);
  
  nd_trace_set_dirty(packet->trace, TRUE); 
}


int         
nd_packet_get_index(const ND_Packet *needle)
{
  ND_Packet *p;
  int        i = 0;

  if (!needle)
    return -1;

  if (!needle->trace)
    return -1;

  p = needle->trace->pl;
  while (p)
    {
      if (p == needle)
	return i;

      i++;
      p = p->next;
    }

  D(("Packet lookup failed!"));

  return -1;
}


int             
nd_packet_get_proto_nesting(const ND_Packet *packet,
			    const ND_Protocol *proto,
			    guchar *data)
{
  GList        *l;
  ND_ProtoData *pd = NULL;

  if (!packet || !proto || !data ||
      (data < packet->data) ||
      (data > packet->data + packet->ph.caplen))
    {
      D(("Warning -- error in input, returning -1\n"));
      return -1;
    }

  for (l = packet->pd; l; l = g_list_next(l))
    {
      pd = (ND_ProtoData *) l->data;

      if (pd->data > data && g_list_previous(l))
	{
	  pd = (ND_ProtoData *) g_list_previous(l)->data;
	  /* The nesting we are looking for is in the 
	     previous proto data */

	  D(("Proto nesting for %s: %i\n",
	     proto->name, pd->inst.nesting));
	  
	  return pd->inst.nesting;
	}
    }

  /* If we get here, the data pointer may be pointing into the innermost
     protocol's data, but not beyond the end of the packet data.
     Check that case.
  */

  if (pd && data < packet->data + packet->ph.caplen)
    return pd->inst.nesting;
    
  D(("Proto nesting for %s: NOT FOUND\n", proto->name));
  return -1;
}
