using Nemerle.Compiler;
using Nemerle.Completion;
using Nemerle.Collections;
using System.IO;
using System.Console;

def engine = Engine ();

def files = Hashtable ();
def rawArgs = List.FromArray (System.Environment.GetCommandLineArgs ());
def (progName, args) = rawArgs.Head::rawArgs.Tail;
when (args.Length == 0) {
	WriteLine($"usage: $progName <filename>\n"+
		       "       <filename> is the Nemerle source code you want "+
	           "to pass to the \n"+
	           "       code completion engine");
	System.Environment.Exit(1);
}
foreach (arg in args) {
  using (sr = StreamReader (arg)) {
    def str = sr.ReadToEnd ();
    engine.Sources.Add (arg, str);
    files [arg] = str;
  }
}

def cut_by_location (file_contents, loc) {
  def find_line_col (file_contents : string, line, col) {
    def loop (cur_line = 1, i = 0) {
      if (i >= file_contents.Length)
        -1
      else if (cur_line == line)
        i + col - 1
      else if (file_contents [i] == '\n')
        loop (cur_line + 1, i + 1)
      else 
        loop (cur_line, i + 1)
    }
    loop ()
  }

  def beg = find_line_col (file_contents, loc.Line, loc.Column);
  def end = find_line_col (file_contents, loc.EndLine, loc.EndColumn);
  if (beg < 0 || end < 0) null
  else
    file_contents.Substring (beg, end - beg)
}

def type_tree = engine.GetTypeTree ();

mutable my_method = null;

foreach (t in type_tree.Types) {
  System.Console.WriteLine (t.Name);
  foreach (m in t.Methods) {
    when (m.Name == "Bar")
      my_method = m._member :> MethodBuilder;
    def mem = m._member :> MemberBuilder;
    def loc = mem.BodyLocation;
    if (files [loc.File] == null)
      WriteLine ($ "null file $(loc.File)")
    else {
      def body = cut_by_location (files [loc.File], loc)
                 .Replace ("\n", " ")
                 .Replace ("\r", "");
                 
      when (loc != Location.Default && loc.Line != loc.EndLine &&
            (body [0] != '{' || body [body.Length - 1] != '}'))
        WriteLine ($ "problem $(mem) $loc '$(body)'");
    
      System.Console.WriteLine ($ "  $(mem) $loc '$(body)'");
    }
  }
}

def try_completion (body) {
  System.Console.WriteLine(body);
  def env = my_method.DeclaringType.GlobalEnv;
  def lexer = LexerCompletion (body + " ", body.Length);
  my_method.GetHeader ().body =
    FunBody.Parsed (MainParser.ParseExpr (env, lexer));

  try {
    my_method.RunBodyTyper ()
  } catch {
    | e is CompletionResult =>
      System.Console.WriteLine ($ "$body --> $(e.NamePrefix) --> $(e.Overloads)");
    | e =>
      System.Console.WriteLine (e);
  }
}

try_completion ("string.For");
try_completion ("this.Fo");
try_completion ("System.Collections.Generic.Q");
try_completion ("Q");
try_completion ("def a = \"a\"; a.");
