/* lang.h: types and functions involved in parsing and evaluating nawmrc */

/* Copyright (C) 1999 by the Massachusetts Institute of Technology.
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting
 * documentation, and that the name of M.I.T. not be used in
 * advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 * M.I.T. makes no representations about the suitability of
 * this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 */

typedef long nawmval;

typedef nawmval dtype;
#define T_INT 1
#define T_WIN 2
#define T_STR 3
#define T_ARRAY 4

#define is_simple_type(t) (t < 4)
#define is_atomic_type(t) (t < 3)

typedef struct _node {
  int type;
  dtype etype;
  struct _node *next;
  struct _node *vals[1];
} node;

typedef struct _variable {
  dtype type;
  int slot;
  nawmval data;
} variable;

typedef struct _function {
  void (*body)();
  dtype type; /* 0 for a command */
  long numvars, numargs;
  dtype vartype[3]; /* can actually be more or less */
} function;

typedef struct _bindlist {
  struct _bindlist *next;
  int type;
  node *cmds;
  union {
    char *name;
    struct {
      int mods;
      KeyCode kc;
    } key;
    struct {
      int mods, button;
    } button;
  } u;
} bindlist;

typedef struct _varbinding {
  char *name;
  int type;
  void *data;
  struct _varbinding *next;
} varbinding;

typedef struct _arrayiter {
  int ind;
  struct _achain *chain;
} arrayiter;

typedef struct _event_handler {
  void (*handler)(XEvent *);
  struct _event_handler *next;
} event_handler;

enum {PLUS, NOP, MINUS, NEGATE, TIMES, DIVIDE, REMAINDER, EQUAL, NOTEQUAL,
      LESSEQUAL, GREATEREQUAL, LESS, GREATER, ASSIGN, AND, OR, NOT, CONCAT,
      DECL, BODY};

/* prototypes from array.c */
struct _array;
typedef struct _array array;

void initarrays(void);
dtype array_type(dtype base, dtype sub);
dtype array_basetype(dtype atype);
dtype array_subtype(dtype atype);
char *array_typename(dtype atype);
array *create_array(dtype type, int nbuckets);
void free_array(array *arr);
int array_size(array *arr);
nawmval array_lookup(array *arr, nawmval subscript);
void array_insert(array *arr, nawmval subscript, nawmval value);
void array_delete(array *arr, nawmval subscript);
nawmval array_first(array *arr, arrayiter *ai);
nawmval array_next(array *arr, arrayiter *ai);
int array_contains(array *arr, nawmval value);

/* prototypes from bindings.c */
bindlist *mkbinding(int type, char *data, node *cmds);
void add_to_anymode(bindlist *binding);
void defmode(char *name, bindlist *bindings);
void do_bindings(bindlist *mode);
void undo_bindings(bindlist *mode);
void set_mode(char *mode);

/* prototypes from builtins.c */
void *search_builtins(char *name, int *type);
char *nameof_builtin(void *data);

/* prototypes from eval.c */
void initrunscopes(void);
int eval_cmds(node *);
void assign_var(variable *var, nawmval val);

/* prototypes from lexer.l */
void initlex(void);
void parse_file(char *name);
void parse_string(char *s);
void setupbuiltins(void);
void pushlexscope(int nestable);
void poplexscope(void);
int getscopenumvars(void);
dtype *getscopevartypes(void);

void register_bindings(varbinding *bindings);
void define(int type, char *name, void *data);
void *lookup(char *name, int *type, int norecurse);
char *nameof(void *data);
variable *mkvar(dtype);

/* prototypes from mem.c */
void initgc(void);
void *ref(void *);
void unref(void *);
void new_generation(void);
void end_generation(void);
void *gcmalloc(size_t size, void (*freer)(void *));
char *gcstrdup(const char *str);

/* prototypes from parser.y */
node *mknode(int, dtype, int, ...);
void parse_error(char *);

/* prototypes from dtype.c */
void inittypes(void);
int newtype(char *name);
char *typename(dtype type);
nawmval initial_value(dtype type);
