/*
 * NASPRO - The NASPRO Architecture for Sound Processing
 * Core library
 *
 * Copyright (C) 2007-2012 NASPRO core development team
 *
 * See the COPYING file for license conditions.
 */

#include "internal.h"

struct _nacore_msg_context
  {
	char is_msg_box;
	union
	  {
		struct
		  {
			nacore_msg_status_begin_cb	 status_begin_cb;
			nacore_msg_status_end_cb	 status_end_cb;
			nacore_msg_text_cb		 text_cb;
			void				*opaque;
		  } box;
		struct
		  {
			nacore_msg_context		 msg_box;
			nacore_msg_context		 parent;
			void				*opaque;
		  } status;
	  } u;
  };

_NACORE_DEF nacore_msg_context
nacore_msg_box_new(void *opaque)
{
	nacore_msg_context ret;

	ret = malloc(sizeof(struct _nacore_msg_context));
	if (ret == NULL)
		return NULL;

	ret->is_msg_box			= 1;
	ret->u.box.opaque		= opaque;
	ret->u.box.status_begin_cb	= NULL;
	ret->u.box.status_end_cb	= NULL;
	ret->u.box.text_cb		= NULL;

	return ret;
}

_NACORE_DEF void
nacore_msg_box_free(nacore_msg_context msg_box)
{
	free(msg_box);
}

_NACORE_DEF void
nacore_msg_box_set_callbacks(nacore_msg_context msg_box,
			     nacore_msg_status_begin_cb status_begin_cb,
			     nacore_msg_status_end_cb status_end_cb,
			     nacore_msg_text_cb text_cb)
{
	msg_box->u.box.status_begin_cb	= status_begin_cb;
	msg_box->u.box.status_end_cb	= status_end_cb;
	msg_box->u.box.text_cb		= text_cb;
}

_NACORE_DEF NACORE_FORMAT_PRINTF(3, 4) nacore_msg_context
nacore_msg_status_begin(nacore_msg_context context, void *opaque,
			const char *fmt, ...)
{
	nacore_msg_context ret;
	nacore_msg_context msg_box;
	char *text;
	va_list ap;

	if (context == NULL)
		return NULL;

	msg_box = context->is_msg_box ? context : context->u.status.msg_box;

	if (msg_box->u.box.status_begin_cb == NULL)
		return NULL;

	ret = malloc(sizeof(struct _nacore_msg_context));
	if (ret == NULL)
		return NULL;

	va_start(ap, fmt);
	nacore_vasprintf(&text, fmt, ap);
	va_end(ap);

	if (text == NULL)
	  {
		free(ret);
		return NULL;
	  }

	ret->is_msg_box		= 0;
	ret->u.status.msg_box	= msg_box;
	ret->u.status.parent	= context;
	ret->u.status.opaque	= opaque;

	msg_box->u.box.status_begin_cb(ret, text, opaque);

	free(text);

	return ret;
}

_NACORE_DEF void
nacore_msg_status_end(nacore_msg_context status_msg, nacore_msg_result result)
{
	if (status_msg == NULL)
		return;

	if (status_msg->u.status.msg_box->u.box.status_end_cb != NULL)
		status_msg->u.status.msg_box->u.box.status_end_cb(
			status_msg, result, status_msg->u.status.opaque);

	free(status_msg);
}

_NACORE_DEF NACORE_FORMAT_PRINTF(4, 5) void
nacore_msg_text(nacore_msg_context context, nacore_msg_severity severity,
		void *opaque, const char *fmt, ...)
{
	nacore_msg_context msg_box;
	char *text;
	va_list ap;

	if (context == NULL)
		return;

	msg_box = context->is_msg_box ? context : context->u.status.msg_box;

	if (msg_box->u.box.text_cb == NULL)
		return;

	va_start(ap, fmt);
	nacore_vasprintf(&text, fmt, ap);
	va_end(ap);

	if (text == NULL)
		return;

	msg_box->u.box.text_cb(context, severity, text, opaque);

	free(text);
}

_NACORE_DEF void *
nacore_msg_context_get_opaque(nacore_msg_context context)
{
	return context->is_msg_box ? context->u.box.opaque
				   : context->u.status.opaque;
}

_NACORE_DEF nacore_msg_context
nacore_msg_context_get_parent(nacore_msg_context context)
{
	return context->is_msg_box ? NULL : context->u.status.parent;
}
