--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Ada.Containers;
with Ada.Exceptions;
with Ada.Characters.Handling;

with Options;
with System_Calls;
with Types;
with Files_Handling;
with Uuid;

with McKae.XML.EZ_Out.String_Stream;
with Events_Receivers;

with Narval.Actors;
with Narval.Configurator.Compilation;
with Narval.Narval_Naming_Registry;
with Narval.Environment;

package body
  Narval.Configurator.Abstract_Actors_Coordination.Actors_Coordination is

   use Ada.Exceptions;
   use Ada.Strings.Unbounded;
   use Log4ada.Loggers;
   use Actors_Description;
   use type Ada.Containers.Count_Type;

   T_ACTION : constant String := "action";

   type Actors_Coordination_Parameter_Type is
     (Actors,
      Wait_Time,
      Wait_Kill_Time,
      Action,
      Prefix,
      List_Actions,
      Run_Number,
      Run_Numbering,
      State,
      Configuration_File,
      Configuration_Files,
      Name);

   procedure Clean_Config (Config : access Actor_Coordination_Type;
                           Fin_Normale : Boolean);
   procedure Clean_Config (Config : access Actor_Coordination_Type;
                           Fin_Normale : Boolean) is
      procedure Kill_Actor (Actor_Position : Actors_Description_Vector.Cursor);
      procedure Kill_Actor (Actor_Position : Actors_Description_Vector.Cursor)
      is
         Actor : constant Actor_Description_Access :=
           Actors_Description_Vector.Element (Actor_Position);
         Command : constant String := "ssh -x " &
           To_String (Actor.Host_Name) & " kill -9 `ssh -x " &
           To_String (Actor.Host_Name) & " ps -edf | grep -w " &
           To_String (Actor.Process_Name) & " | grep -v grep | grep -w " &
           To_String (Actor.Name) & " | grep -w " & To_String (Config.Name) &
           " | grep " & Uuid.Image (Actor.Uuid_Value) &
           " | awk '{print $2}'` >& /dev/null &";
      begin
         Info_Out (Config.Logger'Access,
                   "issuing command : " & Command);
         System_Calls.Exec_Command (Command);
         Uuid.Uuid_Clear (Actor.Uuid_Value);
      end Kill_Actor;
   begin
      if Config.Actors.Length /= 0 then
         Info_Out (Config.Logger'Access,
                   "deleting" &
                     Config.Actors.Length'Img &
                     " intermediary");
      end if;
      Actors_Description_Vector.Iterate (Config.Actors, Kill_Actor'Access);
      if not Fin_Normale then
         Compilation.Reset;
         Config.Slave_State_Machine.Put_In_Error;
      end if;
   end Clean_Config;

   procedure Add_Value
     (Config : access Actor_Coordination_Type;
      Parameter : Actors_Coordination_Parameter_Type;
      Xml_Buffer : in out McKae.XML.EZ_Out.String_Stream.String_Buffer);
   procedure Add_Value
     (Config : access Actor_Coordination_Type;
      Parameter_String : String;
      Xml_Buffer : in out McKae.XML.EZ_Out.String_Stream.String_Buffer);

   procedure Add_Value
     (Config : access Actor_Coordination_Type;
      Parameter_String : String;
      Xml_Buffer : in out McKae.XML.EZ_Out.String_Stream.String_Buffer) is
      Parameter : Actors_Coordination_Parameter_Type;
   begin
      begin
         Parameter :=
           Actors_Coordination_Parameter_Type'Value (Parameter_String);
      exception
         when others =>
            Error_Out (Config.Logger'Access,
                       Parameter_String & " isn't known");
            raise Unknown_Parameter;
      end;
      Add_Value (Config, Parameter, Xml_Buffer);
   end Add_Value;

   procedure Add_Value
     (Config : access Actor_Coordination_Type;
      Parameter : Actors_Coordination_Parameter_Type;
      Xml_Buffer : in out McKae.XML.EZ_Out.String_Stream.String_Buffer) is
      use McKae.XML.EZ_Out.String_Stream.String_Buffering;
      use McKae.XML.EZ_Out.String_Stream.XML_String_Buffer;
      Parameter_Name : constant String :=
        Ada.Characters.Handling.To_Lower (Parameter'Img);
      Pointer : Narval.Actors.Actor_Class_Access;
      procedure Null_Output;
      procedure Null_Output is
      begin
         Start_Element (Xml_Buffer, "data",
                        ("mode" = Read_Only'Img,
                         "monitor" = Never'Img,
                         "actors_number" = 0,
                         "type" = "string",
                         "name" = "actors"));
         Output_Element (Xml_Buffer, "value", "");
         End_Element (Xml_Buffer, "data");
      end Null_Output;
   begin
      case Parameter is
         when Actors =>
            if Config.Actors.Length = 0 or
              Config.Slave_State_Machine.Current_State = Configured then
               Null_Output;
               return;
            elsif Config.Actors.Length /= 0 then
               declare
                  Cursor : constant Actors_Description_Vector.Cursor :=
                    Config.Actors.First;
               begin
                  select
                     Actors_Description_Vector.Element
                       (Cursor).Task_Register.Get_Pointer (Pointer);
                  else
                     Null_Output;
                     return;
                  end select;
               end;
            end if;
            Start_Element (Xml_Buffer, "data",
                           ("mode" = Read_Only'Img,
                            "monitor" = Never'Img,
                            "actors_number" = Natural (Config.Actors.Length),
                            "type" = "string",
                            "name" = "actors"));
            declare
               Cursor : Actors_Description_Vector.Cursor;
               use type Actors_Description_Vector.Cursor;
            begin
               Cursor := Config.Actors.First;
               loop
                  exit when Cursor = Actors_Description_Vector.No_Element;
                  Output_Element (Xml_Buffer, "value",
                                  To_String (Actors_Description_Vector.Element
                                               (Cursor).Name));
                  Cursor := Actors_Description_Vector.Next (Cursor);
               end loop;
            end;
            End_Element (Xml_Buffer, "data");
            return;
         when Configuration_Files =>
            declare
               Files : constant Types.String_Array :=
                 Files_Handling.Get_Files_In
                 (Environment.Configuration_Directory.all,
                  "*.xml");
            begin
               Start_Element (Xml_Buffer, "data",
                              ("mode" = Read_Only'Img,
                               "monitor" = Never'Img,
                               "files_count" = Files'Length,
                               "type" = "string",
                               "name" = Parameter_Name));
               for I in Files'Range loop
                  Output_Element (Xml_Buffer, "value", To_String (Files (I)));
               end loop;
            end;
         when Prefix =>
            Start_Element (Xml_Buffer, "data",
                           ("mode" = Read_Write'Img,
                            "monitor" = Never'Img,
                            "type" = "string",
                            "name" = Parameter_Name));
            Output_Element (Xml_Buffer, "value", To_String (Config.Prefix));
         when Name =>
            Start_Element (Xml_Buffer, "data",
                           ("mode" = Read_Only'Img,
                            "monitor" = Never'Img,
                            "type" = "string",
                            "name" = Parameter_Name));
            Output_Element (Xml_Buffer, "value", To_String (Config.Name));
         when Configuration_File =>
            Start_Element (Xml_Buffer, "data",
                           ("mode" = Read_Write'Img,
                            "monitor" = Never'Img,
                            "type" = "string",
                            "name" = Parameter_Name));
            Output_Element (Xml_Buffer, "value",
                            To_String (Config.Configuration_File));
         when State =>
            Start_Element (Xml_Buffer, "data",
                           ("mode" = Read_Only'Img,
                            "monitor" = Request'Img,
                            "type" = "string",
                            "name" = Parameter_Name));
            Output_Element (Xml_Buffer, "value",
                            Config.Slave_State_Machine.Image);
         when Wait_Time =>
            Start_Element (Xml_Buffer, "data",
                           ("mode" = Read_Write'Img,
                            "monitor" = Request'Img,
                            "type" = "integer",
                            "size" = "32",
                            "name" = Parameter_Name));
            Output_Element (Xml_Buffer, "value",
                            Integer'Image (Config.Delay_To_Wait / 10));
         when Wait_Kill_Time =>
            Start_Element (Xml_Buffer, "data",
                           ("mode" = Read_Write'Img,
                            "monitor" = Request'Img,
                            "type" = "integer",
                            "size" = "32",
                            "name" = Parameter_Name));
            Output_Element (Xml_Buffer, "value",
                            Config.Delay_To_Wait_Before_Mass_Killing'Img);
         when Run_Number =>
            Start_Element (Xml_Buffer, "data",
                           ("mode" = Read_Write'Img,
                            "monitor" = Request'Img,
                            "type" = "integer",
                            "size" = "32",
                            "name" = Parameter_Name));
            Output_Element (Xml_Buffer, "value", Config.Run_Number'Img);

         when Run_Numbering =>
            Start_Element (Xml_Buffer, "data",
                           ("mode" = Read_Write'Img,
                            "monitor" = Request'Img,
                            "type" = "boolean",
                            "name" = Parameter_Name));
            Output_Element (Xml_Buffer, "value", Config.Run_Numbering'Img);


         when List_Actions =>
            declare
               Actions : constant Action_Array :=
                 Config.Slave_State_Machine.Available_Orders;
            begin
               if Config.Slave_State_Machine.Current_State =
                 Initial then
                  Start_Element (Xml_Buffer, "data",
                                 ("mode" = Read_Only'Img,
                                  "monitor" = Request'Img,
                                  "actions_count" = (1 + Actions'Length),
                                  "type" = T_ACTION,
                                  "name" = Parameter_Name));
               else
                  Start_Element (Xml_Buffer, "data",
                                 ("mode" = Read_Only'Img,
                                  "monitor" = Request'Img,
                                  "actions_count" = Actions'Length,
                                  "type" = T_ACTION,
                                  "name" = Parameter_Name));
               end if;
               for I in Actions'Range loop
                  Output_Element (Xml_Buffer, "value", Actions (I)'Img);
               end loop;
               if Config.Slave_State_Machine.Current_State = Initial then
                  Output_Element (Xml_Buffer, "value", "FINISH");
               end if;
            end;
         when Action =>
            Error_Out (Config.Logger'Access,
                       "action is a write only parameter");
            Ada.Exceptions.Raise_Exception
              (Write_Only_Parameter'Identity,
               "action is a write only parameter");
      end case;
      End_Element (Xml_Buffer, "data");
   end Add_Value;

   function Get_Xml (Config : access Actor_Coordination_Type;
                     Parameter : String) return String is
      Parameter_Low_Case : constant String :=
        Ada.Characters.Handling.To_Lower (Parameter);
      Configuration_Name : constant String :=
        Options.Get_Option ("config_name");
      use McKae.XML.EZ_Out.String_Stream.String_Buffering;
      use McKae.XML.EZ_Out.String_Stream.XML_String_Buffer;
      Xml_Buffer : String_Buffer;
   begin
      Current_Format := McKae.XML.EZ_Out.Continuous_Stream;
      Start_Element (Xml_Buffer, "result",
                     ("cmd" = "get",
                      "status" = "OK",
                      "sub_system_name" = Configuration_Name));
      Add_Value (Config, Parameter_Low_Case, Xml_Buffer);
      End_Element (Xml_Buffer, "result");
      declare
         String_To_Return : constant String := Get_String (Xml_Buffer);
      begin
         Full_Clear (Xml_Buffer);
         return String_To_Return;
      end;
   end Get_Xml;

   function Get_Image (Config : access Actor_Coordination_Type;
                       Parameter_Name : String) return String is
      Parameter : Actors_Coordination_Parameter_Type;
   begin
      begin
         Parameter :=
           Actors_Coordination_Parameter_Type'Value (Parameter_Name);
      exception
         when others =>
            Error_Out (Config.Logger'Access,
                       Parameter_Name & " isn't known");
            raise Unknown_Parameter;
      end;
      case Parameter is
         when Actors =>
            return "no_image_available_for_actors";
         when Configuration_Files =>
            declare
               Return_Base : Unbounded_String := Null_Unbounded_String;
               Files : constant Types.String_Array :=
                 Files_Handling.Get_Files_In
                 (Environment.Configuration_Directory.all,
                  "*.xml");
            begin
               for I in Files'Range loop
                  Return_Base := Return_Base & Files (I);
                  if I /= Files'Last then
                     Return_Base := Return_Base & ",";
                  end if;
               end loop;
               return To_String (Return_Base);
            end;
         when Prefix =>
            return To_String (Config.Prefix);
         when Name =>
            return To_String (Config.Name);
         when Configuration_File =>
            return To_String (Config.Configuration_File);
         when State =>
            return Config.Slave_State_Machine.Image;
         when Run_Number =>
            return Config.Run_Number'Img;
         when Run_Numbering =>
            return Config.Run_Numbering'Img;
         when Wait_Time =>
            return Config.Delay_To_Wait'Img;
         when Wait_Kill_Time =>
            return Config.Delay_To_Wait_Before_Mass_Killing'Img;
         when List_Actions =>
            declare
               Actions : constant Action_Array :=
                 Config.Slave_State_Machine.Available_Orders;
               Return_String : Unbounded_String := Null_Unbounded_String;
            begin
               if Actions'Length = 1 then
                  return Actions (Actions'First)'Img;
               else
                  for I in Actions'Range loop
                     Return_String := Return_String & Actions (I)'Img;
                     if I /= Actions'Last then
                        Return_String := Return_String & ",";
                     end if;
                  end loop;
                  return To_String (Return_String);
               end if;
            end;
         when Action =>
            Error_Out (Config.Logger'Access,
                       "action is a write only parameter");
            Ada.Exceptions.Raise_Exception
              (Write_Only_Parameter'Identity,
               "action is a write only parameter");
      end case;
   end Get_Image;

   --------------------------------------------
   -- procedure Changer_Etat_ordre_croissant --
   --------------------------------------------

   procedure Ordered_Change_State
     (Config : access Actor_Coordination_Type;
      Order : Narval.Action;
      Inversion : Boolean := False);
   procedure Ordered_Change_State
     (Config : access Actor_Coordination_Type;
      Order : Narval.Action;
      Inversion : Boolean := False)
   is
      Actors : constant Narval.Actors.Actor_Class_Access_Array :=
        Get_Active_Actors (Config);
   begin
      if not Inversion then
         for I in Actors'Range loop
            Narval.Actors.Change_State (Actors (I), Order);
         end loop;
      else
         for I in reverse Actors'Range loop
            Narval.Actors.Change_State (Actors (I), Order);
         end loop;
      end if;
   exception
      when E : others =>
         Fatal_Out (Config.Logger'Access, "Changer_Etat_Ordre_Croissant", E);
         raise;
   end Ordered_Change_State;

   procedure Change_State (Config : access Actor_Coordination_Type;
                           Order : Narval.Action) is
      use Narval.Configurator.Compilation;
      use Actors_Description_Vector;
      procedure Send_Error (Routine : String;
                            E : Ada.Exceptions.Exception_Occurrence);
      procedure On_Configure;
      procedure On_Load;
      procedure On_Unload;
      procedure On_Unconfigure;
      procedure Send_Error (Routine : String;
                            E : Ada.Exceptions.Exception_Occurrence) is
      begin
         Fatal_Out (Config.Logger'Access, Routine & " envoie erreur", E);
         Config.Slave_State_Machine.Put_In_Error;
      end Send_Error;
      procedure On_Configure is
         Text : constant String := "Passage d etat configurer rate";
      begin
         Reset;
         Configure;
      exception
         when Configuration_Failed =>
            Config.Slave_State_Machine.Put_In_Error;
            Error_Out (Config.Logger'Access, Text);
            raise;
      end On_Configure;
      procedure On_Load is
         procedure Prepare_To_Launch (Actor_Position : Cursor);
         procedure Prepare_To_Launch (Actor_Position : Cursor) is
            Actor : constant Actor_Description_Access :=
              Element (Actor_Position);
         begin
            Uuid.Uuid_Clear (Actor.Uuid_Value);
            Uuid.Uuid_Generate (Actor.Uuid_Value);
         end Prepare_To_Launch;
         procedure Launch_Actor (Actor_Position : Cursor);
         procedure Launch_Actor (Actor_Position : Cursor) is
            Actor : constant Actor_Description_Access :=
              Element (Actor_Position);
            Command : constant String := "ssh " &
              To_String (Actor.Host_Name) & " " &
              To_String (Config.Prefix & " " & Actor.Process_Name) &
              " --name " & To_String (Actor.Name) & " --config_name " &
              To_String (Config.Name) &
              " --uuid " & Uuid.Image (Actor.Uuid_Value) & " &";
         begin
            Info_Out (Config.Logger'Access,
                      "issuing command : " & Command);
            System_Calls.Exec_Command (Command);
         end Launch_Actor;
         procedure Check_Actor (Actor_Position : Cursor);
         procedure Check_Actor (Actor_Position : Cursor) is
            Actor : constant Actor_Description_Access :=
              Element (Actor_Position);
            Waiting_Loop : Natural := 0;
            use type Narval.Actors.Actor_Class_Access;
            Actor_Pointer : Narval.Actors.Actor_Class_Access := null;
         begin
            loop
               select
                  Actor.Task_Register.Get_Pointer (Actor_Pointer);
                  exit;
               or
                  delay 0.1;
               end select;
               Waiting_Loop := Waiting_Loop + 1;
               if Waiting_Loop > Config.Delay_To_Wait then
                  Waiting_Loop := 0;
                  Error_Out (Config.Logger'Access,
                             To_String (Actor.Name) &
                             " is too long to subscribe");
                  raise Shell_Problem;
               end if;
            end loop;
         end Check_Actor;
      begin
         Iterate (Config.Actors, Prepare_To_Launch'Access);
         Iterate (Config.Actors, Launch_Actor'Access);
         Iterate (Config.Actors, Check_Actor'Access);
      exception
         when E : others =>
            Send_Error ("Sur_Load", E);
            raise;
      end On_Load;
      procedure On_Unload is
         procedure Unload_Actor (Actor_Position : Cursor);
         procedure Unload_Actor (Actor_Position : Cursor) is
            Actor : constant Actor_Description_Access :=
              Element (Actor_Position);
            Tmp_Access : Narval.Actors.Actor_Class_Access;
         begin
            select
               Actor.Task_Register.Get_Pointer (Tmp_Access);
            else
               raise Actor_Not_Registred;
            end select;
            Narval.Actors.End_Process (Tmp_Access);
         end Unload_Actor;
      begin
         Iterate (Config.Actors, Unload_Actor'Access);
         Info_Out (Config.Logger'Access,
                   "waiting" & Config.Delay_To_Wait_Before_Mass_Killing'Img &
                   " seconds before killing all actors");
         delay 1.0 * Duration (Config.Delay_To_Wait_Before_Mass_Killing);
         Clean_Config (Config, True);
         Pointers_Reset;
      exception
         when E : others =>
            Send_Error ("Sur_Unload", E);
            raise;
      end On_Unload;
      procedure On_Unconfigure is
      begin
         Reset;
      exception
         when E : others =>
            Send_Error ("Sur_Unconfigure", E);
            raise;
      end On_Unconfigure;
      Ordre_Depart : Narval.Action;
   begin
      case Order is
         when Configure_Load =>
            Ordre_Depart := Configure;
         when Unload_Unconfigure =>
            Ordre_Depart := Unload;
         when others =>
            Ordre_Depart := Order;
      end case;
      if not Config.Slave_State_Machine.Order_Coherency (Ordre_Depart) then
         Warn_Out (Config.Logger'Access,
                   "etat courant : " & Config.Slave_State_Machine.Image &
                   ",ordre incoherent :" & Order'Img);
         return;
      end if;
      case Order is
         when Partial_Reset =>
            Clean_Config (Config, False);
            On_Configure;
            Config.Slave_State_Machine.Change_State (Partial_Reset);
         when Full_Reset =>
            Clean_Config (Config, False);
            Config.Slave_State_Machine.Change_State (Full_Reset);
         when Configure =>
            On_Configure;
            Config.Slave_State_Machine.Change_State (Configure);
         when Configure_Load =>
            Change_State (Config, Configure);
            Change_State (Config, Load);
            return;
         when Unload_Unconfigure =>
            Change_State (Config, Unload);
            Change_State (Config, Unconfigure);
            return;
         when Unconfigure =>
            On_Unconfigure;
            Config.Slave_State_Machine.Change_State (Order);
         when Load =>
            On_Load;
            Config.Slave_State_Machine.Change_State (Order);
         when Unload =>
            On_Unload;
            Config.Slave_State_Machine.Change_State (Order);
         when Initialise =>
            begin
               Ordered_Change_State (Config, Initialise);
               Config.Slave_State_Machine.Change_State (Order);
            exception
               when E : others =>
                  Send_Error ("Initialise", E);
                  raise;
            end;
         when Reset_Com =>
            begin
               Ordered_Change_State (Config,
                                     Reset_Com,
                                     Inversion => True);
               Config.Slave_State_Machine.Change_State (Order);
            exception
               when E : others =>
                  Send_Error ("Reset_Com", E);
                  raise;
            end;
         when Start =>
            begin
               Ordered_Change_State (Config,
                                     Start,
                                     Inversion => True);
               Config.Slave_State_Machine.Change_State (Order);
            exception
               when E : others =>
                  Send_Error ("Start", E);
                  raise;
            end;
         when Stop =>
            begin
               if Config.Run_Numbering then
                  Config.Run_Number :=  Config.Run_Number + 1;
               end if;
               Ordered_Change_State (Config, Stop);
               Config.Slave_State_Machine.Change_State (Order);
            exception
               when E : others =>
                  Send_Error ("Stop", E);
                  Pointers_Reset;
                  raise;
            end;
         when Pause =>
            begin
               Ordered_Change_State (Config, Pause);
               Config.Slave_State_Machine.Change_State (Order);
            exception
               when E : others =>
                  Send_Error ("Pause", E);
                  raise;
            end;
         when Resume =>
            begin
               Ordered_Change_State (Config,
                                     Resume,
                                     Inversion => True);
               Config.Slave_State_Machine.Change_State (Order);
            exception
               when E : others =>
                  Send_Error ("Resume", E);
                  raise;
            end;
      end case;
   exception
      when Configuration_Failed =>
         Config.Slave_State_Machine.Put_In_Error;
         Clean_Config (Config, False);
         Info_Out (Config.Logger'Access,
                   "Configuration_Ratee, set state to initial");
         raise;
      when E : others =>
         Fatal_Out (Config.Logger'Access, "Changer_Etat", E);
         Info_Out (Config.Logger'Access, "set state to error");
         Config.Slave_State_Machine.Put_In_Error;
         raise;
   end Change_State;

   procedure Special_Domi (Config : access Actor_Coordination_Type) is
   begin
      Config.Slave_State_Machine.Put_In_Error;
   end Special_Domi;

   procedure Set (Config : access Actor_Coordination_Type;
                  Parameter_Name : String;
                  Value : String) is
      Parameter : Actors_Coordination_Parameter_Type;
   begin
      begin
         Parameter :=
           Actors_Coordination_Parameter_Type'Value (Parameter_Name);
      exception
         when others =>
            Error_Out (Config.Logger'Access,
                       Parameter_Name & " isn't known");
            raise Unknown_Parameter;
      end;
      case Parameter is
         when Name | Configuration_Files | Actors =>
            Ada.Exceptions.Raise_Exception
              (Read_Only_Parameter'Identity,
               Parameter'Img & " is a read only parameter");
         when Action =>
            declare
               Order : constant Narval.Action := Narval.Action'Value (Value);
            begin
               Change_State (Config, Order);
            end;
         when Configuration_File =>
            Config.Configuration_File := To_Unbounded_String (Value);
         when Prefix =>
            Config.Prefix := To_Unbounded_String (Value);
         when Run_Number =>
            Config.Run_Number := Integer'Value (Value);
         when Run_Numbering =>
            Config.Run_Numbering := Boolean'Value (Value);
         when Wait_Time =>
            Config.Delay_To_Wait := 10 * Natural'Value (Value);
         when Wait_Kill_Time =>
            Config.Delay_To_Wait_Before_Mass_Killing := Natural'Value (Value);
         when State | List_Actions =>
            Error_Out (Config.Logger'Access,
                       Parameter'Img & " is a read only parameter");
            Ada.Exceptions.Raise_Exception
              (Read_Only_Parameter'Identity,
               Parameter'Img & " is a read only parameter");
      end case;
   exception
      when E : Constraint_Error =>
         Error_Out (Config.Logger'Access,
                    "exception in set procedure", E);
         if Parameter = Action then
            Ada.Exceptions.Raise_Exception
              (Bad_Format'Identity,
               Value & " : not conformant with type action");
         else
            raise;
         end if;
   end Set;

   function Arguments (Config : access Actor_Coordination_Type;
                       Xml : Boolean := True)
                      return String is
      use McKae.XML.EZ_Out.String_Stream.String_Buffering;
      use McKae.XML.EZ_Out.String_Stream.XML_String_Buffer;
      Xml_Buffer : String_Buffer;
   begin
      if Xml then
         Start_Element (Xml_Buffer, "result",
                        ("cmd" = "get", "status" = "OK",
                         "sub_system_name" = To_String (Config.Name)));
         for I in Actors_Coordination_Parameter_Type'Range loop
            if I /= Action then
               Add_Value (Config, I, Xml_Buffer);
            else
               Start_Element (Xml_Buffer, "data",
                              ("mode" = Write_Only'Img,
                               "monitor" = Never'Img,
                               "type" = T_ACTION,
                               "name" = "action"));
               Output_Element (Xml_Buffer, "value", "");
               End_Element (Xml_Buffer, "data");
            end if;
         end loop;
         End_Element (Xml_Buffer, "result");
         declare
            String_To_Return : constant String := Get_String (Xml_Buffer);
         begin
            Full_Clear (Xml_Buffer);
            return String_To_Return;
         end;
      end if;
      declare
         Return_String : Unbounded_String := To_Unbounded_String
           ("fichier_configuration");
      begin
         for I in Actors_Coordination_Parameter_Type'Range loop
            Return_String := Return_String & "," & I'Img;
         end loop;
         return To_String (Return_String);
      end;
   end Arguments;

   --------------
   -- Register --
   --------------

   procedure Register (Config : access Actor_Coordination_Type;
                       Name : String;
                       Pointer : Narval.Actors.Actor_Class_Access) is
      Uncontrained_Name : constant Unbounded_String :=
        To_Unbounded_String (Name);
      Actor_Position : Actors_Description_Vector.Cursor;
      use type Actors_Description_Vector.Cursor;
   begin
      Actor_Position := Find_Actor (Config.Actors,
                                    Uncontrained_Name);
      if Actor_Position = Actors_Description_Vector.No_Element then
         raise No_Such_Actor_In_Configuration;
      end if;
      Actors_Description_Vector.Element
        (Actor_Position).Task_Register.Post (Pointer);
      Info_Out (Config.Logger'Access, Name & " inscrit");
   exception
      when E : others =>
         Fatal_Out (Config.Logger'Access, "Register", E);
         raise;
   end Register;

   ----------------
   -- Rechercher --
   ----------------

   function Get_Actor (Config : access Actor_Coordination_Type;
                       Name : String)
                      return Narval.Actors.Actor_Class_Access is
      Uncontrained_Name : constant Unbounded_String :=
        To_Unbounded_String (Name);
      Actor_Pointer : Narval.Actors.Actor_Class_Access;
      Actor_Position : Actors_Description_Vector.Cursor;
      use type Actors_Description_Vector.Cursor;
   begin
      Actor_Position := Find_Actor (Config.Actors,
                                    Uncontrained_Name);
      if Actor_Position = Actors_Description_Vector.No_Element then
         raise Unknown_Actor;
      end if;
      select
         Actors_Description_Vector.Element
           (Actor_Position).Task_Register.Get_Pointer (Actor_Pointer);
      else
         raise Actor_Not_Registred;
      end select;
      return Actor_Pointer;
   exception
      when E : others =>
         Error_Out (Config.Logger'Access, "rechercher acteur", E);
         raise;
   end Get_Actor;

   -------------------------------
   -- Rechercher_Acteurs_Actifs --
   -------------------------------

   function Get_Active_Actors
     (Config : access Actor_Coordination_Type)
     return Narval.Actors.Actor_Class_Access_Array is
      Actor_Number : Positive := 1;
      Cursor : Actors_Description_Vector.Cursor;
      use Actors_Description_Vector;
   begin
      declare
         Active_Actor_Number : constant Natural :=
           Natural (Config.Actors.Length);
         Vector : Narval.Actors.Actor_Class_Access_Array
           (1 .. Active_Actor_Number);
      begin
         if Active_Actor_Number = 0 then
            raise No_Active_Actors;
         end if;
         Cursor := First (Config.Actors);
         loop
            exit when Cursor = No_Element;
            declare
               Actor_Pointer : Narval.Actors.Actor_Class_Access;
            begin
               select
                  Element (Cursor).Task_Register.Get_Pointer
                    (Actor_Pointer);
               else
                  raise Actor_Not_Registred;
               end select;
               Vector (Actor_Number) := Actor_Pointer;
            end;
            Actor_Number := Actor_Number + 1;
            Cursor := Actors_Description_Vector.Next (Cursor);
         end loop;
         return Vector;
      end;
   exception
      when No_Active_Actors =>
         raise;
      when E : others =>
         Error_Out (Config.Logger'Access, "rechercher acteurs actifs", E);
         raise;
   end Get_Active_Actors;

   function Get_Informations
     (Config : access Actor_Coordination_Type; Name : String) return
     String is
      Uncontrained_Name : constant Unbounded_String :=
        To_Unbounded_String (Name);
      Actor_Position : Actors_Description_Vector.Cursor;
      use type Actors_Description_Vector.Cursor;
   begin
      Actor_Position := Find_Actor (Config.Actors,
                                    Uncontrained_Name);
      if Actor_Position = Actors_Description_Vector.No_Element then
         raise Unknown_Actor;
      end if;
      return To_String (Actors_Description_Vector.Element
                          (Actor_Position).all);
   end Get_Informations;

   function Current_State (Config : access Actor_Coordination_Type)
                          return Acquisition_State_Type is
   begin
      return Config.Slave_State_Machine.Current_State;
   end Current_State;

   function Order_Coherency (Config : access Actor_Coordination_Type;
                             Order : Narval.Action) return Boolean is
   begin
      return Config.Slave_State_Machine.Order_Coherency (Order);
   end Order_Coherency;

   function Available_Orders (Config : access Actor_Coordination_Type)
                             return Action_Array is
   begin
      return Config.Slave_State_Machine.Available_Orders;
   end Available_Orders;
   procedure End_Actor_Handler (Config : access Actor_Coordination_Type) is
      Current_State : constant Acquisition_State_Type :=
        Config.Slave_State_Machine.Current_State;
   begin
      case Current_State is
         when Initial =>
            Narval.Narval_Naming_Registry.Remove_Sub_System
              (To_String (Config.Name));
            Config.End_Process := True;
         when others =>
            raise Invalid_Order;
      end case;
   end End_Actor_Handler;
   function Have_To_End (Config : Actor_Coordination_Type) return Boolean is
   begin
      return Config.End_Process;
   end Have_To_End;

   procedure Initialise (Config : in out Actor_Coordination_Type;
                         Configuration_Name : String) is
      Logger_Name : constant String := "log from chef_orchestre : " &
        Configuration_Name;
      Event_Receiver : Events_Receivers.Events_Receivers_Class_Access;
   begin
      Config.Name := To_Unbounded_String (Configuration_Name);
      Config.Prefix := To_Unbounded_String ("-x");
      Set_Name (Config.Logger'Access, Logger_Name);
      Set_Level (Config.Logger'Access, Log4ada.Info);
      Add_Appender (Config.Logger'Access,
                    Config.Console'Unchecked_Access);
      Add_Appender (Config.Logger'Access,
                    Config.Remote_Appender'Unchecked_Access);
      begin
         Event_Receiver := Narval_Naming_Registry.Get_Event_Receiver;
         Log4ada.Appenders.Annex_E.Set_Receiver
           (Config.Remote_Appender, Event_Receiver);
      exception
         when Narval_Naming_Registry.No_Event_Receiver =>
            Log4ada.Appenders.Annex_E.Disable
              (Config.Remote_Appender'Access);
            Log4ada.Loggers.Warn_Out (Config.Logger'Access,
                                      "no remote logging");
      end;
   end Initialise;

   function Get_Configuration_Files (Config : access Actor_Coordination_Type)
                                    return String is
      pragma Unreferenced (Config);
   begin
      declare
         Files : constant Types.String_Array :=
           Files_Handling.Get_Files_In
           (Environment.Configuration_Directory.all,
            "*.xml");
         Long_Unbounded_String : Unbounded_String := Files (Files'First);
      begin
         for I in Files'First + 1 .. Files'Last loop
            Long_Unbounded_String := Long_Unbounded_String & ";" & Files (I);
         end loop;
         return To_String (Long_Unbounded_String);
      end;
   end Get_Configuration_Files;

   procedure Trig_Event (Config : access Actor_Coordination_Type;
                         Event : String) is
      Event_Receiver : Events_Receivers.Events_Receivers_Class_Access;
   begin
      if Event = "log_reload" then
         begin
            Event_Receiver := Narval_Naming_Registry.Get_Event_Receiver;
         exception
            when Narval_Naming_Registry.No_Event_Receiver =>
               Log4ada.Appenders.Annex_E.Disable
                 (Config.Remote_Appender'Access);
               Log4ada.Loggers.Warn_Out (Config.Logger'Access,
                                         "log_reload asked but " &
                                         "no remote logging");
               return;
         end;
         Log4ada.Appenders.Annex_E.Set_Receiver
           (Config.Remote_Appender, Event_Receiver);
         Log4ada.Appenders.Annex_E.Enable (Config.Remote_Appender'Access);
         declare
            use Actors_Description_Vector;
            procedure Send_Trig (Actor_Position : Cursor);
            procedure Send_Trig (Actor_Position : Cursor) is
               Actor : constant Actor_Description_Access :=
                 Element (Actor_Position);
               Actor_Pointer : Narval.Actors.Actor_Class_Access := null;
            begin
               select
                  Actor.Task_Register.Get_Pointer (Actor_Pointer);
                  Narval.Actors.Trig_Event (Actor_Pointer, Event);
               or
                  delay 0.1;
               end select;
            end Send_Trig;
         begin
            Iterate (Config.Actors, Send_Trig'Access);
         end;
      end if;
   end Trig_Event;

end Narval.Configurator.Abstract_Actors_Coordination.Actors_Coordination;
