--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Ada.Unchecked_Deallocation;
with Ada.Streams;
with System;
with Ada.Strings.Unbounded;
with Interfaces;

with Narval.Protected_Memory;

with Log4ada.Loggers;
with Start_Stop_Handling;

package Narval.Communication is

   type Stream_Access is access all Ada.Streams.Root_Stream_Type'Class;

   type Bytes_Count_Type is mod 2**64;

   type Link_Type is abstract tagged limited record
      Logger : Log4ada.Loggers.Logger_Class_Access;
      Bypass : Boolean := False;
      Bytes_In_Count : Bytes_Count_Type := 0;
      Bytes_Out_Count : Bytes_Count_Type := 0;
      Asker : Ada.Strings.Unbounded.Unbounded_String;
      Provider : Ada.Strings.Unbounded.Unbounded_String;
   end record;
   type Link_Access is access all Link_Type'Class;

   function Get_Init_Info (Link : access Link_Type)
                                          return Integer is abstract;
   procedure Set_Asker (Link : access Link_Type;
                              Asker : String);
   function Asker_Link (Link : access Link_Type) return String;
   function Provider_Link (Link : access Link_Type) return String;

   procedure Connect (Link : access Link_Type;
                      Bit_Order_To_Send : in out System.Bit_Order) is abstract;
   function Stream (Link : access Link_Type) return Stream_Access is abstract;
   procedure Free (Link : access Link_Type;
                   Stream_To_Free : in out Stream_Access) is abstract;
   procedure Send (Link : access Link_Type;
                      Data : Ada.Streams.Stream_Element_Array)
      is abstract;
   procedure Send_Size (Link : access Link_Type;
                        Size : Interfaces.Unsigned_64);
   function Receive_Size (Link : access Link_Type)
                            return Interfaces.Unsigned_64;
   procedure Receive (Link : access Link_Type;
                       Data : out Ada.Streams.Stream_Element_Array)
      is abstract;

   procedure Close (Link : access Link_Type) is abstract;

   type Link_Array is array (Positive range <>) of Link_Access;
   type Link_Array_Access is access Link_Array;

   task type Output_Task_Type
     (Memory : Protected_Memory.Memory_Access;
      Links : Link_Array_Access;
      Start_Stop : Start_Stop_Handling.Synchro_Start_Access;
      Logger : Log4ada.Loggers.Logger_Class_Access) is
      entry Start;
      entry Stop;
   end Output_Task_Type;
   type Output_Task_Access is access Output_Task_Type;
   procedure Free is new Ada.Unchecked_Deallocation
     (Output_Task_Type, Output_Task_Access);
   task type Input_Task_Type
     (Memory : Protected_Memory.Memory_Access;
      Link : Link_Access;
      Logger : Log4ada.Loggers.Logger_Class_Access) is
      entry Start;
      entry Stop;
   end Input_Task_Type;
   type Input_Task_Access is access Input_Task_Type;
   procedure Free is new Ada.Unchecked_Deallocation
     (Input_Task_Type, Input_Task_Access);

end Narval.Communication;
