--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Socket_Send;
with Socket_Receive;
with Low_Level_Network.Utils;
with Utils;

with Narval.Random_Draw_Thousand;

package body Narval.Communication.Sockets is

   use Log4ada.Loggers;
   use Ada.Strings.Unbounded;

   Root : Random_Draw_Thousand.Draw_Package.Generator;

   Offset_Port : constant := 8000;

   -----------------------------------
   -- Recuperer_Info_Initialisation --
   -----------------------------------

   function Get_Init_Info
     (Link : access Socket_Link_Type)
     return Integer is
   begin
      return Integer (Link.Port_Number);
   end Get_Init_Info;

   ------------
   -- Stream --
   ------------

   function Stream (Link : access Socket_Link_Type) return Stream_Access is
   begin
      return Stream_Access (GNAT.Sockets.Stream (Link.Socket));
   end Stream;

   procedure Free (Link : access Socket_Link_Type;
                   Stream_To_Free : in out Stream_Access) is
      pragma Unreferenced (Link);
      Socket_Stream : GNAT.Sockets.Stream_Access;
   begin
      Socket_Stream := GNAT.Sockets.Stream_Access (Stream_To_Free);
      GNAT.Sockets.Free (Socket_Stream);
      Stream_To_Free := null;
   end Free;

   ----------
   -- Send --
   ----------

   procedure Send
     (Link : access Socket_Link_Type;
      Data : Ada.Streams.Stream_Element_Array)
   is
   begin
      Socket_Send (Link.Socket, Data);
   end Send;

   -------------
   -- Receive --
   -------------

   procedure Receive
     (Link : access Socket_Link_Type;
      Data : out Ada.Streams.Stream_Element_Array)
   is
   begin
      Socket_Receive (Link.Socket, Data);
   end Receive;

   -----------
   -- Clore --
   -----------

   procedure Close (Link : access Socket_Link_Type) is
   begin
      GNAT.Sockets.Close_Socket (Link.Socket);
   end Close;

   -------------
   -- Etablir --
   -------------

   procedure Connect (Link : access Client_Socket_Link_Type;
                      Bit_Order_To_Send : in out System.Bit_Order) is
   begin
      GNAT.Sockets.Connect_Socket (Link.Socket, Link.Address);
      Bit_Order_To_Send := System.Bit_Order'Input (Stream (Link));
   end Connect;

   task body Connect_Task_Type is
      Local_Bit_Order_To_Send : System.Bit_Order;
   begin
      accept Connect (Bit_Order_To_Send : in System.Bit_Order) do
         Local_Bit_Order_To_Send := Bit_Order_To_Send;
      end Connect;
      GNAT.Sockets.Accept_Socket
        (Wrapper.Server_Access.Server,
         Wrapper.Socket,
         Wrapper.Address);
      System.Bit_Order'Output (Stream (Wrapper), Local_Bit_Order_To_Send);
   end Connect_Task_Type;

   procedure Connect (Link : access Server_Socket_Link_Type;
                      Bit_Order_To_Send : in out System.Bit_Order) is
   begin
      Link.Connect_Task.Connect (Bit_Order_To_Send);
   end Connect;

   -----------
   -- Clore --
   -----------

   procedure Close (Link : access Server_Socket_Link_Type) is
   begin
      GNAT.Sockets.Close_Socket (Link.Socket);
      Link.Server_Access.Lock_Reference_Number :=
        Link.Server_Access.Lock_Reference_Number - 1;
      if Link.Server_Access.Lock_Reference_Number = 0 then
         GNAT.Sockets.Close_Socket (Link.Server_Access.Server);
         Free (Link.Server_Access);
      end if;
   end Close;

   ------------------------
   -- Initialiser_Client --
   ------------------------

   function Init_Client
     (Port_Number : Positive;
      Address : String;
      Logger : Log4ada.Loggers.Logger_Class_Access)
     return Link_Access is
      Link : Client_Socket_Link_Access;
   begin
      Link := new Client_Socket_Link_Type;
      Link.Port_Number := GNAT.Sockets.Port_Type (Port_Number);
      Link.Logger := Logger;
      Link.Address.Addr := GNAT.Sockets.Inet_Addr
        (Utils.Cut_Last_Spaces (Address));
      Link.Address.Port := GNAT.Sockets.Port_Type (Port_Number);
      Link.Host := To_Unbounded_String (Address);
      GNAT.Sockets.Create_Socket (Link.Socket);
      GNAT.Sockets.Set_Socket_Option (Link.Socket,
                                      GNAT.Sockets.Socket_Level,
                                      (GNAT.Sockets.Reuse_Address, True));
      return Link_Access (Link);
   end Init_Client;

   -------------------------
   -- Initialiser_Serveur --
   -------------------------

   function Init_Server
     (Port : String;
      First : Boolean;
      Server : Server_Socket_Link_Access;
      Logger : Log4ada.Loggers.Logger_Class_Access)
     return Server_Socket_Link_Access is
      Host : constant String := Low_Level_Network.Utils.Get_Address (Port);
      Position : Random_Draw_Thousand.Thousand_Type;
      Link : Server_Socket_Link_Access;
      use type GNAT.Sockets.Port_Type;
   begin
      Link := new Server_Socket_Link_Type (False);
      Link.Logger := Logger;
      Link.Host := To_Unbounded_String (Host);
      Link.Server_Access := Server;
      if First then
         Link.Server_Access := new Server_Socket_Link_Type (True);
         Link.Server_Access.Address.Addr := GNAT.Sockets.Inet_Addr (Host);
         loop
            Position := Random_Draw_Thousand.Draw_Package.Random (Root);
            begin
               Link.Server_Access.Address.Port :=
                 GNAT.Sockets.Port_Type (Position) + Offset_Port;
               GNAT.Sockets.Create_Socket (Link.Server_Access.Server);
               GNAT.Sockets.Set_Socket_Option (Link.Server_Access.Server,
                                               GNAT.Sockets.Socket_Level,
                                               (GNAT.Sockets.Reuse_Address,
                                                True));
               GNAT.Sockets.Bind_Socket (Link.Server_Access.Server,
                                         Link.Server_Access.Address);
               Link.Server_Access.Lock_Reference_Number := 1;
               GNAT.Sockets.Listen_Socket (Link.Server_Access.Server);
               Link.Port_Number := GNAT.Sockets.Port_Type (Position) +
                 Offset_Port;
               Link.Server_Access.Port_Number := Link.Port_Number;
               return Link;
            exception
               when GNAT.Sockets.Socket_Error =>
                  null;
            end;
         end loop;
      else
         Link.Server_Access := Server;
         Server.Lock_Reference_Number := Server.Lock_Reference_Number + 1;
         return Link;
      end if;
   end Init_Server;

begin
   Narval.Random_Draw_Thousand.Draw_Package.Reset (Root);
end Narval.Communication.Sockets;
