/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */


#include <glib.h>
#include <string.h>
#include <sys/socket.h>
#include "net_nntp_client.h"


static int do_authentication(MNetNNTPClient *client);

#define TOERROR(rc, err)  switch (err) {\
case -2: err= MNNTPOutOfMemory; break; \
case -1: err= MNNTPNetError; break; \
default: err= MNNTPProtocolError; break;\
}


MNetNNTPClient *mnet_init_nntp(int use_ssl)
{
  MNetNNTPClient *cli= g_malloc0(sizeof(MNetNNTPClient));

  if (use_ssl)
    cli->client= mnet_init_ssl_client(256);
  else
    cli->client= mnet_init_client(256);

  if (!cli)
    return NULL;

  return cli;
}


static int parse_status(char *line, char **text)
{
  char buf[32];
  char *ptr= strchr(line, ' ');
  int num;
  
/*  g_message("%s", line);*/
  
  if (!ptr)
  {
    *text= line;
    return -1;
  }

  if (ptr-line >= sizeof(buf)-1)
  {
    *text= line;
    return -1;
  }
  strncpy(buf, line, ptr-line);
  buf[ptr-line]= 0;
  
  *text= ptr+1;

  if (sscanf(buf, "%i", &num)==1)
    return num;

  return -1;
}


static int wait_response(MNetNNTPClient *client, char **resp, char **buffer)
{
  char *line;
  int status= -1;

  for (;;)
  {
    if ((line= mnet_get_line(client->client)))
    {
      char *ptr;
      status= parse_status(line, &ptr);
      if (resp)
      {
        *resp= ptr;
        *buffer= line;
      }
      else
      {
        if (buffer)
          *buffer= NULL;
        if (resp)
          *resp= NULL;
        g_free(line);
      }
      break;
    }
    if ((status= mnet_wait_ready(client->client, -1)) < 0)
    {
      break;
    }
    if ((status= mnet_read_data(client->client)) < 0)
    {
      break;
    }
  }
  return status;
}


static int send_command(MNetNNTPClient *client, char *cmd)
{
  int err;
  if ((err=mnet_send_line(client->client, cmd)) < 0)
    return err;
  if (mnet_flush_data(client->client) < 0)
    return -1;
  return 0;
}


static MNNTPError change_group(MNetNNTPClient *client, const char *group)
{
  char *cmd;
  int err;
  int retried= 0;

again:
  cmd= g_strdup_printf("GROUP %s", group);
  if ((err=send_command(client, cmd)) < 0)
  {
    MNNTPError error;
    g_free(cmd);
    TOERROR(err, error);
    return error;
  }
  g_free(cmd);

  err= wait_response(client, NULL, NULL);

  if (err == 480 && !retried)
  {
    MNNTPError error;
    retried= 1;
    error= do_authentication(client);
    if (error != MNNTPNoError)
      return error;
    goto again;
  }
  if (err != 211)
    return MNNTPProtocolError;

  return MNNTPNoError;
}


static int do_authentication(MNetNNTPClient *client)

{  
  char *buffer;
  
  buffer= g_strdup_printf("AUTHINFO USER %s", client->username);
  if (send_command(client, buffer) < 0)
  {
    g_free(buffer);
    return MNNTPNetError;
  }
  g_free(buffer);
  
  if (wait_response(client, NULL, NULL)!=381)
    return MNNTPProtocolError;
  
  buffer= g_strdup_printf("AUTHINFO PASS %s", client->password);
  if (send_command(client, buffer) < 0)
  {
    g_free(buffer);
    return MNNTPNetError;
  }
  g_free(buffer);

  switch (wait_response(client, NULL, NULL))
  {
  case 281: /* auth ok */
    return MNNTPNoError;
  case 502: /* auth denied */
  case 482: /* auth error */
    return MNNTPAuthError;
  case -1:
    return MNNTPNetError;
  default:
    return MNNTPProtocolError;
  }
}


void mnet_nntp_set_auth_info(MNetNNTPClient *client, 
                             const char *username,
                             const char *password)
{
  client->username= g_strdup(username);
  client->password= g_strdup(password);
}


MNNTPError mnet_nntp_connect(MNetNNTPClient *client, const char *host, int port)
{
  int err;
  
  if (client->client->socket)
    mnet_client_disconnect(client->client);

  if ((err= mnet_client_connect(client->client, host, port)) < 0)
  {
    if (err==-2)
      return MNNTPOutOfMemory;
    switch (client->client->state)
    {
    case MSResolveError: return MNNTPResolveError;
    case MSConnectError: return MNNTPConnectError;
    default: return MNNTPNetError;
    }
  }

  if (mnet_flush_data(client->client) < 0)
    return MNNTPNetError;

  err= wait_response(client, NULL, NULL);

  switch (err)
  {
  case 200:
    client->can_post= 1;
    break;
  case 201:
    client->can_post= 0;
    break;
  case -2:
    return MNNTPProtocolError;
  default:
    return MNNTPNetError;
  }

  return MNNTPNoError;
}


MNNTPError mnet_nntp_disconnect(MNetNNTPClient *client)
{
  send_command(client, "QUIT");

  mnet_client_disconnect(client->client);

  return MNNTPNoError;
}



MNNTPGroups *mnet_nntp_list_groups(MNetNNTPClient *client,
                                   MNNTPError *error)
{
  MNNTPGroups *groups;
  int alloced=0;
  char *line;
  int done= 0;
  int err;
  int retried= 0;

  *error= MNNTPNoError;
  
  g_return_val_if_fail(client->client->state == MSConnected, NULL);

retry:
  if ((err=send_command(client, "LIST")) < 0)
  {
    TOERROR(err, *error);
    return NULL;
  }

  if ((err=wait_response(client, NULL, NULL)) != 215)
  {
    if (err == 480 && !retried)
    {
      *error= do_authentication(client);
      if (*error != MNNTPNoError)
        return NULL;
      retried= 1;
      goto retry;
    }
    TOERROR(err, *error);
    return NULL;
  }

  alloced= 0;
  groups= g_malloc0(sizeof(MNNTPGroups));

  /* retrieve groups and information */
  while (!done)
  {
    while ((line= mnet_get_line(client->client)))
    {
      char *ptr;
      if (strcmp(line, ".")==0)
      {
        done= 1;
        break;
      }
      else
      {
        if (groups->groups_num == alloced)
        {
          alloced += 32;
          groups->groups= g_realloc(groups->groups, sizeof(MNNTPGroup)*alloced);
        }

        ptr= strtok(line, " ");
        if (ptr)
          groups->groups[groups->groups_num].name= g_strdup(ptr);
        ptr= strtok(NULL, " ");
        if (ptr)
          groups->groups[groups->groups_num].last= strtoul(ptr, NULL, 10);
        ptr= strtok(NULL, " ");
        if (ptr)
          groups->groups[groups->groups_num].first= strtoul(ptr, NULL, 10);
        ptr= strtok(NULL, " ");
        if (ptr)
          groups->groups[groups->groups_num].can_post= *ptr == 'y';

        if (!ptr)
        {
          g_free(line);
          goto error;
        }
        else
        {
          groups->groups_num++;
        }
      }
      g_free(line);
    }

    if (!done)
    {
      if ((err= mnet_wait_ready(client->client, -1)) < 0)
        goto error;

      if ((err= mnet_read_data(client->client)) < 0)
        goto error;
    }
  }

  groups->groups= g_realloc(groups->groups, sizeof(MNNTPGroup)*groups->groups_num);

  return groups;
  
error:
  TOERROR(err, *error);
  mnet_free_group_list(groups);
  return NULL;
}


int mnet_free_group_list(MNNTPGroups *groups)
{
  unsigned int i;
  for (i= 0; i < groups->groups_num; i++)
  {
    g_free(groups->groups[i].name);
  }
  g_free(groups->groups);
  g_free(groups);
  
  return 0;
}


#if 0
MNNTPArticleHeaders *mnet_nntp_list_articles(MNetNNTPClient *client,
                                             const char *group,
                                             unsigned int first,
                                             unsigned int last,
                                             unsigned int *next,
                                             MNNTPError *error)
{
  MNNTPArticleHeaders *headers;
  int alloced=0;
  char *line;
  int status;
  char *cmd;
  int err;
  struct timeval tv1, tv2;

  *error= MNNTPNoError;

  *next= first;

  g_return_val_if_fail(client->client->state == MSConnected, NULL);

  *error= change_group(group);
  if (*error != MNNTPNoError)
  {
    return NULL;
  }

  alloced= 32;
  headers= g_malloc0(sizeof(MNNTPArticleHeaders));
  headers->articles= g_malloc0(sizeof(MNNTPArticleHeader)*alloced);

  for (;;)
  {
    char *ptr;
    MNNTPArticleHeader *article;
    int lines_alloced;
    int article_done= 0;
    
    gettimeofday(&tv1, NULL);
    
    if (headers->articles_num == 0)
    { /* request the 1st */
      char buffer[128];
      sprintf(buffer, "STAT %i", first);
      if ((err=send_command(client, buffer)) < 0)
        goto error;
    }
    else
    {
      /* request next */
      if ((err=send_command(client, "NEXT")) < 0)
        goto error;
    }
    
    /* wait response */
    status= wait_response(client, &ptr, &line);

    if (debug_news)
      g_message("selected article: %s", line);

    /* the article number */
    *next= atol(strtok(ptr, " "));
    g_free(line);

    if (status == 421) /* end of articles */
      break;
    else if (status < 0)
    {
      err= status;
      goto error;
    }
    else if (status != 223) /* some other error, unexpected response */
      break;

    /* check if this is the last one we want for this round */
    if (last > first && *next > last)
      break;

    /* request header */
    if ((err= send_command(client, "HEAD")) < 0)
      goto error;

    /* wait response */
    status= wait_response(client, &ptr, &line);

    if (debug_news)
      g_message("request header: %s", line);

    if (status == 221) /* article ok, header to come */
    {
      if (headers->articles_num == alloced)
      {
        alloced += 32;
        headers->articles= g_realloc(headers->articles,
                                    sizeof(MNNTPArticleHeader)*alloced);
      }

      article= headers->articles+headers->articles_num++;

      memset(article, 0, sizeof(MNNTPArticleHeader));
      
      /* get msg index and id */
      ptr= strtok(ptr, " ");
      article->index= atol(ptr);
      ptr= strtok(NULL, " ");
      article->msg_id= g_strdup(ptr);
      g_free(line);
    }
    else
    {
      g_free(line);
      /* error occurred, return will be incomplete! */
      break;
    }


    lines_alloced= 0;
    article_done= 0;

    while (!article_done)
    {
      /* read in the article header */
      while ((line= mnet_get_line(client->client)))
      {
        if (strcmp(line, ".")==0)
        {
          article_done= 1;
          break;
        }
        else
        {
          if (article->lines_num >= lines_alloced)
          {
            lines_alloced+= 32;
            article->lines= g_realloc(article->lines, sizeof(char*)*lines_alloced);
          }
          
          article->lines[article->lines_num++]= line;
        }
      }

      if (!article_done)
      {
        if ((err= mnet_wait_ready(client->client, -1)) < 0)
          goto error;
       
        if ((err= mnet_read_data(client->client)) < 0)
          goto error;
      }
    }

    if (debug_news)
      g_message("fetched %i header lines", article->lines_num);
    
    if (lines_alloced > article->lines_num)
      article->lines= g_realloc(article->lines,
                                sizeof(char*)*article->lines_num);
  }

  if (alloced > headers->articles_num)
    headers->articles= g_realloc(headers->articles, 
                                 sizeof(MNNTPArticleHeader)*headers->articles_num);

  return headers;
  
error:
  TOERROR(err, *error);
  mnet_free_article_header_list(headers);
  return NULL;
}
#endif


MNNTPGroupIndex *mnet_nntp_list_articles(MNetNNTPClient *client,
                                         const char *group,
                                         unsigned int first,
                                         unsigned int last,
                                         MNNTPError *error)
{
  MNNTPGroupIndex *headers= NULL;
  int alloced=0;
  char *line;
  int status;
  char *cmd;
  int err;

  *error= MNNTPNoError;

  g_return_val_if_fail(client->client->state == MSConnected, NULL);

  *error= change_group(client, group);
  if (*error != MNNTPNoError)
  {
    return NULL;
  }

  /* request range */
  cmd= g_strdup_printf("XOVER %i-%i", first, last);
  if ((err= send_command(client, cmd)) < 0)
  {
    g_free(cmd);
    goto error;
  }
  g_free(cmd);

  /* wait response */
  status= wait_response(client, NULL, NULL);
  if (status < 0)
  {
    err= status;
    goto error;
  }
  else if (status != 224)
  {
    err= status;
    goto error;
  }

  alloced= last-first+1;
  headers= g_malloc0(sizeof(MNNTPGroupIndex));
  headers->articles= g_malloc0(sizeof(MNNTPArticleInfo)*alloced);

  for (;;)
  {
    char *ptr;
    unsigned int i;

    if ((line= mnet_get_line(client->client)))
    {
      if (strcmp(line, ".")==0)
      {
        g_free(line);
        break;
      }
      
      if (headers->articles_num >= alloced)
      {
        /* more data than expected!? */
        break;
      }
      
      i= headers->articles_num;
      
      ptr= strtok(line, "\t");
      headers->articles[i].index= strtoul(ptr, NULL, 10);
      ptr= strtok(NULL, "\t");
      headers->articles[i].subject= g_strdup(ptr);
      ptr= strtok(NULL, "\t");
      headers->articles[i].from= g_strdup(ptr);
      ptr= strtok(NULL, "\t");
      headers->articles[i].date= g_strdup(ptr);
      ptr= strtok(NULL, "\t");
      headers->articles[i].msg_id= g_strdup(ptr);
      ptr= strtok(NULL, "\t");
      headers->articles[i].references= g_strdup(ptr);
      /* ignore Bytes: Lines: and Xref:full */

      g_free(line);
      headers->articles_num++;
    }
    if (!line)
    {
      if ((err= mnet_wait_ready(client->client, -1)) < 0)
        break;
      if ((err= mnet_read_data(client->client)) < 0)
        break;
    }
  }

  if (alloced > headers->articles_num)
    headers->articles= g_realloc(headers->articles, 
                                 sizeof(MNNTPArticleInfo)*headers->articles_num);

  return headers;

error:
  TOERROR(err, *error);
  if (headers)
    mnet_free_group_index(headers);
  return NULL;
}



int mnet_free_group_index(MNNTPGroupIndex *index)
{
  unsigned int i;
  for (i= 0; i < index->articles_num; i++)
  {
    g_free(index->articles[i].msg_id);
    g_free(index->articles[i].subject);
    g_free(index->articles[i].from);
    g_free(index->articles[i].date);
    g_free(index->articles[i].references);
  }
  g_free(index->articles);
  g_free(index);
  return 0;
}


static int get_lines(MNetNNTPClient *client, char ***lines_ret,
                        unsigned int *count)
{
  unsigned int lines_alloced= 0;
  unsigned int article_done= 0;
  char **lines= NULL;
  unsigned int lines_num= 0;
  char *line;
  int rc= 0;

  while (!article_done)
  {
    /* read in the article header */
    while ((line= mnet_get_line(client->client)))
    {
      if (strcmp(line, ".")==0)
      {
        article_done= 1;
        break;
      }
      else
      {
        if (lines_num >= lines_alloced)
        {
          lines_alloced+= 32;
          lines= g_realloc(lines, sizeof(char*)*lines_alloced);
        }
        lines[lines_num++]= line;
      }
    }
    
    if (!article_done)
    {
      if ((rc= mnet_wait_ready(client->client, -1)) < 0)
        goto error;
      
      if ((rc= mnet_read_data(client->client)) < 0)
        goto error;
      }
  }

  if (lines_alloced > lines_num)
    lines= g_realloc(lines, sizeof(char*)*lines_num);

  *count= lines_num;
  *lines_ret= lines;

error:
  return rc;
}


MNNTPArticleBody *mnet_nntp_get_article(MNetNNTPClient *client,
                                        const char *group,
                                        unsigned int id,
                                        MNNTPError *error)
{
  MNNTPArticleBody *body;
  char *line;
  int status;
  char *cmd;
  char *ptr;
  int err;
  
  *error= MNNTPNoError;

  g_return_val_if_fail(client->client->state == MSConnected, NULL);

  *error= change_group(client, group);
  if (*error != MNNTPNoError)
  {
    return NULL;
  }

  /* request article */
  cmd= g_strdup_printf("BODY %i", id);
  if (send_command(client, cmd) < 0)
  {
    TOERROR(err, *error);
    g_free(cmd);
    return NULL;
  }
  g_free(cmd);

  /* wait response */
  status= wait_response(client, &ptr, &line);
  if (status != 222)
  {
    TOERROR(err, *error);
    g_free(line);
    return NULL;
  }

  body= g_malloc0(sizeof(MNNTPArticleBody));
  /* get msg index and id */
  ptr= strtok(ptr, " ");
  body->index= atol(ptr);
  ptr= strtok(NULL, " ");
  body->msg_id= g_strdup(ptr);
  g_free(line);

  /* get body */
  err= get_lines(client, &body->lines, &body->lines_num);
  if (err < 0)
    goto error;

  return body;

error:
  TOERROR(err, *error);
  mnet_free_article_body(body);
  return NULL;
}


MNNTPArticleHeader *mnet_nntp_get_header(MNetNNTPClient *client,
                                        const char *group,
                                        unsigned int id,
                                        MNNTPError *error)
{
  MNNTPArticleHeader *header;
  char *line;
  int status;
  char *cmd;
  char *ptr;
  int err;
  
  *error= MNNTPNoError;

  g_return_val_if_fail(client->client->state == MSConnected, NULL);

  *error= change_group(client, group);
  if (*error != MNNTPNoError)
  {
    return NULL;
  }


  /* request article */
  cmd= g_strdup_printf("HEAD %i", id);
  if (send_command(client, cmd) < 0)
  {
    TOERROR(err, *error);
    g_free(cmd);
    return NULL;
  }
  g_free(cmd);

  /* wait response */
  status= wait_response(client, &ptr, &line);
  if (status != 221)
  {
    TOERROR(err, *error);
    g_free(line);
    return NULL;
  }

  header= g_malloc0(sizeof(MNNTPArticleHeader));
  /* get msg index and id */
  ptr= strtok(ptr, " ");
  header->index= atol(ptr);
  ptr= strtok(NULL, " ");
  header->msg_id= g_strdup(ptr);
  g_free(line);

  /* get header */
  err= get_lines(client, &header->lines, &header->lines_num);
  if (err < 0)
    goto error;

  return header;

error:
  TOERROR(err, *error);
  mnet_free_article_header(header);
  return NULL;
}


int mnet_free_article_body(MNNTPArticleBody *body)
{
  unsigned int i;
  for (i= 0; i < body->lines_num; i++)
    g_free(body->lines[i]);
  g_free(body->lines);
  g_free(body);
  
  return 0;
}


int mnet_free_article_header(MNNTPArticleHeader *header)
{
  unsigned int i;
  for (i= 0; i < header->lines_num; i++)
    g_free(header->lines[i]);
  g_free(header->lines);
  g_free(header);
  
  return 0;
}


MNNTPError mnet_nntp_post_article(MNetNNTPClient *client,
                                  const char *article)
{
  const char *ptr, *end;
  int status;
  int err;
  char *msgid;
  char *line;
  g_return_val_if_fail(client->client->state == MSConnected, -1);
  g_return_val_if_fail(mnet_get_line(client->client)==NULL, -1);

  if ((err= send_command(client, "POST")) < 0)
    goto error;
  
  if ((err= wait_response(client, &msgid, &line))!=340)
  {
    g_free(line);
    return MNNTPProtocolError;
  }
  /* if we got a recommended msg id, use it */
  msgid= strchr(msgid, '<');
  if (msgid)
  {
    char *tmp= g_strdup_printf("Message-ID: %s", msgid);
    if ((err= mnet_send_line(client->client, tmp)) < 0)
    {
      g_free(line);
      g_free(tmp);
      goto error;
    }
    g_free(tmp);
  }
  else
  {
    char name[1024];
    socklen_t len= sizeof(name);
    
    if (getpeername(client->client->socket, (struct sockaddr*)name, &len) < 0)
      strcpy(name, "localhost");
    else
      name[len]= 0;
    
    char *tmp= g_strdup_printf("Message-ID: <%x#%x@%s>", 
                               (unsigned int)time(NULL), g_random_int(), 
                               name);
    if ((err=mnet_send_line(client->client, tmp)) < 0)
    {
      g_free(line);
      g_free(tmp);
      goto error;
    }
    g_free(tmp);
  }
  g_free(line);

  ptr= article;
  while ((end= strchr(ptr,'\n')))
  {
    end++;
    if (*ptr=='.')
    {
      if ((err= mnet_send(client->client, ".", 1)) < 0) /* double starting . */
        goto error;
    }
    if ((err= mnet_send(client->client, ptr, end-ptr)) < 0)
      goto error;

    if ((err= mnet_flush_data(client->client)) < 0)
      goto error;

    ptr= end;
  }
  /* terminating dot */
  if ((err= mnet_send_line(client->client, ".")) < 0
      || (err= mnet_flush_data(client->client)) < 0)
    goto error;

  /* wait until everything is flushed */
  while (client->client->out_buffer_len > 0)
  {
    status= mnet_wait_ready(client->client, -1);
    if (status < 0)
    {
      err= status;
      goto error;
    }
    if (status&MWReadOK)
    {
      char *tmp;
      if ((err= mnet_read_data(client->client)) < 0)
        goto error;
      tmp= mnet_get_line(client->client);
      if (tmp)
      {
        g_warning("While posting through nntp got: %s", tmp);
        return MNNTPProtocolError;
      }
    }
    if ((err= mnet_flush_data(client->client)) < 0)
      goto error;
  }

  status= wait_response(client, NULL, NULL);
  if (status == 240)
    return MNNTPNoError;

  if (status > 0)
    return MNNTPProtocolError;

  err= status;

error:
  switch (err)
  {
  case -2:
    return MNNTPOutOfMemory;
  case -1:
    return MNNTPNetError;
  default:
    return MNNTPProtocolError;
  }
}

                           
