/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_texture.cpp 
 * @brief Implementation of a texture class.
 * 
 */

#ifdef _WINDOWS
  #include <windows.h>
#else
#endif // ifdef _WINDOWS

#include <gl/gl.h>
#include <gl/glu.h>
#include <math.h>

#include "myx_gc_texture.h"
#include "myx_gc_canvas.h"
#include "myx_gc_utilities.h"

//----------------- CGCTexture -----------------------------------------------------------------------------------------

CGCTexture::CGCTexture(CTextureManager* Controller, const string& Filename, const string& ID, GLenum WrapModeS, GLenum WrapModeT, 
                       GLenum MinFilter, GLenum MagFilter, int Dimensions, GLenum TextureMode)
{
  FManager = Controller;
  FLoaded = false;
  FName = ID;
  FFilename = Filename;
  FWrapModeS = WrapModeS;
  FWrapModeT = WrapModeT;
  FMinFilter = MinFilter;
  FMagFilter = MagFilter;
  FTarget = Dimensions == 1 ? GL_TEXTURE_1D : GL_TEXTURE_2D;
  FMode = TextureMode;

  glGenTextures(1, &FHandle);
}

//----------------------------------------------------------------------------------------------------------------------

CGCTexture::~CGCTexture(void)
{
  glDeleteTextures(1, &FHandle);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 *  Rounds the given value up to the next power of two boundary.
 *
 *  @param Value The value to round up.
 *  @return Returns the rounded value.
 */
int RoundUpToPowerOf2(int Value)
{
#define LOG2(value) (log(value) / log(2.0))

  double LogTwo = LOG2((double) Value);
  if (floor(LogTwo) < LogTwo)
    return (int) floor(pow(2.0, floor(LogTwo) + 1));
  else
    return Value;

#undef LOG2
}

//----------------------------------------------------------------------------------------------------------------------

/**
 *  Delay loads texture data. Called from ActivateTexture, that is, when the texture is used the first time.
 */
void CGCTexture::LoadTexture(void)
{
  FLoaded = true;
  TImage* Image = LoadPNG(FFilename);

  if (Image != NULL)
  {
    int ActualWidth = RoundUpToPowerOf2(Image->Width);
    int ActualHeight = RoundUpToPowerOf2(Image->Height);
    unsigned char* Buffer;
    bool FreeBuffer = false;

    GLenum Format = GL_RGB;
    switch (Image->ColorType)
    {
      case COLOR_TYPE_PALETTE:
        {
          Format = GL_COLOR_INDEX;
          break;
        };
      case COLOR_TYPE_GRAY:
        {
          Format = GL_LUMINANCE;
          break;
        };
      case COLOR_TYPE_GRAY_ALPHA:
        {
          Format = GL_LUMINANCE_ALPHA;
          break;
        };
      case COLOR_TYPE_RGB:
        {
          Format = GL_RGB;
          break;
        };
      case COLOR_TYPE_RGB_ALPHA:
        {
          Format = GL_RGBA;
          break;
        };
    };

    if (ActualWidth != Image->Width || ActualHeight != Image->Height)
    {
      FreeBuffer = true;
      Buffer = (unsigned char*) malloc(ActualWidth * ActualHeight * Image->Channels);
      gluScaleImage(Format, Image->Width, Image->Height, GL_UNSIGNED_BYTE, Image->Data, ActualWidth, ActualHeight, 
        GL_UNSIGNED_BYTE, Buffer);
    }
    else
      Buffer = Image->Data;
                 
    if (FTarget == GL_TEXTURE_1D)
    {
      if ((FMinFilter == GL_NEAREST) || (FMinFilter == GL_LINEAR))
        glTexImage1D(GL_TEXTURE_1D, 0, Format, ActualWidth, 0, Format, GL_UNSIGNED_BYTE, Buffer);
      else
        gluBuild1DMipmaps(GL_TEXTURE_1D, Format, ActualWidth, Format, GL_UNSIGNED_BYTE, Buffer);
    }
    else
    {
      if ((FMinFilter == GL_NEAREST) || (FMinFilter == GL_LINEAR))
        glTexImage2D(GL_TEXTURE_2D, 0, Format, ActualWidth, ActualHeight, 0, Format, GL_UNSIGNED_BYTE, Buffer);
      else
        gluBuild2DMipmaps(GL_TEXTURE_2D, Format, ActualWidth, ActualHeight, Format, GL_UNSIGNED_BYTE, Buffer);
    };

    if (Image != NULL)
      FreeImage(Image);
    if (FreeBuffer)
      free(Buffer);
  };
}

//----------------------------------------------------------------------------------------------------------------------

// Activates this texture in OpenGL so all following vertex definitions are textured using this texture.
// If the texture has not been loaded yet it will be done now. Additionally, texture mode is enabled in OpenGL.
void CGCTexture::ActivateTexture(void)
{                                                          
  glEnable(FTarget);
  glBindTexture(FTarget, FHandle); 

  if (!FLoaded)
    LoadTexture();

  glTexParameteri(FTarget, GL_TEXTURE_WRAP_S, FWrapModeS);
  if (FTarget == GL_TEXTURE_2D)
    glTexParameteri(FTarget, GL_TEXTURE_WRAP_T, FWrapModeT);
	glTexParameteri(FTarget, GL_TEXTURE_MIN_FILTER, FMinFilter);
	glTexParameteri(FTarget, GL_TEXTURE_MAG_FILTER, FMagFilter);

  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, FMode);
}

//----------------------------------------------------------------------------------------------------------------------

// Deactivates this texture and the texture mode in OpenGL.
void CGCTexture::DeactivateTexture(void)
{
  glDisable(FTarget);
  glBindTexture(FTarget, 0);
}

//----------------- CTextureManager ------------------------------------------------------------------------------------

CTextureManager InternalTextureManager; // Singleton texture manager instance.

CTextureManager* TextureManager()
{
  return &InternalTextureManager;
}

//----------------------------------------------------------------------------------------------------------------------

CTextureManager::~CTextureManager(void)
{
  ClearTextures();
}

//----------------------------------------------------------------------------------------------------------------------

// Looks throught the textures and attempts to find one with the given name.
CGCTexture* CTextureManager::FindTexture(const string& Name)
{
  CTextureIterator Iterator = FTextures.find(Name);
  if (Iterator == FTextures.end())
    return NULL;
  else
    return Iterator->second;
}

//----------------------------------------------------------------------------------------------------------------------

// Creates a new texture entry and adds the entry to the texture list.
// No image data is loaded yet. This will happen when the texture is used the first time.
void CTextureManager::CreateTextureEntry(const string& Filename, const string& ID, const string& WrapH, const string& WrapV, 
                                         const string& MinificationFilterStr, const string& MagnificationFilterStr,
                                         int Dimensions, const string& Mode)
{
  // These both constants are only available with OpenGL 1.2 or up and are not defined in the standard gl.h header file.
  #define GL_CLAMP_TO_BORDER 0x812D
  #define GL_CLAMP_TO_EDGE   0x812F

  static stdext::hash_map<string, GLenum> Mapper;
  stdext::hash_map<string, GLenum>::const_iterator Iterator;

  // Fill our lookup table first if not yet done.
  if (Mapper.size() == 0)
  {
    Mapper["clamp"] = GL_CLAMP;
    Mapper["clamp-to-border"] = GL_CLAMP_TO_BORDER;
    Mapper["clamp-to-edge"] = GL_CLAMP_TO_EDGE;
    Mapper["repeat"] = GL_REPEAT;
    Mapper["nearest"] = GL_NEAREST;
    Mapper["linear"] = GL_LINEAR;
    Mapper["nearest-mipmap-nearest"] = GL_NEAREST_MIPMAP_NEAREST;
    Mapper["linear-mipmap-nearest"] = GL_LINEAR_MIPMAP_NEAREST;
    Mapper["nearest-mipmap-linear"] = GL_NEAREST_MIPMAP_LINEAR;
    Mapper["linear-mipmap-linear"] = GL_LINEAR_MIPMAP_LINEAR;
    Mapper["decal"] = GL_DECAL;
    Mapper["modulate"] = GL_MODULATE;
    Mapper["blend"] = GL_BLEND;
    Mapper["replace"] = GL_REPLACE;
  };

  GLenum WrapModeS = GL_CLAMP;
  Iterator = Mapper.find(WrapH);
  if (Iterator != Mapper.end())
    WrapModeS = Iterator->second;

  GLenum WrapModeT = GL_CLAMP;
  Iterator = Mapper.find(WrapV);
  if (Iterator != Mapper.end())
    WrapModeT = Iterator->second;

  GLenum MinFilter = GL_NEAREST;
  Iterator = Mapper.find(MinificationFilterStr);
  if (Iterator != Mapper.end())
    MinFilter = Iterator->second;

  GLenum MagFilter = GL_NEAREST;
  Iterator = Mapper.find(MagnificationFilterStr);
  if (Iterator != Mapper.end())
    MagFilter = Iterator->second;

  GLenum TextureMode = GL_DECAL;
  Iterator = Mapper.find(Mode);
  if (Iterator != Mapper.end())
    TextureMode = Iterator->second;

  FTextures[ID] = new CGCTexture(this, Filename, ID, WrapModeS, WrapModeT, MinFilter, MagFilter, Dimensions, TextureMode);
}

//----------------------------------------------------------------------------------------------------------------------

void CTextureManager::ClearTextures(void)
{
  for (CTextureIterator Iterator = FTextures.begin(); Iterator != FTextures.end(); Iterator++)
    delete Iterator->second;

  FTextures.clear();
}

//----------------------------------------------------------------------------------------------------------------------

