/* Copyright (C) 1995-2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   There are special exceptions to the terms and conditions of the GPL as it
   is applied to this software. View the full text of the exception in file
   EXCEPTIONS in the directory of this software distribution.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/***************************************************************************
 * MISC.C								   *
 *									   *
 * @description: Handling misc routines					   *
 *									   *
 * @author     : MySQL AB(monty@mysql.com, venu@mysql.com)		   *
 * @date       : 2001-Aug-15						   *
 * @product    : myodbc3						   *
 *									   *
****************************************************************************/

#include "myodbc3.h"

#if defined(_UNIX_) && !defined(HAVE_SQLGETPRIVATEPROFILESTRING)

#include <pwd.h>

static void open_ini_files(FILE **sys_file, FILE **user_file)
{
  char *home_dir, *odbcini;
  FILE *system_ini= NULL, *user_ini= NULL;

  if ((odbcini= getenv("ODBCINI")))
    user_ini= fopen(odbcini,"r");

  /* only look in $HOME/.odbc.ini if $ODBCINI wasn't there */
  if (!user_ini)
  {
    /* find the homedir */
    if (!(home_dir= getenv("HOME")))
    {
      struct passwd *pwd;
      /*
	WARNING:
	The following is clearly not threadsafe in
        most environments
      */
      if ((pwd= getpwuid(getuid())))
	home_dir= pwd->pw_dir;
    }

    if (home_dir && home_dir[0])
    {
      char tmp[1024];
      strxmov(tmp, home_dir, "/.odbc.ini", NullS);
      user_ini= fopen(tmp,"r");
    }
  }

  system_ini= fopen(SYSTEM_ODBC_INI,"r");
  *sys_file= system_ini;
  *user_file= user_ini;
}


static void close_ini_files(FILE *sys_file, FILE *user_file)
{
  if (sys_file)
    fclose(sys_file);
  if (user_file)
    fclose(user_file);
}

/* skip space */

static inline char *skip_space(char *pos)
{
  while (isspace(*pos))
    pos++;
  return pos;
}


/* skip end space;  Terminate string with \0 */

static char *skip_end_space(char *pos, char *end)
{
  while (end > pos && isspace(end[-1]))
    end--;
  *end= 0;
  return end;
}


/*
  Searches for a section, returns non-zero on success

  If successeful, positions on the line after the section
  name.
*/

static int
find_section(FILE *ini, const char *section)
{
  char buf[1024];

  while (fgets(buf,sizeof(buf)-1,ini))
  {
    char *pos= skip_space(buf);
    char *end;

    if (!*pos || *pos == ';' || *pos =='#')
      continue;					/* comment or empty line */

    if (*pos == '[')				/* Found new section */
    {
      pos= skip_space(pos+1);
      end= strchr(pos+1, ']');
      if (end)
      {
	skip_end_space(pos, end);		/* Terminates pos */
	if (!myodbc_strcasecmp(pos, section))
	  return 1;				/* Found section */
      }
    }
  }
  return 0;
}


/*
  Returns NULL if not found, otherwise result must be freed with my_free()
  Expects ini to be positioned at the beginning of a section
*/

static char *
find_key(FILE *ini, const char *key)
{
  char buf[1024], *value= NullS;

  /* if any of those starts a line, the line is ignored */
  static const char *skipchars= ";#=\r\n";

  while (fgets(buf,sizeof(buf)-1,ini))
  {
    char *tmp, *pos= buf;

    pos= skip_space(pos);
    if (!*pos || strchr(skipchars,*pos))
      continue;					/* comment or empty line */
    if (*pos == '[')
      break;					/* new section */

    if ((tmp= strchr(pos,'=')))
    {
      skip_end_space(pos, tmp);			/* Will \0 terminate pos */
      if (!myodbc_strcasecmp(pos, key))
      {
	char *end;
	tmp= skip_space(tmp+1);			/* Skip '=' and spaces */
	end= skip_end_space(tmp, strend(tmp));
	value= my_strdup_with_length(tmp, (uint) (end-tmp), MYF(0));
	break;
      }
    }
  }
  return value;
}


/*
  First look for key in the user ini file, then in system.
  Do not mix up values from both.

  NOTE: key must be freed with my_free()
*/

static char *
find_key_by_dsn(const char *dsn, const char *key)
{
  FILE *sys;
  FILE *user;
  char *ret= NULL;

  open_ini_files(&sys,&user);

  if (user && find_section(user,dsn))
    ret= find_key(user,key);
  else if (sys && find_section(sys,dsn))
    ret= find_key(sys,key);

  close_ini_files(sys,user);
  return ret;
}



int _myodbc_SQLGetPrivateProfileString(char *section, char *entry, char *def_value,
				       char *buf,  int buf_len,
				       char *filename)
{
  char *value, *src;
  int length= -1;

  if (buf == NULL || buf_len <= 1 || section == NULL || entry== NULL)
    return -1;					/* no need to bother */

  if (!strcmp(filename,"ODBC.INI"))
  {
    value= find_key_by_dsn(section,entry);
    src= value ? value : def_value;

    length= 0;
    buf[0]= 0;
    if (src)
      length= (int) (strmake(buf, src, buf_len-1) - buf);
    my_free(value, MYF(MY_ALLOW_ZERO_PTR));
  }
  return length;
}

#endif /* defined(_UNIX_) && !defined(HAVE_SQLGETPRIVATEPROFILESTRING) */
