// pvocmodifier.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "controller.h"
#include "envelope.h"
#include "pvocdata.h"
#include "pvocmodifier.h"
#include "request.h"
#include "valuerequester.h"

class SingleFactorRequester : public ValueRequester<double> {
	friend class SpectrumTransposer;
	friend class SpectrumStretcher;
protected:
	SingleFactorRequester(const char* title, const char* label,
	                      SingleFactorModifier* s);
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
protected:
	SingleFactorModifier* client;	                     
	double upperFreq;
	double lowerFreq;
	ChoiceValue useEnvelope;
};

SingleFactorRequester::SingleFactorRequester(const char* title,
                                             const char* label,
                                             SingleFactorModifier* s)
	: ValueRequester<double>(title, label, s->factor), client(s),
	  upperFreq(client->sampRate()/2.0), lowerFreq(0), useEnvelope(False) {}

void
SingleFactorRequester::configureRequest(Request* request) {
	ValueRequester<double>::configureRequest(request);
	Range allFreqs(0.0, client->sampRate()/2.0);
	request->appendValue("Modify Frequencies From:", &lowerFreq, allFreqs);
	request->appendValue("To:", &upperFreq, allFreqs);
	request->appendChoice("Use envelope for factor", "|No|Yes|",
	                      &useEnvelope, true);
}

boolean
SingleFactorRequester::confirmValues() {
	if(lowerFreq > upperFreq) {
		Application::alert("Invalid range:  upper must be > lower.");
		return false;
	}
	else {
		client->lowerBand = client->pvTarget()->getBandNumber(lowerFreq);
		client->upperBand = client->pvTarget()->getBandNumber(upperFreq);
	}
	if(useEnvelope == True) {
		if(client->envelope == nil) {
			Application::alert("No envelope is currently selected.");
			return false;
		}
	}
	else {
		Resource::unref(client->envelope);
		client->envelope = nil;		// to make sure it is not used
	}
	return client->checkFactor();
}

//********

SingleFactorModifier::SingleFactorModifier(Data* data, double facDefault)
		: PvocModifier(data), maxBand(pvTarget()->nBands() - 1),
		  envelope(nil), factor(facDefault) {}

SingleFactorModifier::~SingleFactorModifier() {
	Resource::unref(envelope);
}

Status
SingleFactorModifier::configure(Controller *c) {
	envelope = (Envelope *) c->findSelection();
	return PvocModifier::configure(c);
}

void
SingleFactorModifier::initialize() {
	if(envelope != nil) {
		envelope->ref();
		if(!envelope->isA(EVP_Data)) {
			Envelope* evp = new Envelope(envelope->length());
			evp->copyFrom(envelope);
			Resource::unref(envelope);
			envelope = evp;
			envelope->ref();
		}
		envelope->setMappingLength(target()->length());
	}
	Super::initialize();
}

double
SingleFactorModifier::currentFactor() {
	return (envelope != nil) ?
		1.0 + envelope->next() * (factor - 1.0) :
		factor;
}

////////

SpectrumTransposer::SpectrumTransposer(Data* data)
	: SingleFactorModifier(data, 2.0) {}

Requester *
SpectrumTransposer::createRequester() {
	return new SingleFactorRequester(
		"Harmonically Shift (Transpose) Portion of Spectrum:",
		"Transposition factor:",
		this);
}

int
SpectrumTransposer::modifyFrame(Data* frame) {
	int fromBand = 0, eraseBand = 0;
	double factor = currentFactor();
	if(factor > 1.0) {			// moving spectrum towards nyquist
		int upperTargetBand = min(maxBand, iround(upperBand * factor));
		int intoBand;
		for(intoBand = upperTargetBand; intoBand >= 0 ; intoBand--) {
			if((fromBand = iround(intoBand / factor)) < lowerBand)
				break;
			frame->set(frame->get(fromBand, 0), intoBand, 0);
			frame->set(factor * frame->get(fromBand, 1), intoBand, 1);
		}
		// zero out portion of old range that doesnt overlap new
		for(eraseBand = min(upperBand, intoBand); eraseBand >= lowerBand;
				eraseBand--) {
			frame->set(0.0, eraseBand, 0);
			frame->set(0.0, eraseBand, 1);
		}
	}
	else if(factor < 1.0) {		// moving spectrum towards zero
		int intoBand;
		for(intoBand = lowerBand; intoBand <= upperBand; intoBand++) {
			if((fromBand = iround(intoBand / factor)) > maxBand)
				break;
			frame->set(frame->get(fromBand, 0), intoBand, 0);
			frame->set(factor * frame->get(fromBand, 1), intoBand, 1);
		}
		// zero out portion of old range that doesnt overlap new
		for(eraseBand = max(lowerBand, intoBand); eraseBand <= upperBand;
				eraseBand++) {
			frame->set(0.0, eraseBand, 0);
			frame->set(0.0, eraseBand, 1);
		}
	}
	return true;
}

boolean
SpectrumTransposer::checkFactor() {
	return true;
}

////////

SpectrumStretcher::SpectrumStretcher(Data* data)
	: SingleFactorModifier(data, 2.0) {}

Requester *
SpectrumStretcher::createRequester() {
	return new SingleFactorRequester("Stretch/Shrink Portion of Spectrum:",
	                                 "Stretch factor:", this);
}

int
SpectrumStretcher::modifyFrame(Data* frame) {
	int fromBand = 0, eraseBand = 0;
	double factor = currentFactor();
	if(factor >= 1.0) {		// stretching spectrum upwards
		int upperTargetBand = min(maxBand, iround(upperBand * factor));
		int totalBands = upperTargetBand - lowerBand;
		int intoBand;
		for(intoBand = upperTargetBand;  intoBand >= 0; intoBand--) {
			double fac = ((factor-1.)*(intoBand-lowerBand)/totalBands) + 1;
			if((fromBand = iround(intoBand / fac)) < lowerBand)
				break;
			frame->set(frame->get(fromBand, 0), intoBand, 0);
			frame->set(fac * frame->get(fromBand, 1), intoBand, 1);
		}
		// zero out portion of old range that doesnt overlap new
		for(eraseBand = min(upperBand, intoBand); eraseBand >= lowerBand;
				eraseBand--) {
			frame->set(0.0, eraseBand, 0);
			frame->set(0.0, eraseBand, 1);
		}
	}
	else {					// compressing spectrum downwards
		int upperTargetBand = iround(upperBand * factor);
		int totalBands = upperTargetBand - lowerBand;
		int intoBand;
		for(intoBand = lowerBand; intoBand <= upperBand; intoBand++) {
			double fac = 1. - ((1.-factor)*(intoBand-lowerBand)/totalBands);
			if((fromBand = iround(intoBand / fac)) > maxBand)
				break;
			frame->set(frame->get(fromBand, 0), intoBand, 0);
			frame->set(fac * frame->get(fromBand, 1), intoBand, 1);
		}
		// zero out portion of old range that doesnt overlap new
		for(eraseBand = max(lowerBand, intoBand); eraseBand <= upperBand;
				eraseBand++) {
			frame->set(0.0, eraseBand, 0);
			frame->set(0.0, eraseBand, 1);
		}
	}
	return true;
}

boolean
SpectrumStretcher::checkFactor() {
	if(iround(upperBand * factor) <= lowerBand) {
		Application::alert("Factor incompatible with given frequency range.");
		return false;
	}
	return true;
}
