// header.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// This class contains the interface, implementation and state common to all
// Header subclasses.  A Header object is responsible for reading, parsing, and
// validating the on-disk header information for each data file, and well as
// performing the reverse operation at save time.  Header subclass instances
// are created within the Data subclass objects, and in the current implemen-
// tation, they only exist during the read and write operations.  In future
// versions, they may be kept as a permenant member.

#ifndef HEADER_H
#ifdef __GNUG__
#pragma interface
#endif
#define HEADER_H

#include <InterViews/resource.h>
#include "localdefs.h"
#include "requester.h"

#define MPEG1_MASK	((int)0xfff00000)
#define MPEG1_MAGIC	((int)0xfff00000) /* MPEG1 header with mask */

class Comment;
class Controller;
class DataFile;
class Data;
class Request;

class Header : public Resource {
public:
	enum ByteOrder { Little = 0x1, Big = 0x2 };
	static int readMagicNumber(DataFile *);
public:
	Header(int magic=0, DataType type=NoData, int chans=0); 
	virtual ~Header();
	virtual boolean configure(Controller *);
	virtual int read(DataFile *);
	virtual int write(DataFile *);
	virtual int magic() const { return magic_number; }
	int dataOffset() const { return data_offset; }
	DataType dataType() const { return DataType(data_type); }
	int dataSize() const { return data_size; }
	int nChans() const { return nchans; }
	void setComment(const char *com);
	void setComment(Comment *com);
	Comment *getComment() { return comment; }
	void setDataSize(int size) { data_size = size; }
protected:
	void forceByteSwapping(boolean flag) { forceSwapped = flag; }
	virtual boolean isLittleEndian() { return false; }
	boolean needsByteSwapping();
	boolean setDataOffset(int offset) { data_offset = offset; return true;}
	virtual int seekInFile(DataFile*, int);
	int sampleSize();
	boolean isRaw() { return raw; }
	void setRaw() { raw = true; }
	virtual int diskHeaderSize() { return 0; }
	virtual int diskHeaderCommentSize() { return 0; }
	virtual int diskHeaderInfoSize() {
		return diskHeaderSize() - diskHeaderCommentSize();
	}
	int checkMagic(DataFile *);
	virtual int readMagic(DataFile *);
	virtual boolean isMagic() { return true; }
	virtual boolean magicIsSwapped() { return false; }
	virtual int secondsToBytes(double) { return 0; }
	virtual int checkHeader() { return true; }
	virtual int readInfo(DataFile *) { return true; }
	virtual int readComment(DataFile *);
	virtual int writeInfo(DataFile *) { return true; }
	virtual int writeComment(DataFile *);
	virtual int commentLength();
	virtual const char* magicError() { return ""; }
	virtual void reset() {}
protected:
	class ConfigRequester : virtual public TitledRequester {
	public:
		ConfigRequester(const char* title, Header* header);
	protected:
		redefined void configureRequest(Request *);
		redefined boolean confirmValues();
		Header* myHeader;
	private:
		enum ByteState { NeedSwapping = 0x1, DontNeedSwapping = 0x2 };
		unsigned short byteState;
	};
	friend class ConfigRequester;
	virtual Requester* configRequester() { return nil; }
protected:
	int magic_number;
	int data_offset;
	int data_size;
	unsigned short data_type;
	int nchans;
	Comment *comment;
private:
	boolean forceSwapped;	// used to override header default during raw read
	boolean raw;		// if raw is true, header will not be read from file
	void initialize();
};

// abstract "intermediate" base class

class FrameDataHeader : public Header {
public:
	virtual int sampleRate() { return samprate; }
	virtual double frameRate() { return framerate; }
	virtual double duration() { return dur; }
protected:
	FrameDataHeader(int magic, DataType type) : Header(magic, type, 0),
		framerate(0), samprate(0), dur(0) {}
	FrameDataHeader(int magic, DataType type, int chans, double frate,
			int srate, double duration=0)
		: Header(magic, type, chans),
		framerate(frate), samprate(srate), dur(duration) {}
	redefined int secondsToBytes(double);
protected:
	class FrameConfigRequester : public Header::ConfigRequester {
	public:
		FrameConfigRequester(const char* title, Header* h)
			: ConfigRequester(title, h) {}
	protected:
		redefined void configureRequest(Request *);
	};
	friend void FrameConfigRequester::configureRequest(Request *);
	double framerate;
	int samprate;
	double dur;
};


class PCHHeader : public FrameDataHeader {
	typedef FrameDataHeader Super;
	enum { PCH_MAGIC = 900 };
public:
	PCHHeader();
	PCHHeader(double framert, int fsize, int srate, double dur=0);
	virtual int frameSize() { return framesize; }
protected:
	redefined int readMagic(DataFile *);
	redefined boolean isMagic() { return true; }
	redefined int readInfo(DataFile *);
private:
	int framesize;
};

class FFTHeader : public FrameDataHeader {
	typedef FrameDataHeader Super;
	enum { FFT_MAGIC = 8888 };
	enum Points { fftMinPoints = 64 };
public:
	FFTHeader(int npoints, double frmrate, int srate, double dur=0);
	FFTHeader();
	int nPoints() { return nChans(); }
protected:
	redefined int diskHeaderSize() { return sizeof(DiskHeader); }
	redefined int diskHeaderCommentSize() { return 4; }
	redefined boolean isMagic() { return magic() == FFT_MAGIC; }
	redefined int readInfo(DataFile *);
	redefined int checkHeader();
	redefined int writeInfo(DataFile *);
	redefined const char* magicError();
protected:
	class FFTConfigRequester : public FrameConfigRequester {
		typedef FrameConfigRequester Super;
	public:
		FFTConfigRequester(Header* h)
			: FrameConfigRequester("Read Raw FFT File:", h) {}
	protected:
		redefined void configureRequest(Request *);
	};
	friend void FFTConfigRequester::configureRequest(Request *);
public:
	redefined Requester* configRequester() {
		return new FFTConfigRequester(this);
	}
private:
	class DiskHeader;
	friend class DiskHeader;
	// This is the struct for the FFT header on disk
	class DiskHeader {
	public:
		DiskHeader(int m=FFT_MAGIC, int hs=0, int np=0, float sr=0)
			: fftmagic(m), headersize(hs), npoints(np), srate(sr) {}
		int	fftmagic,
			headersize,
			npoints;		// number of points in fft
		float	srate; 		// sampling rate of orig data
		char	text[4];
	};
};

#endif
