// filecommand.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/

// FileCommand is the base class for classes which either create, read, or
// write to files after querying the user for appropriate information.
 
#ifndef FILECOMMAND_H
#ifdef __GNUG__
#pragma interface
#endif
#define FILECOMMAND_H

#include "localdefs.h"
#include "modifier.h"
#include "MyString.h"

class FileCommand : public Modifier {
	typedef Modifier Super;
public:
	redefined Status configure(Controller *);
	redefined int apply();
protected:
	FileCommand(const char* filename);
	virtual int doApply() = 0;
	String& fileName() { return theFileName; }
	void setFileName(const char* name) { theFileName = name; }
	Controller* controller() { return theController; }
private:
	String theFileName;
	Controller* theController;
};

class Data;

class FileCreator : public FileCommand {
	typedef FileCommand Super;
	friend class FileCreateRequester;
public:
	Controller* getNewController() { return newController; }
protected:
	FileCreator(const char* filename, int length);
	redefined int doApply();
	virtual Data* newData();
	int dataLength() { return max(4, newLength); }
	boolean setData(Data *);
private:
	Controller* newController;
	Data* theData;
	int newLength;
};

class SoundCreator : public FileCreator {
	typedef FileCreator Super;
	friend class SoundCreateRequester;
public:
	SoundCreator();
	redefined const char* message() { return "Creating new sound..."; }
protected:
	redefined Requester* createRequester();
	redefined void initialize();
private:
	TimeVal duration;
	int sampleRate;
	unsigned short channels;
	unsigned short format;
};

class LPCFileCreator : public FileCreator {
	typedef FileCreator Super;
	friend class LPCFileCreateRequester;
public:
	LPCFileCreator();
	redefined const char* message() { return "Creating new LPC file..."; }
protected:
	redefined Requester* createRequester();
	redefined void initialize();
private:
	int nPoles;
	double frameRate;
	int sampleRate;
};

class PvocFileCreator : public FileCreator {
	typedef FileCreator Super;
	friend class PvocFileCreateRequester;
public:
	PvocFileCreator();
	redefined const char* message() { return "Creating new Pvoc file..."; }
protected:
	redefined Requester* createRequester();
	redefined void initialize();
private:
	double frameRate;
	int sampleRate;
	unsigned short nBands;
};

class EnvelopeFileCreator : public FileCreator {
	typedef FileCreator Super;
public:
	EnvelopeFileCreator();
	redefined const char* message() { return "Creating new envelope..."; }
protected:
	redefined Requester* createRequester();
	redefined void initialize();
};

class FileAccessCommand : public FileCommand {
	typedef FileCommand Super;
	friend class FileAccessRequester;
	friend class FileOpenRequester;
public:
	FileAccessCommand(const char* dir);
protected:
	virtual const char* defaultDirectory() { return defaultDir; }
	virtual const char* defaultSuffixes() { return nil; }
private:
	String defaultDir;
};

class FileSaver : public FileAccessCommand {
	typedef FileAccessCommand Super;
	friend class FileSaveRequester;
public:
	FileSaver(const char* dir, Data* data);
	FileSaver(const char* name, const char* dir, Data* data);
	redefined const char* message();
protected:
	redefined Requester* createRequester();
	redefined void initialize();
	redefined int doApply();
protected:
	enum HeaderFormat { Raw = 0x1, WithHeader = 0x2 };
	Data* dataToSave;
	ChoiceValue headerFormat;
private:
	boolean isNewFile;
	boolean forceHeader;
};

class SoundFileSaver : public FileSaver {
	typedef FileSaver Super;
	friend class SoundFileSaveRequester;
public:
	SoundFileSaver(const char* dir, Data* data);
	SoundFileSaver(const char* name, const char* dir, Data* data);
protected:
	redefined Requester* createRequester();
	redefined void initialize();
private:
	ChoiceValue soundHeaderFormat;
};

class LPCFileSaver : public FileSaver {
	typedef FileSaver Super;
public:
	LPCFileSaver(const char* dir, Data* data);
	LPCFileSaver(const char* name, const char* dir, Data* data);
protected:
	redefined void initialize();
};

class EnvelopeFileSaver : public FileSaver {
	typedef FileSaver Super;
public:
	EnvelopeFileSaver(const char* dir, Data* data);
	EnvelopeFileSaver(const char* name, const char* dir, Data* data);
protected:
	redefined void initialize();
};

class DataDumper : public FileSaver {
	typedef FileSaver Super;
	friend class DataDumpRequester;
public:
	DataDumper(const char* dir, Data* data) : FileSaver(dir, data) {}
	redefined const char* message();
protected:
	redefined Requester* createRequester();
	redefined int doApply();
};

class FileOpener : public FileAccessCommand {
	friend class FileOpenRequester;
public:
	FileOpener(const char* dir, const char* title);
	redefined const char* message();
	Controller* getNewController() { return newController; }
protected:
	redefined Requester* createRequester();
	redefined int doApply();
private:
	String theTitle;
	TimeVal skipTime;
	TimeVal duration;
	Controller* newController;
};

#endif
