// converter.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// The base class for all objects implementing an interface to a D/A A/D 
// converter.  At least one subclass is defined for each architecture, more if
// there are multiple devices available.  The base class makes no assumptions
// about how the sound is delivered to the device.  There is a derived class
// ConverterDevice which implements additional methods for writing and reading
// buffers (see conv_device.h).  Note:  Only one Converter class instance can
// exist at one time -- if switching converters, the old one must be deleted
// first.

#ifndef CONVERTER_H
#ifdef __GNUG__
#pragma interface
#endif
#define CONVERTER_H

#include <InterViews/resource.h>
#include "localdefs.h"
#include "requester.h"

class Action;		// encapsulated function to call when recording done
class ProgressAction;	// same for current progress
class Sound;

class Converter : virtual public Resource {
public:		// class methods
	enum GoingTo { Record, Play };
	static Converter* getInstance();		// assures single object instance
	static void destroyInstance();
	static boolean useNative() { return usingNative; }
	static void useNative(boolean b);
protected:	// class methods
	static Converter* create();
public:		// object methods
	virtual boolean ready() const = 0;
	virtual void configureHardware(class Controller*) = 0;
	virtual DataType bestPlayableType() = 0;
	virtual boolean hasPlayableFormat(Sound *) = 0;
	virtual int configure(Sound *sound, GoingTo) = 0;
	virtual int start(ProgressAction *, Action *) = 0;
	virtual int pause() = 0;
	virtual int resume() = 0;
	virtual int stop() = 0;
protected:
	virtual Requester* configRequester() { return nil; }
private:	// class members
	static boolean usingNative;
	static Converter* converterInstance;
};

class NullConverter : public Converter {
public:		// object methods
	redefined boolean ready() const { return true; }
	redefined void configureHardware(class Controller*) {}
	redefined DataType bestPlayableType() { return NoData; }
	redefined boolean hasPlayableFormat(Sound *) { return true; }
	redefined int configure(Sound *sound, GoingTo) { return false; }
	redefined int start(ProgressAction *, Action *) { return false; }
	redefined int pause() { return false; }
	redefined int resume() { return false; }
	redefined int stop() { return false; }
};

class RealConverter : public Converter {
public:		// object methods
	redefined boolean ready() const;			// check for successful constr.
	redefined void configureHardware(class Controller*);
	redefined boolean hasPlayableFormat(Sound *);
	redefined int configure(Sound *sound, GoingTo);
	redefined int start(ProgressAction *, Action* whenDone);
	redefined int pause();
	redefined int resume();
	redefined int stop();
protected:		// class methods
	enum Status { Error = 0, Stopped, Paused, Recording, Playing };
	virtual int startPlaying(ProgressAction *);
	virtual int startRecording(ProgressAction *, Action* whenDone);
	static void setStatus(Status s) { theStatus = s; }
	static void setRecordDoneAction(Action *);
	static void doRecordDoneAction();
	static boolean playing() { return theStatus == Playing; }
	static boolean recording() { return theStatus == Recording; }
	static boolean paused() { return theStatus == Paused; }
	static boolean running();
	static boolean good() { return theStatus != Error; }
protected:		// object methods
	RealConverter();
	virtual ~RealConverter();
	virtual boolean isPlayableFormat(DataType type) = 0;
	virtual int currentPlayLevel() const;	// scaled between 0 - 100
	virtual int currentRecordLevel() const;
	virtual boolean setPlayLevel(int volume);
	virtual boolean setRecordLevel(int volume);
	virtual boolean setSpeakerOutput(boolean);
	virtual int doConfigure() = 0;		// implm. by all subclasses
	virtual int doConversion(ProgressAction *) = 0;
	virtual int doRecording(ProgressAction *) = 0;
	virtual int waitForStop(ProgressAction *) { return true; }
	// status methods
	virtual boolean willPlay() const { return goingTo == Play; }
	virtual boolean willRecord() const { return goingTo == Record; }
	virtual boolean soundIsLoaded() { return mySound != nil; }
	int checkLength(int);
	// these should be redefined for each converter
	virtual int checkSampleRate(int) { return true; }
	virtual int checkChannels(int) { return true; }
	virtual int checkDataType(DataType) { return true; }
	// these return information about current sound chunk
	virtual int sampleRate();
	virtual int channels();
	virtual DataType dataType();
	virtual const void* pointerToData();
	virtual off_t dataSize();
	virtual double duration();
	virtual double peakAmplitude();
	virtual void fail() { setStatus(Error); }
	virtual void failIf(int condition) { if(condition) fail(); }
	virtual int error(const char* msg1, const char* msg2=nil);
	virtual void catchSignals(boolean flag) {}
	virtual void ignoreSignals(boolean flag) {}
	void initialize();
	void reset();
	boolean notImplemented() const;	// pops up message and returns false
	// internal base class which queries the user for config. parameters
	class ConfigRequester : public TitledRequester {
	public:
		ConfigRequester(RealConverter* converter);
	protected:
		redefined void configureRequest(Request *);
		RealConverter* myConverter;
	};
	friend class ConfigRequester;
private:
	void setSound(Sound *);
private:	// class members
	static Status theStatus;
	static Action* recordDoneAction;
private:
	friend class Controller;
	Sound* mySound;
	GoingTo goingTo;
};

inline int
RealConverter::pause() {
	setStatus(Paused);
	return true;
}

inline int
RealConverter::resume() {
	setStatus(willPlay() ? Playing : Recording);
	return true;
}

// these are defined as not being implemented, so subclasses do not need to
// redefine these unless they will be functional.

inline int
RealConverter::currentPlayLevel() const { notImplemented(); return -1; }

inline int
RealConverter::currentRecordLevel() const { notImplemented(); return -1; }

inline boolean
RealConverter::setPlayLevel(int volume) {		// between 0 and 100
	return notImplemented();
}

inline boolean
RealConverter::setRecordLevel(int volume) {		// between 0 and 100
	return notImplemented();
}

inline boolean
RealConverter::setSpeakerOutput(boolean) {
	return notImplemented();
}

#endif
