//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: track.cpp,v 1.1 2002/01/30 14:10:07 muse Exp $
//
//  (C) Copyright 2000 Werner Schweer (ws@seh.de)
//=========================================================

#include "track.h"
#include "event.h"
#include "driver/mididev.h"
#include "driver/audiodev.h"
#include "midiport.h"
#include "audioport.h"
#include "song.h"
#include "sf/sndfile.h"
#include "midithread.h"
#include "globals.h"

int Track::snGen=100;

//---------------------------------------------------------
//   Track::init
//---------------------------------------------------------

void Track::init()
      {
      _recordFlag  = false;
      _inPortMask  = 0xffff;
      _outPort     = 0;
      _outChannel  = 0;
      _inChannelMask   = 0xffff;      // "ALL"
      _locked      = false;
      _selected    = false;
      _sn          = newSn();
      }

Track::Track(Track::TrackType t)
      {
      init();
      _type = t;
      }

Track::Track(const QString& name, TrackType t)
   : _name(name)
      {
      init();
      _type = t;
      }

Track::Track(const Track& t)
      {
      _recordFlag  = t.recordFlag();
      _comment     = t.comment();
      _outPort     = t.outPort();
      _outChannel  = t.outChannel();
      _inPortMask      = t.inPortMask();
      _inChannelMask   = t.inChannelMask();
      _name        = t.tname();
      _type        = t.type();
      _locked      = t.locked();
      _selected    = t.selected();
      _sn          = newSn();

      const PartList* pl = t.cparts();
      for (ciPart ip = pl->begin(); ip != pl->end(); ++ip) {
            Part* newPart = ip->second->clone();
            newPart->setTrack(this);
            _parts.add(newPart);
            }
      }

Track::~Track()
      {
      }

void Track::dump() const
      {
      printf("Track <%s>: typ %d, parts %d sel %d rec %d\n",
         _name.latin1(), _type, _parts.size(), _selected, _recordFlag);
      }

void Track::setSn(int n)
      {
//      printf("set sn %d\n", n);
      _sn = n;
      }

//---------------------------------------------------------
//   MidiTrack
//---------------------------------------------------------

MidiTrack::MidiTrack()
   : Track(MIDI), key(0), scale(0)
      {
      init();
      _events = new EventList;
      }
MidiTrack::MidiTrack(const QString& name)
   : Track(name, Track::MIDI), key(0), scale(0)
      {
      init();
      _events = new EventList;
      }

MidiTrack::MidiTrack(const MidiTrack& mt)
   : Track(mt), SoundSource(mt)
      {
      _events       = new EventList;
      key           = mt.key;
      scale         = mt.scale;
      keyL          = mt.keyL;
      scaleL        = mt.scaleL;
      noteQuant     = mt.noteQuant;
      restQuant     = mt.restQuant;
      mode          = mt.mode;
      splitpoint    = mt.splitpoint;
      transposition = mt.transposition;
      velocity      = mt.velocity;
      delay         = mt.delay;
      len           = mt.len;
      compression   = mt.compression;
      }

MidiTrack::~MidiTrack()
      {
      delete _events;
      }

void MidiTrack::init()
      {
      noteQuant     = 96*4;     // 4'tel Noten Darstellungs-Quantisierung
      restQuant     = 96*4;     // Pausen Darstellungs-Quantisierung
      transposition = 0;
      velocity      = 0;
      delay         = 0;
      len           = 100;          // percent
      compression   = 100;          // percent
      mode          = Single;
      splitpoint    = 60;
      _midiThruFlag = true;
      keyL.set(4);
      scaleL.set(0);
      }

MidiTrack* MidiTrack::clone() const
      {
      return new MidiTrack(*this);
      }

void MidiTrack::copy(const Track* /*t*/)
      {
      printf("COPY MIDITRACK not implemented\n");  // TODO
      }

//---------------------------------------------------------
//   WaveTrack
//---------------------------------------------------------

void WaveTrack::init()
      {
      _recFile = 0;
      setVolume(1.0);
      for (int i = 0; i < PREFETCH_BUFFER; ++i)
            buffer[i] = 0;
      }

WaveTrack::WaveTrack(const QString& str)
   : Track(str, Track::WAVE)
      {
      init();
      }

WaveTrack::WaveTrack()
   : Track(Track::WAVE)
      {
      init();
      }

WaveTrack::WaveTrack(const WaveTrack& wt)
   : Track(wt), AudioNode(wt)
      {
      init();
      }

WaveTrack* WaveTrack::clone() const
      {
      return new WaveTrack(*this);
      }

void WaveTrack::copy(const Track* /*t*/)
      {
      printf("COPY WAVE TRACK not impl.\n");   // TODO
      buffer[0] = 0;
      buffer[1] = 0;
      }

void WaveTrack::setRecFile(SndFile* sf)
      {
      if (_recFile)
            delete _recFile;
      _recFile = sf;
      }

WaveTrack::~WaveTrack()
      {
      if (_recFile)
            delete _recFile;
      if (buffer[0])
            delete buffer[0];
      if (buffer[1])
            delete buffer[1];
      }

void WaveTrack::setSn(int n)
      {
      Track::setSn(n);
      }

//---------------------------------------------------------
//   add
//---------------------------------------------------------

Track* TrackList::add(Track* track)
      {
      if (track->tname().isEmpty()) {
            QString s("Track ");
            QString k;
            k.setNum(size()+1);
            track->setTName(s+k);
            }
      push_back(track);
      return track;
      }

//---------------------------------------------------------
//   index
//---------------------------------------------------------

unsigned TrackList::index(Track* track)
      {
      unsigned index = 0;
      for (iTrack i = begin(); i != end(); ++i, ++index)
            if (*i == track)
                  return index;
      printf("TrackList::index(): not found!\n");
      return 0;
      }

//---------------------------------------------------------
//   sfind
//    search track with specified serial number
//---------------------------------------------------------

Track* TrackList::sfind(int sno)
      {
      for (iTrack i = begin(); i != end(); ++i) {
            if ((*i)->sn() == sno) {
                  return *i;
                  }
            }
      int index = 0;
      for (iTrack i = begin(); i != end(); ++i) {
            printf("search %d  in %d, index %d %p\n", sno, (*i)->sn(), index, *i);
            ++index;
            }
printf("sfind: track sno %d not found\n", sno);
      return 0;
      }

//---------------------------------------------------------
//   findIdx
//    search track with specified serial number
//---------------------------------------------------------

Track* TrackList::findIdx(int idx)
      {
      int index = 0;
      for (iTrack i = begin(); i != end(); ++i, ++index) {
            if (index == idx)
                  return *i;
            }
printf("findIdx: track %d not found\n", idx);
      return 0;
      }

//---------------------------------------------------------
//   find
//    search track with specified serial number
//---------------------------------------------------------

iTrack TrackList::find(int sno)
      {
      int index = 0;
      iTrack i;
      for (i = begin(); i != end(); ++i, ++index) {
            if (index == sno)
                  return i;
            }
printf("TrackList find(): track serial %d not found\n", sno);
      return i;
      }

//---------------------------------------------------------
//   deselect
//---------------------------------------------------------

void TrackList::deselect()
      {
      for (iTrack i = begin(); i != end(); ++i)
            (*i)->setSelected(false);
      }

//---------------------------------------------------------
//   addPart
//---------------------------------------------------------

iPart Track::addPart(Part* p)
      {
      p->setTrack(this);
      return _parts.add(p);
      }

//---------------------------------------------------------
//   findPart
//---------------------------------------------------------

Part* Track::findPart(int tick)
      {
      for (iPart i = _parts.begin(); i != _parts.end(); ++i) {
            Part* part = i->second;
            if (tick >= part->posTick() && tick < (part->posTick()+part->lenTick()))
                  return part;
            }
      return 0;
      }

//---------------------------------------------------------
//   device
//---------------------------------------------------------

Device* MidiTrack::outDevice() const
      {
      return midiPorts[outPort()].device();
      }

Device* WaveTrack::outDevice() const
      {
      return audioPort.device();
      }

Device* MidiTrack::inDevice() const
      {
      return 0;
      }

Device* WaveTrack::inDevice() const
      {
      return audioPort.device();
      }

//---------------------------------------------------------
//   setOutChannel
//    TODO: lock access
//---------------------------------------------------------

void Track::setOutChannel(int i)
      {
      _outChannel = i;
      for (iPart ip = _parts.begin(); ip != _parts.end(); ++ip) {
            EventList* el = ip->second->events();
            for (iEvent ie = el->begin(); ie != el->end(); ++ie) {
                  ie->second->setChannel(i);
                  }
            }
      }
//---------------------------------------------------------
//   setInChannel
//---------------------------------------------------------

void Track::setInChannelMask(int c)
      {
      _inChannelMask = c;
      }

//---------------------------------------------------------
//   setOutPort
//    TODO: lock access
//---------------------------------------------------------

void Track::setOutPort(int i)
      {
      _outPort = i;
      for (iPart ip = _parts.begin(); ip != _parts.end(); ++ip) {
            EventList* el = ip->second->events();
            for (iEvent ie = el->begin(); ie != el->end(); ++ie) {
                  ie->second->setPort(i);
                  }
            }
      }

//---------------------------------------------------------
//   setInPort
//---------------------------------------------------------

void Track::setInPortMask(int i)
      {
      _inPortMask = i;
      }

//---------------------------------------------------------
//   getData
//---------------------------------------------------------

void WaveTrack::getData(float* bp, int idx)
      {
      int n = segmentSize * channels();
      if (mute() || !midiThread->isPlaying() || (buffer[idx] == 0)) {
            memset(bp, 0, n * sizeof(float));
            return;
            }
//      printf("get %d\n", idx);
      memcpy(bp, buffer[idx], n * sizeof(float));
      }

//---------------------------------------------------------
//   getData
//---------------------------------------------------------

void WaveTrack::fetchData(int bufferNo, int curPos, bool realloc)
      {
//      printf("read %d\n", bufferNo);
      int n = segmentSize * channels();
      float* bp = buffer[bufferNo];
      if (realloc) {
            if (buffer[0])
                  delete buffer[0];
            float* p = new float[n * PREFETCH_BUFFER];
            memset(p, 0, n * PREFETCH_BUFFER * sizeof(float));
            for (int i = 0; i < PREFETCH_BUFFER; ++i) {
                  buffer[i] = p;
                  p += n;
                  }
            bp = buffer[bufferNo];
            }
      else
            memset(bp, 0, n * sizeof(float));

      PartList* pl = parts();

      n = segmentSize;
      for (iPart ip = pl->begin(); ip != pl->end(); ++ip) {
            WavePart* part = (WavePart*)(ip->second);
            EventList* events = part->events();
            int p_spos = part->posSample();
            int p_epos = p_spos + part->lenSample();
            if (curPos+n < p_spos) {
                  break;
                  }
            if (curPos >= p_epos) {
                  continue;
                  }
            for (iEvent ie = events->begin(); ie != events->end(); ++ie) {
                  WaveEvent* event = (WaveEvent*)ie->second;
                  int p_spos = event->posSample();
                  int p_epos = p_spos + event->lenSample();
                  int nn     = p_epos - p_spos;
                  if (curPos + n < p_spos)
                        break;
                  if (curPos >= p_epos)
                        continue;
                  float* dpos = bp;
                  int offset = p_spos - curPos;
                  if (offset > 0) {
                        dpos += offset;
                        nn = n - offset;
                        offset = 0;
                        }
                  else {
                        offset = -offset;
                        nn -= offset;
                        if (nn > n)
                              nn = n;
                        }
                  Clip* clip = event->clip();
                  float* bp[channels()];
                  bp[0] = dpos;
                  for (int i = 1; i < channels(); ++i)
                       bp[i] =  bp[i-1] + segmentSize;
      int v[channels()];      // meter values
                  clip->read(offset, bp, channels(), nn, v, _volume);
                  }
            }
      }

//---------------------------------------------------------
//   newPart
//---------------------------------------------------------

Part* MidiTrack::newPart(Part*p=0)
      {
      MidiPart* part = new MidiPart(this);
      if (p) {
            part->setName(p->name());
            part->setColorIndex(p->colorIndex());
            *(PosLen*)part = *(PosLen*)p;
            }
      return part;
      }

//---------------------------------------------------------
//   newPart
//---------------------------------------------------------

Part* WaveTrack::newPart(Part*p=0)
      {
      WavePart* part = new WavePart(this);
      if (p) {
            part->setName(p->name());
            part->setColorIndex(p->colorIndex());
            *(PosLen*)part = *(PosLen*)p;
            }
      return part;
      }

