//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: part.cpp,v 1.1 2002/01/30 14:10:07 muse Exp $
//
//  (C) Copyright 1999/2000 Werner Schweer (ws@seh.de)
//=========================================================

#include <stdio.h>

#include "song.h"
#include "part.h"
#include "track.h"
#include "globals.h"
#include "event.h"
#include "midithread.h"

int Part::snGen;

PartColor partColors[NUM_PARTCOLORS] = {
      { "White",        0xff, 0xff, 0xff },
      { "Refrain",      0xff, 0x00, 0x00 },
      { "Bridge",       0x00, 0xff, 0x00 },
      { "Intro",        0x00, 0x00, 0xff },
      { "Coda",         0xff, 0xff, 0x00 },
      { "Chorus",       0x00, 0xff, 0xff },
      { "Solo",         0xff, 0x00, 0xff },
      { "Brass",        0x9f, 0xc7, 0xef },
      { "Percussion",   0x00, 0xff, 0x7f },
      { "Drums",        0x7f, 0x00, 0x00 },
      { "Guitar",       0x00, 0x7f, 0x00 },
      { "Bass",         0x00, 0x00, 0x7f },
      { "Flute",        0x7f, 0x7f, 0x3f },
      { "Strings",      0x00, 0x7f, 0x7f },
      { "Keyboard",     0x7f, 0x00, 0x7f },
      { "Piano",        0x00, 0x7f, 0xff },
      { "Saxophon",     0x00, 0x3f, 0x3f },
      };

iEvent Part::addEvent(Event* p)
      {
      return _events->add(p);
      }

//---------------------------------------------------------
//   index
//---------------------------------------------------------

int PartList::index(Part* part)
      {
      int index = 0;
      for (iPart i = begin(); i != end(); ++i, ++index)
            if (i->second == part) {
                  return index;
                  }
      printf("PartList::index(): not found!\n");
      return 0;
      }

//---------------------------------------------------------
//   find
//---------------------------------------------------------

Part* PartList::find(int idx)
      {
      int index = 0;
      for (iPart i = begin(); i != end(); ++i, ++index)
            if (index == idx)
                  return i->second;
      return 0;
      }

//---------------------------------------------------------
//   Part
//---------------------------------------------------------

Part::Part(Track* t)
      {
      setSn(newSn());
      _track      = t;
      _selected   = false;
      _mute       = false;
      _colorIndex = 0;
      _events     = new EventList;
      }

WavePart::WavePart(WaveTrack* t)
   : Part(t)
      {
      setType(SAMPLES);
      }

//---------------------------------------------------------
//   Part
//---------------------------------------------------------

Part::~Part()
      {
      delete _events;
      }

//---------------------------------------------------------
//   findPart
//---------------------------------------------------------

iPart PartList::findPart(int tick)
      {
      iPart i;
      for (i = begin(); i != end(); ++i)
            if (i->second->posTick() == tick)
                  break;
      return i;
      }

//---------------------------------------------------------
//   add
//---------------------------------------------------------

iPart PartList::add(Part* part)
      {
      return insert(std::pair<const int, Part*> (part->posTick(), part));
      }

//---------------------------------------------------------
//   remove
//---------------------------------------------------------

void PartList::remove(Part* part)
      {
      iPart i;
      for (i = begin(); i != end(); ++i) {
            if (i->second == part) {
                  erase(i);
                  break;
                  }
            }
      assert(i != end());
      }

//---------------------------------------------------------
//   removePart
//---------------------------------------------------------

void Song::removePart(Part* part)
      {
      part->track()->parts()->remove(part);
      }

//---------------------------------------------------------
//   addPart
//---------------------------------------------------------

void Song::addPart(Part* part)
      {
      // adjust song len:
      int epos = part->posTick() + part->lenTick();

      if (epos > len())
            setLen(epos);

      Track* track = part->track();
      track->addPart(part);
      }

//---------------------------------------------------------
//   cmdResizePart
//---------------------------------------------------------

void Song::cmdResizePart(Track* track, Part* oPart, int len)
      {
      switch(track->type()) {
            case Track::WAVE:
                  break;
            case Track::MIDI:
            case Track::DRUM:
                  {
                  startUndo();
                  MidiPart* nPart = new MidiPart(*(MidiPart*)oPart);
                  nPart->setLenTick(len);

                  //
                  // cut Events in nPart
                  if (oPart->lenTick() > len) {
                        EventList* el = nPart->events();
                        iEvent ie = el->lower_bound(oPart->posTick() + len);
                        for (; ie != el->end();) {
                              iEvent i = ie;
                              ++ie;
                              midiThread->msgDeleteEvent((MidiEvent*)i->second, nPart, false);
                              }
                        }
                  midiThread->msgChangePart(oPart, nPart, false);
                  endUndo(SC_PART_MODIFIED);
                  break;
                  }
            }
      }

//---------------------------------------------------------
//   splitPart
//    split part "part" at "tick" position
//    create two new parts p1 and p2
//---------------------------------------------------------

void Track::splitPart(Part* part, int ptick, Part*& p1, Part*& p2)
      {
      int l1=0, l2=0;
      int stick = tempomap.tick2samples(ptick);

      switch (type()) {
            case WAVE:
                  l1 = stick - part->posSample();
                  l2 = part->lenSample() - l1;
                  break;
            case MIDI:
            case DRUM:
                  l1 = ptick - part->posTick();
                  l2 = part->lenTick() - l1;
                  break;
            }

      if (l1 <= 0 || l2 <= 0)
            return;

      //  p1 is new modified left part
      //  p2 is new right part of split

      p1 = newPart(part);
      p2 = newPart(part);

      switch (type()) {
            case WAVE:
                  *(Pos*)p1 = *(Pos*)part;
                  p1->setLenSample(l1);
                  p2->setPosSample(stick);
                  p2->setLenSample(l2);
                  break;
            case MIDI:
            case DRUM:
                  p1->setPosTick(part->posTick());
                  p1->setLenTick(l1);
                  p2->setPosTick(ptick);
                  p2->setLenTick(l2);
                  break;
            }

      p2->setSn(p2->newSn());

      EventList* se  = part->events();
      EventList* de1 = p1->events();
      EventList* de2 = p2->events();

      for (iEvent ie = se->begin(); ie != se->end(); ++ie) {
            Event* event = ie->second;
            switch (type()) {
                  case WAVE:
                        if (event->endSample() > p1->posSample() && event->posSample() < p1->endSample()) {
                              Event* si = event->mid(*p1, p1->end());
                              if (si) {
                                    de1->add(si);
                                    }
                              }
                        if (event->endSample() > p2->posSample() && event->posSample() < p2->endSample()) {
                              Event* si = event->mid(*p2, p2->end());
                              if (si) {
                                    de2->add(si);
                                    }
                              }
                        break;
                  case MIDI:
                  case DRUM:
                        if (event->endTick() > p1->posTick() && event->posTick() < p1->endTick()) {
                              Event* si = event->mid(*p1, p1->end());
                              if (si)
                                    de1->add(si);
                              }
                        if (event->endTick() > p2->posTick() && event->posTick() < p2->endTick()) {
                              Event* si = event->mid(*p2, p2->end());
                              if (si)
                                    de2->add(si);
                              }
                        break;
                  }
            }
      }

//---------------------------------------------------------
//   cmdSplitPart
//---------------------------------------------------------

void Song::cmdSplitPart(Track* track, Part* part, int tick)
      {
      int l1 = tick - part->posTick();
      int l2 = part->lenTick() - l1;
      if (l1 <= 0 || l2 <= 0)
            return;
      Part* p1;
      Part* p2;
      track->splitPart(part, tick, p1, p2);

      startUndo();
      midiThread->msgChangePart(part, p1, false);
      midiThread->msgAddPart(p2, false);
      endUndo(SC_TRACK_MODIFIED | SC_PART_MODIFIED | SC_PART_INSERTED);
      }

//---------------------------------------------------------
//   changePart
//---------------------------------------------------------

void Song::changePart(Part* oPart, Part* nPart)
      {
      nPart->setSn(oPart->sn());

      Track* oTrack = oPart->track();
      Track* nTrack = nPart->track();

      switch(oTrack->type()) {
            case Track::WAVE:
                  oTrack->parts()->remove(oPart);
                  nTrack->parts()->add(nPart);
                  break;
            case Track::MIDI:
            case Track::DRUM:
                  {
                  oTrack->parts()->remove(oPart);
                  nTrack->parts()->add(nPart);
                  }
                  break;
            }
      }

//---------------------------------------------------------
//   cmdGluePart
//---------------------------------------------------------

void Song::cmdGluePart(Track* track, Part* oPart)
      {
      PartList* pl   = track->parts();
      Part* nextPart = 0;

      for (iPart ip = pl->begin(); ip != pl->end(); ++ip) {
            if (ip->second == oPart) {
                  ++ip;
                  if (ip == pl->end())
                        return;
                  nextPart = ip->second;
                  break;
                  }
            }

      Part* nPart = track->newPart(oPart);
      nPart->setLenTick(nextPart->posTick()+nextPart->lenTick()-oPart->posTick());

      // populate nPart with Events from oPart and nextPart

      EventList* sl1 = oPart->events();
      EventList* sl2 = nextPart->events();
      EventList* dl  = nPart->events();
      for (iEvent ie = sl1->begin(); ie != sl1->end(); ++ie)
            dl->add(ie->second);
      for (iEvent ie = sl2->begin(); ie != sl2->end(); ++ie)
            dl->add(ie->second);
      startUndo();
      midiThread->msgRemovePart(nextPart, false);
      midiThread->msgChangePart(oPart, nPart, false);
      endUndo(SC_PART_MODIFIED | SC_PART_REMOVED);
      }

//---------------------------------------------------------
//   dump
//---------------------------------------------------------

void Part::dump() const
      {
      printf("Part: <%s> ", _name.latin1());
      PosLen::dump();
      }

void WavePart::dump() const
      {
      Part::dump();
      printf("  WavePart\n");
      }

void MidiPart::dump() const
      {
      Part::dump();
      printf("  MidiPart\n");
      }

Event* WavePart::newEvent() const
      {
      return new WaveEvent();
      }

Event* MidiPart::newEvent() const
      {
      return new MidiEvent();
      }

//---------------------------------------------------------
//   clone
//---------------------------------------------------------

MidiPart* MidiPart::clone() const
      {
      return new MidiPart(*this);
      }

WavePart* WavePart::clone() const
      {
      return new WavePart(*this);
      }

