//=========================================================
//  MusE
//  Linux Music Editor
//    $Id: drumedit.cpp,v 1.1 2002/01/30 14:10:08 muse Exp $
//  (C) Copyright 1999 Werner Schweer (ws@seh.de)
//=========================================================

#include "drumedit.h"

#include "mtscale.h"
#include "scrollscale.h"
#include "intlabel.h"
#include "xml.h"
#include "dlist.h"
#include "dcanvas.h"
#include "ttoolbar.h"
#include "tb1.h"
#include "splitter.h"
#include "utils.h"
#include "../ctrl/ctrledit.h"
#include "vscale.h"
#include "swidget.h"
#include "globals.h"
#include "icons.h"
#include "event.h"
#include "filedialog.h"
#include "drummap.h"
#include "midithread.h"

#include <qtoolbutton.h>
#include <qaccel.h>
#include <qlayout.h>
#include <qhbox.h>
#include <qsizegrip.h>
#include <qscrollbar.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qbuttongroup.h>
#include <qlistbox.h>
#include <qpopupmenu.h>
#include <qmenubar.h>
#include <qtooltip.h>
#include <qapplication.h>
#include <qclipboard.h>

static const char* map_file_pattern[] = {
      "presets (*.map *.map.gz *.map.bz2)",
      "All Files (*)",
      0
      };
int DrumEdit::_quantInit = 96;
int DrumEdit::_rasterInit = 96;

static const int xscale = -10;
static const int yscale = 1;

//---------------------------------------------------------
//   DWhatsThis::text
//---------------------------------------------------------

QString DWhatsThis::text(const QPoint& pos)
      {
      int section = header->sectionAt(pos.x());
      if (section == -1)
            return 0;
      switch(section) {
            case 0: return  header->tr("mute instrument"); break;
            case 1: return  header->tr("sound name"); break;
            case 2: return  header->tr("quantisation"
                  "\ncurrently not used"); break;
            case 3: return  header->tr("this input note triggers the sound"); break;
            case 4: return  header->tr("note length"); break;
            case 5: return  header->tr("this note is send for the sound"); break;
            case 6: return  header->tr("output channel"
                  "\ncurrently not used"); break;
            case 7: return  header->tr("output port"
                  "\ncurrently not used"); break;
            case 8: return  header->tr("velocity level 1"); break;
            case 9: return  header->tr("velocity level 2"); break;
            case 10: return header->tr("velocity level 3"); break;
            case 11: return header->tr("velocity level 4"); break;
            default: break;
            }
      return 0;
      }

//---------------------------------------------------------
//   DHeaderTip::maybeTip
//---------------------------------------------------------

void DHeaderTip::maybeTip(const QPoint &pos)
      {
      Header* w  = (Header*)parentWidget();
      int section = w->sectionAt(pos.x());
      if (section == -1)
            return;
      QRect r(w->sectionPos(section), 0, w->sectionSize(section),
         w->height());
      QString p;
      switch(section) {
            case 0:  p = w->tr("mute instrument"); break;
            case 1:  p = w->tr("sound name"); break;
            case 2:  p = w->tr("quantisation"); break;
            case 3:  p = w->tr("this input note triggers the sound"); break;
            case 4:  p = w->tr("note length"); break;
            case 5:  p = w->tr("this note is send for the sound"); break;
            case 6:  p = w->tr("output channel"); break;
            case 7:  p = w->tr("output port"); break;
            case 8:  p = w->tr("velocity level 1"); break;
            case 9:  p = w->tr("velocity level 2"); break;
            case 10: p = w->tr("velocity level 3"); break;
            case 11: p = w->tr("velocity level 4"); break;
            default: return;
            }
      tip(r, p);
      }

//---------------------------------------------------------
//   closeEvent
//---------------------------------------------------------

void DrumEdit::closeEvent(QCloseEvent* e)
      {
      emit deleted((int)this);
      e->accept();
      }

//---------------------------------------------------------
//   DrumEdit
//---------------------------------------------------------

DrumEdit::DrumEdit(PartList* pl, QWidget* parent = 0, const char* name = 0)
   : MidiEditor(_quantInit, _rasterInit, pl, parent, name)
      {
      selEvent = 0;
      selPart  = 0;
      //---------Pulldown Menu----------------------------
      QPopupMenu* menuFile = new QPopupMenu(this);
      menuBar()->insertItem("&File", menuFile);

      menuFile->insertItem(*openIcon, tr("Load Map"), DrumCanvas::CMD_LOAD);
      menuFile->setAccel(CTRL+Key_O, DrumCanvas::CMD_LOAD);
      menuFile->insertItem(*saveIcon, tr("Save Map"), DrumCanvas::CMD_SAVE);
      menuFile->setAccel(CTRL+Key_S, DrumCanvas::CMD_SAVE);

      menuEdit = new QPopupMenu(this);
      menuBar()->insertItem(tr("&Edit"), menuEdit);
      menuEdit->insertItem(tr("Cut"),   DrumCanvas::CMD_CUT);
      menuEdit->setAccel(CTRL+Key_X,    DrumCanvas::CMD_CUT);
      menuEdit->insertItem(tr("Copy"),  DrumCanvas::CMD_COPY);
      menuEdit->setAccel(CTRL+Key_C,    DrumCanvas::CMD_COPY);
      menuEdit->insertItem(tr("Paste"), DrumCanvas::CMD_PASTE);
      menuEdit->setAccel(CTRL+Key_V,    DrumCanvas::CMD_PASTE);
      menuEdit->insertSeparator();
      menuEdit->insertItem(tr("Delete Events"), DrumCanvas::CMD_DEL);
      menuEdit->insertSeparator();

      QPopupMenu* menuSelect = new QPopupMenu(this);
      menuSelect->insertItem(tr("Select All"),   DrumCanvas::CMD_SELECT_ALL);
      menuSelect->insertItem(tr("Select None"),  DrumCanvas::CMD_SELECT_NONE);
      menuSelect->insertItem(tr("Invert"),       DrumCanvas::CMD_SELECT_INVERT);
      menuSelect->insertSeparator();
      menuSelect->insertItem(tr("Inside Loop"),  DrumCanvas::CMD_SELECT_ILOOP);
      menuSelect->insertItem(tr("Outside Loop"), DrumCanvas::CMD_SELECT_OLOOP);
      menuEdit->insertItem(tr("&Select"), menuSelect);

      connect(menuFile, SIGNAL(activated(int)), SLOT(cmd(int)));
      connect(menuEdit, SIGNAL(activated(int)), SLOT(cmd(int)));
      connect(menuSelect, SIGNAL(activated(int)),    SLOT(cmd(int)));

      //---------------------------------------------------
      //    Toolbars
      //---------------------------------------------------

      tools = new QToolBar(this, "drum-tools");
      new QToolButton(*openIcon, "Load Drummap",
					    QString::null, this, SLOT(load()),
					    tools, "load drummap from file");

      new QToolButton(*saveIcon, "Store Drummap",
					    QString::null,
					    this, SLOT(save()),
					    tools, "store drummap to file");

      QWhatsThis::whatsThisButton(tools);

      tools->addSeparator();
      undoRedo->addTo(tools);
      tools->addSeparator();

      srec  = new QToolButton(tools, "srec");
      QToolTip::add(srec, tr("Step Record"));
      srec->setPixmap(*steprecIcon);
      srec->setToggleButton(true);

      midiin  = new QToolButton(tools, "midiin");
      QToolTip::add(midiin, tr("Midi Input"));
      midiin->setPixmap(*midiinIcon);
      midiin->setToggleButton(true);

      EditToolBar* etb = new EditToolBar(this,
         PointerTool | PencilTool | RubberTool);

      new TransportToolbar(this);

      // dontt show pitch value in toolbar
      toolbar = new Toolbar1(this, _rasterInit, _quantInit, false);
      info    = new NoteInfo(this);

      //---------------------------------------------------
      //    split
      //---------------------------------------------------

      split1            = new Splitter(Vertical, mainw, "split1");
      QPushButton* ctrl = new QPushButton("ctrl", mainw, "Ctrl");
      hscroll           = new ScrollScale(-25, -2, xscale, 20000, Horizontal, mainw);
      ctrl->setFixedSize(50, hscroll->sizeHint().height());
      QToolTip::add(ctrl, tr("Add Controller View"));

      QSizeGrip* corner = new QSizeGrip(mainw);

      mainGrid->setRowStretch(0, 100);
      mainGrid->setColStretch(1, 100);
      mainGrid->addMultiCellWidget(split1, 0, 0, 0, 2);
      mainGrid->addWidget(ctrl,    1, 0);
      mainGrid->addWidget(hscroll, 1, 1);
      mainGrid->addWidget(corner,  1, 2, AlignBottom|AlignRight);
      mainGrid->addRowSpacing(1, hscroll->sizeHint().height());

      split2              = new Splitter(Horizontal, split1, "split2");
      QWidget* split1w1   = new QWidget(split2, "split1w1");
      QWidget* split1w2   = new QWidget(split2, "split1w2");
      QGridLayout* gridS1 = new QGridLayout(split1w1);
      QGridLayout* gridS2 = new QGridLayout(split1w2);
      time                = new MTScale(&_raster, split1w2, xscale);
      canvas              = new DrumCanvas(this, split1w2, xscale, yscale);
      vscroll             = new ScrollScale(1, 5, yscale, DRUM_MAPSIZE*TH, Vertical, split1w2);
      int offset = -(division/4);
      canvas->setOrigin(offset, 0);
      time->setOrigin(offset, 0);
      vscroll->showMag(false);

      QValueList<int> mops;
      mops.append(50);
      mops.append(300);
      split2->setSizes(mops);

      gridS2->setRowStretch(1, 100);
      gridS2->setColStretch(0, 100);
      gridS2->addMultiCellWidget(time,  0, 0, 0, 1);
      gridS2->addMultiCellWidget(hLine(split1w2), 1, 1, 0, 1);
      gridS2->addWidget(canvas,  2, 0);
      gridS2->addWidget(vscroll, 2, 1);

      //
      //  Reihenfolge in dlist.c festgeschrieben ("Dcols")
      //
      header = new Header(split1w1, "header");
      header->setFixedHeight(30);
      header->addLabel(tr("M"), 20);
      header->addLabel(tr("Sound"), 100);
      header->addLabel(tr("QNT"));
      header->addLabel(tr("E-Note"));
      header->addLabel(tr("Len"));
      header->addLabel(tr("A-Note"));
      header->addLabel(tr("Ch"));
      header->addLabel(tr("Port"), 60);
      header->addLabel(tr("LV1"));
      header->addLabel(tr("LV2"));
      header->addLabel(tr("LV3"));
      header->addLabel(tr("LV4"));
      new DHeaderTip(header);
      new DWhatsThis(header, header);

      dlist = new DList(header, split1w1, yscale);
      connect(dlist, SIGNAL(keyPressed(int, bool)), canvas, SLOT(keyPressed(int, bool)));
      connect(dlist, SIGNAL(keyReleased(int, bool)), canvas, SLOT(keyReleased(int, bool)));
      connect(dlist, SIGNAL(mapChanged()), canvas, SLOT(mapChanged()));

      gridS1->setRowStretch(1, 100);
      gridS1->setColStretch(0, 100);
      gridS1->addWidget(header, 0, 0);
      gridS1->addWidget(dlist, 1, 0);

      connect(canvas, SIGNAL(newWidth(int)), SLOT(newCanvasWidth(int)));
      connect(song, SIGNAL(songChanged(int)), SLOT(songChanged(int)));
      connect(vscroll, SIGNAL(scrollChanged(int)), canvas, SLOT(setYPos(int)));
      connect(hscroll, SIGNAL(scrollChanged(int)), canvas, SLOT(setXPos(int)));
      connect(hscroll, SIGNAL(scaleChanged(int)),  canvas, SLOT(setXMag(int)));
      connect(srec, SIGNAL(toggled(bool)),         canvas, SLOT(setSteprec(bool)));
      connect(midiin, SIGNAL(toggled(bool)),       canvas, SLOT(setMidiin(bool)));

      connect(vscroll, SIGNAL(scrollChanged(int)),   dlist,   SLOT(setYPos(int)));
      connect(hscroll, SIGNAL(scrollChanged(int)),   time,   SLOT(setXPos(int)));
      connect(hscroll, SIGNAL(scaleChanged(int)), time,   SLOT(setXMag(int)));

      connect(etb, SIGNAL(toolChanged(int)), canvas, SLOT(setTool(int)));
//      connect(etb, SIGNAL(toolChanged(int)), ctrlCanvas, SLOT(setTool(int)));

      connect(canvas, SIGNAL(selectionChanged(int, Event*, Part*)), this,
         SLOT(setSelection(int, Event*, Part*)));
      connect(canvas, SIGNAL(followEvent(int)), SLOT(follow(int)));

      setCaption(canvas->getCaption());
      int s, e;
      canvas->range(&s, &e);
      hscroll->setRange(s, e);

      // connect toolbar
      connect(canvas,  SIGNAL(timeChanged(int)),  SLOT(setTime(int)));
      connect(time,    SIGNAL(timeChanged(int)),  SLOT(setTime(int)));
      connect(toolbar, SIGNAL(quantChanged(int)), canvas,  SLOT(setQuant(int)));
      connect(toolbar, SIGNAL(quantChanged(int)),          SLOT(setQuant(int)));
      connect(toolbar, SIGNAL(rasterChanged(int)),         SLOT(setRaster(int)));
      connect(toolbar, SIGNAL(soloChanged(bool)),          SLOT(soloChanged(bool)));
      connect(info, SIGNAL(valueChanged(NoteInfo::ValType, int)), SLOT(noteinfoChanged(NoteInfo::ValType, int)));

      connect(ctrl, SIGNAL(clicked()), SLOT(addCtrl()));

      QClipboard* cb = QApplication::clipboard();
      connect(cb, SIGNAL(dataChanged()), SLOT(clipboardChanged()));

      clipboardChanged(); // enable/disable "Paste"
      selectionChanged(); // enable/disable "Copy" & "Paste"
      }

//---------------------------------------------------------
//   follow
//---------------------------------------------------------

void DrumEdit::follow(int pos)
      {
      int s, e;
      canvas->range(&s, &e);

      if (pos < e && pos >= s)
            hscroll->setOffset(pos);
      if (pos < s)
            hscroll->setOffset(s);
      }

//---------------------------------------------------------
//   setTime
//---------------------------------------------------------

void DrumEdit::setTime(int tick)
      {
      toolbar->setTime(tick);
      time->setPos(3, tick, false);
      }

//---------------------------------------------------------
//   ~DrumEdit
//---------------------------------------------------------

DrumEdit::~DrumEdit()
      {
      undoRedo->removeFrom(tools);
      }

//---------------------------------------------------------
//   setSelection
//    update Info Line
//---------------------------------------------------------

void DrumEdit::setSelection(int tick, Event* e, Part* p)
      {
      selEvent = (MidiEvent*)e;
      selPart  = (MidiPart*)p;
      selTick  = tick;
      info->setEnabled(e);
      if (e) {
            info->setValues(tick,
               selEvent->lenTick(),
               selEvent->pitch(),
               selEvent->velo(),
               selEvent->veloOff(),
               selEvent->channel()+1);
            }
      selectionChanged();
      }

//---------------------------------------------------------
//   soloChanged
//---------------------------------------------------------

void DrumEdit::soloChanged(bool flag)
      {
      song->setSolo(flag ? canvas->track() : 0);
      }

void DrumEdit::soloChanged(Track* t)
      {
      toolbar->setSolo(t == canvas->track());
      }

//---------------------------------------------------------
//   setRaster
//---------------------------------------------------------

void DrumEdit::setRaster(int val)
      {
      _rasterInit = val;
      MidiEditor::setRaster(val);
      }

//---------------------------------------------------------
//   setQuant
//---------------------------------------------------------

void DrumEdit::setQuant(int val)
      {
      _quantInit = val;
      canvas->setQuant(val);
      MidiEditor::setQuant(val);
      }

//---------------------------------------------------------
//    edit currently selected Event
//---------------------------------------------------------

void DrumEdit::noteinfoChanged(NoteInfo::ValType type, int val)
      {
      if (selEvent == 0) {
            printf("noteinfoChanged while note is zero %d\n", type);
            return;
            }
      MidiEvent* event = new MidiEvent(*selEvent);
      switch(type) {
            case NoteInfo::VAL_TIME:
                  break;
            case NoteInfo::VAL_LEN:
                  event->setLenTick(val);
                  break;
            case NoteInfo::VAL_VELON:
                  event->setVelo(val);
                  break;
            case NoteInfo::VAL_VELOFF:
                  event->setVeloOff(val);
                  break;
            case NoteInfo::VAL_CHANNEL:
//                  event->setChannel(val-1);
                  break;
            case NoteInfo::VAL_PITCH:
                  event->setPitch(val);
                  break;
            }
      midiThread->msgChangeEvent(selEvent, event, selPart);
      }

//---------------------------------------------------------
//   writeStatus
//---------------------------------------------------------

void DrumEdit::writeStatus(int level, Xml& xml) const
      {
      writePartList(level, xml);
      xml.tag(level++, "drumedit");
      MidiEditor::writeStatus(level, xml);

      for (std::list<CtrlEdit*>::const_iterator i = ctrlEditList.begin();
         i != ctrlEditList.end(); ++i) {
            (*i)->writeStatus(level, xml);
            }

      split1->writeStatus(level, xml);
      split2->writeStatus(level, xml);

      header->writeStatus(level, xml);
      xml.intTag(level, "steprec", canvas->steprec());
      xml.intTag(level, "midiin",  canvas->midiin());
      xml.intTag(level, "xpos", hscroll->pos());
      xml.intTag(level, "xmag", hscroll->mag());
      xml.intTag(level, "ypos", vscroll->pos());
      xml.intTag(level, "ymag", vscroll->mag());
      xml.tag(level, "/drumedit");
      }

//---------------------------------------------------------
//   readStatus
//---------------------------------------------------------

void DrumEdit::readStatus(Xml& xml)
      {
      for (;;) {
            Xml::Token token = xml.parse();
            const QString& tag = xml.s1();
            switch (token) {
                  case Xml::Error:
                  case Xml::End:
                        return;
                  case Xml::TagStart:
                        if (tag == "steprec") {
                              int val = xml.parseInt();
                              canvas->setSteprec(val);
                              srec->setOn(val);
                              }
                        else if (tag == "midiin") {
                              int val = xml.parseInt();
                              canvas->setMidiin(val);
                              midiin->setOn(val);
                              }
                        else if (tag == "ctrledit") {
                              CtrlEdit* ctrl = addCtrl();
                              ctrl->readStatus(xml);
                              }
                        else if (tag == split1->name())
                              split1->readStatus(xml);
                        else if (tag == split2->name())
                              split2->readStatus(xml);
                        else if (tag == "midieditor")
                              MidiEditor::readStatus(xml);
                        else if (tag == header->name())
                              header->readStatus(xml);
                        else if (tag == "xmag")
                              hscroll->setMag(xml.parseInt());
                        else if (tag == "xpos")
                              hscroll->setPos(xml.parseInt());
                        else if (tag == "ymag")
                              vscroll->setMag(xml.parseInt());
                        else if (tag == "ypos")
                              vscroll->setPos(xml.parseInt());
                        else
                              xml.unknown("DrumEdit");
                        break;
                  case Xml::TagEnd:
                        if (tag == "drumedit") {
                              _quantInit  = _quant;
                              _rasterInit = _raster;
                              toolbar->setRaster(_raster);
                              toolbar->setQuant(_quant);
                              canvas->setQuant(_quant);
                              return;
                              }
                  default:
                        break;
                  }
            }
      }

//---------------------------------------------------------
//   readConfiguration
//---------------------------------------------------------

void DrumEdit::readConfiguration(Xml& xml)
      {
      for (;;) {
            Xml::Token token = xml.parse();
            const QString& tag = xml.s1();
            switch (token) {
                  case Xml::Error:
                  case Xml::End:
                        return;
                  case Xml::TagStart:
                        if (tag == "quant")
                              _quantInit = xml.parseInt();
                        else if (tag == "raster")
                              _rasterInit = xml.parseInt();
                        else
                              xml.unknown("DrumEdit");
                        break;
                  case Xml::TagEnd:
                        if (tag == "drumedit") {
                              return;
                              }
                  default:
                        break;
                  }
            }
      }

//---------------------------------------------------------
//   writeConfiguration
//---------------------------------------------------------

void DrumEdit::writeConfiguration(int level, Xml& xml)
      {
      xml.tag(level++, "drumedit");
      xml.intTag(level, "quant", _quantInit);
      xml.intTag(level, "raster", _rasterInit);
      xml.tag(level, "/drumedit");
      }

//---------------------------------------------------------
//   load
//---------------------------------------------------------

void DrumEdit::load()
      {
      QString fn = getOpenFileName("drummaps", map_file_pattern,
         this, "Muse: Load Drum Map");
      if (fn.isEmpty())
            return;
      bool popenFlag;
      FILE* f = fileOpen(this, fn, ".map", "r", popenFlag, true);
      if (f == 0)
            return;

      Xml xml(f);
      int mode = 0;
      for (;;) {
            Xml::Token token = xml.parse();
            const QString& tag = xml.s1();
            switch (token) {
                  case Xml::Error:
                  case Xml::End:
                        return;
                  case Xml::TagStart:
                        if (mode == 0 && tag == "muse")
                              mode = 1;
                        else if (mode == 1 && tag == "drummap") {
                              readDrumMap(xml, true);
                              mode = 0;
                              }
                        else
                              xml.unknown("DrumEdit");
                        break;
                  case Xml::Attribut:
                        break;
                  case Xml::TagEnd:
                        if (!mode && tag == "muse")
                              goto ende;
                  default:
                        break;
                  }
            }
ende:
      if (popenFlag)
            pclose(f);
      else
            fclose(f);
      dlist->redraw();
      canvas->redraw();
      }

//---------------------------------------------------------
//   save
//---------------------------------------------------------

void DrumEdit::save()
      {
      QString fn = getSaveFileName("drummaps", map_file_pattern,
        this, "MusE: Store Drum Map");
      if (fn.isEmpty())
            return;
      bool popenFlag;
      FILE* f = fileOpen(this, fn, ".map", "w", popenFlag, false, true);
      if (f == 0)
            return;
      Xml xml(f);
      xml.header();
      xml.tag(0, "muse version=\"1.0\"");
      writeDrumMap(1, xml, true);
      xml.tag(1, "/muse");

      if (popenFlag)
            pclose(f);
      else
            fclose(f);
      }

//---------------------------------------------------------
//   cmd
//    pulldown menu commands
//---------------------------------------------------------

void DrumEdit::cmd(int cmd)
      {
      switch(cmd) {
            case DrumCanvas::CMD_LOAD:
                  load();
                  break;
            case DrumCanvas::CMD_SAVE:
                  save();
                  break;
            default:
                  ((DrumCanvas*)(canvas))->cmd(cmd);
                  break;
            }
      }

//---------------------------------------------------------
//   clipboardChanged
//---------------------------------------------------------

void DrumEdit::clipboardChanged()
      {
      QMimeSource* ms = QApplication::clipboard()->data();
      if (ms && ms->format(0)) {
            bool flag = strcmp(ms->format(0), "text/eventlist;charset=UTF-8") == 0;
            menuEdit->setItemEnabled(DrumCanvas::CMD_PASTE, flag);
            }
      }

//---------------------------------------------------------
//   selectionChanged
//---------------------------------------------------------

void DrumEdit::selectionChanged()
      {
      bool flag = canvas->selectionSize() > 0;
      menuEdit->setItemEnabled(DrumCanvas::CMD_CUT, flag);
      menuEdit->setItemEnabled(DrumCanvas::CMD_COPY, flag);
      menuEdit->setItemEnabled(DrumCanvas::CMD_DEL, flag);
      }

//---------------------------------------------------------
//   addCtrl
//---------------------------------------------------------

CtrlEdit* DrumEdit::addCtrl()
      {
      CtrlEdit* ctrlEdit = new CtrlEdit(split1, this, xscale, true, "drumCtrlEdit");
      connect(hscroll,  SIGNAL(scrollChanged(int)), ctrlEdit, SLOT(setXPos(int)));
      connect(hscroll,  SIGNAL(scaleChanged(int)),  ctrlEdit, SLOT(setXMag(int)));
      connect(ctrlEdit, SIGNAL(timeChanged(int)),   SLOT(setTime(int)));
      connect(ctrlEdit, SIGNAL(destroyedCtrl(CtrlEdit*)), SLOT(removeCtrl(CtrlEdit*)));
      connect(ctrlEdit, SIGNAL(yposChanged(int)), toolbar, SLOT(setInt(int)));

      ctrlEdit->setXPos(hscroll->pos());
      ctrlEdit->setXMag(hscroll->getScaleValue());

      ctrlEdit->setCanvasWidth(canvas->width());

      ctrlEdit->show();
      ctrlEditList.push_back(ctrlEdit);
      return ctrlEdit;
      }

//---------------------------------------------------------
//   removeCtrl
//---------------------------------------------------------

void DrumEdit::removeCtrl(CtrlEdit* ctrl)
      {
      for (std::list<CtrlEdit*>::iterator i = ctrlEditList.begin();
         i != ctrlEditList.end(); ++i) {
            if (*i == ctrl) {
                  ctrlEditList.erase(i);
                  break;
                  }
            }
      }

//---------------------------------------------------------
//   newCanvasWidth
//---------------------------------------------------------

void DrumEdit::newCanvasWidth(int w)
      {
      for (std::list<CtrlEdit*>::iterator i = ctrlEditList.begin();
         i != ctrlEditList.end(); ++i) {
            (*i)->setCanvasWidth(w);
            }
      }

