//=========================================================
//  MusE
//  Linux Music Editor
//    $Id: mess.cpp,v 1.6 2002/02/27 08:48:09 muse Exp $
//  (C) Copyright 2001 Werner Schweer (ws@seh.de)
//=========================================================

#include "mess.h"
#include <stdio.h>
#include <string.h>
#include <poll.h>
#include <sys/mman.h>
#include <pthread.h>
#include <sys/time.h>
#include <errno.h>
#include <unistd.h>

#include "ladspa.h"

extern void __init();


//---------------------------------------------------------
//   doSetuid
//    Restore the effective UID to its original value.
//---------------------------------------------------------

static void doSetuid(uid_t euid, uid_t ruid)
      {
      int status;

#ifdef _POSIX_SAVED_IDS
      status = seteuid (euid);
#else
      status = setreuid (ruid, euid);
#endif
      if (status < 0) {
            fprintf(stderr, "Couldn't set uid.\n");
            }
      }

//---------------------------------------------------------
//   undoSetuid
//    Set the effective UID to the real UID.
//---------------------------------------------------------

static void undoSetuid(uid_t euid, uid_t ruid)
      {
      int status;

#ifdef _POSIX_SAVED_IDS
      status = seteuid (ruid);
#else
      status = setreuid (euid, ruid);
#endif
      if (status < 0) {
            fprintf (stderr, "Couldn't set uid.\n");
            exit (status);
            }
      }

//---------------------------------------------------------
//   curTime
//---------------------------------------------------------

static double curTime()
      {
      struct timeval t;
      struct timezone tz;
      gettimeofday(&t, &tz);
      return (double)((double)t.tv_sec + (t.tv_usec / 1000000.0));
      }

//---------------------------------------------------------
//   midi_run
//---------------------------------------------------------

static void* midi_run(void* d)
      {
      Mess* s = (Mess*) d;
      s->midiRun();
      pthread_exit(0);
      }

#ifdef RTCAP
//---------------------------------------------------------
//   getCapabilities
//---------------------------------------------------------

void getCapabilities()
      {
#ifdef __linux__
      const char* napp = getenv("GIVERTCAP");
      system(napp ? napp : "givertcap");
#endif // __linux__
      }
#endif

//---------------------------------------------------------
//   midiRun
//---------------------------------------------------------

void Mess::midiRun()
      {
#ifdef RTCAP
      extern void getCapabilities();
      getCapabilities();
#else
      doSetuid(euid, ruid);
#endif
      struct sched_param sp;
      pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, 0);
      pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, 0);

      sp.sched_priority = 20;
	if (sched_setscheduler(0, SCHED_RR, &sp) == 0) {
            fprintf(stderr, "mess: %s: running as realtime process now (priority %d)\n",
               _name, 20);
            }
	else {
            fprintf(stderr, "mess: %s: Can't get realtime priority: %s\n",
               _name, strerror(errno));
            }
      if (mlockall(MCL_FUTURE))
            perror("mess: Cannot lock memory");
#ifndef RTCAP
      undoSetuid(euid, ruid);
#endif

      MEvent* ev;
      for (;;) {
            int n = poll(pfd, npfd, -1);  // wait infinite for event
            if (n <= 0)
                  continue;
            for(;;) {
                  snd_seq_event_t* event;
                  int rv = snd_seq_event_input(alsaSeq, &event);
                  if (rv < 0)
                        break;

                  pthread_mutex_lock(&lock);
                  double frame = startFrame;
                  pthread_mutex_unlock(&lock);

                  if (event->type == SND_SEQ_EVENT_PORT_START)
                        continue;

                  unsigned pos = int ((curTime() - frame) * _sampleRate);

                  ev = 0;
                  switch(event->type) {
//                        case SND_SEQ_EVENT_PORT_START:
                        case SND_SEQ_EVENT_PORT_EXIT:
                              break;

                        case SND_SEQ_EVENT_NOTEON:
                        case SND_SEQ_EVENT_KEYPRESS:
                        case SND_SEQ_EVENT_NOTEOFF:
                              ev = new MEvent(pos, event->type,
                                      event->data.note.channel,
                                      event->data.note.note,
                                      event->data.note.velocity);
                              break;

                        case SND_SEQ_EVENT_CONTROL14:
                        case SND_SEQ_EVENT_NONREGPARAM:
                        case SND_SEQ_EVENT_REGPARAM:
                        case SND_SEQ_EVENT_CONTROLLER:
                        case SND_SEQ_EVENT_PGMCHANGE:
                        case SND_SEQ_EVENT_CHANPRESS:
                              ev = new MEvent(pos, event->type,
                                      event->data.control.channel,
                                      event->data.control.param,
                                      event->data.control.value);
                              break;


                        case SND_SEQ_EVENT_PITCHBEND:
                              ev = new MEvent(pos, event->type,
                                      event->data.control.channel,
                                      (event->data.control.value >> 7) & 0x7f,
                                      event->data.control.value & 0x7f);
                              break;

                        case SND_SEQ_EVENT_SYSEX:
                              {
                              int len = event->data.ext.len;
                              unsigned char* data = new unsigned char[len];
                              memcpy(data, event->data.ext.ptr, len);
                              ev = new MEvent(pos, event->type, 0, 0, 0, data, len);
                              }
                              break;

                        default:
                              printf("mess: ALSA Seq input: type %d not handled\n", event->type);
                              break;
                        }
                  snd_seq_free_event(event);
                  if (ev) {
                        if (frame == 0.0) {
                              // audio not running
                              // allow for initialize events
                              processEvent(ev);
                              }
                        else {
                              pthread_mutex_lock(&lock);
                              events.push_back(ev);
                              pthread_mutex_unlock(&lock);
                              }
                        }
                  }
            }
      }

//---------------------------------------------------------
//   Mess
//---------------------------------------------------------

Mess::Mess(const char* cn, int chn)
   : _className(strdup(cn))
      {
      _channels   = chn;
      outputPorts = new float*[_channels];
      _name       = 0;
      alsaSeq     = 0;
      pfd         = 0;
      npfd        = 0;
      startFrame  = 0.0;
      _sampleRate = 44100;
      pthread_mutex_init(&lock, 0);
      }

//---------------------------------------------------------
//   Mess
//---------------------------------------------------------

#if (SND_LIB_MAJOR==0) && (SND_LIB_MINOR==9)
Mess::~Mess()
      {
      delete[] outputPorts;
      if (_name)
            delete _name;
      if (_className)
            delete _className;
      if (alsaSeq) {
#ifdef snd_seq_port_info_alloca
            int error = snd_seq_delete_port(alsaSeq, _alsaPort.port);
#else
            snd_seq_port_info_t *pinfo;
            pinfo = (snd_seq_port_info_t*)alloca(sizeof(*pinfo));
	      memset(pinfo, 0, sizeof(*pinfo));
	      pinfo->client = _alsaPort.client;
	      pinfo->port   = _alsaPort.port;
            int error     = snd_seq_delete_port(alsaSeq, pinfo);
#endif
            if (error < 0) {
                  fprintf(stderr, "ALSA: cannot delete port: %s\n",
                     snd_strerror(error));
                  }
            }
      }
#endif

//---------------------------------------------------------
//   Mess
//---------------------------------------------------------

#if (SND_LIB_MAJOR==0) && (SND_LIB_MINOR==5)
Mess::~Mess()
      {
      delete outputPorts;
      if (_name)
            delete _name;
      if (_className)
            delete _className;
      if (alsaSeq) {
            snd_seq_port_info_t* pinfo = new snd_seq_port_info_t;
	      memset(pinfo, 0, sizeof(*pinfo));
	      pinfo->client = _alsaPort.client;
	      pinfo->port   = _alsaPort.port;
            int error     = snd_seq_delete_port(alsaSeq, pinfo);
            if (error < 0) {
                  fprintf(stderr, "ALSA: cannot delete port: %s\n",
                     snd_strerror(error));
                  }
            delete pinfo;
            }
      }
#endif

//---------------------------------------------------------
//   setName
//---------------------------------------------------------

void Mess::setName(const char* s)
      {
      _name = strdup(s);
      }

//---------------------------------------------------------
//   registerAlsa
//---------------------------------------------------------

#if (SND_LIB_MAJOR==0) && (SND_LIB_MINOR==9)
void Mess::registerAlsa()
      {
      //-----------------------------------------
      //  connect to ALSA and get poll
      //  descriptors
      //-----------------------------------------

      if (alsaSeq == 0) {
            int error = snd_seq_open(&alsaSeq, "hw", SND_SEQ_OPEN_DUPLEX|SND_SEQ_NONBLOCK, 0);
            if (error < 0) {
                  fprintf(stderr, "Could not open ALSA sequencer: %s\n",
                     snd_strerror(error));
                  alsaSeq = 0;
                  return;
                  }
            }

      snd_seq_set_client_name(alsaSeq, _className);
      npfd = snd_seq_poll_descriptors_count(alsaSeq, POLLIN);
      pfd  = new pollfd[npfd];
      snd_seq_poll_descriptors(alsaSeq, pfd, npfd, POLLIN);
      for (int i = 0; i < npfd; ++i) {
            pfd[i].events  = POLLIN;
            pfd[i].revents = 0;
            }

      //-----------------------------------------
      //  find an unused alsa port name
      //-----------------------------------------

      char buffer[256];
      for (int i = 1;; ++i) {
            bool found = false;
            sprintf(buffer, "%s-%d", _className, i);

            snd_seq_client_info_t* cinfo;
#ifdef snd_seq_client_info_alloca
            snd_seq_client_info_alloca(&cinfo);
            snd_seq_client_info_set_client(cinfo, -1);
#else
            cinfo = (snd_seq_client_info_t*)alloca(sizeof(*cinfo));
            memset(cinfo, 0, sizeof(*cinfo));
            cinfo->client = -1;
#endif
            while (snd_seq_query_next_client(alsaSeq, cinfo) >= 0) {
#ifdef SND_SEQ_GROUP_SYSTEM
                  if (strcmp(cinfo->group, SND_SEQ_GROUP_SYSTEM) == 0)
                        continue;
#endif
                  snd_seq_port_info_t *pinfo;
#ifdef snd_seq_port_info_alloca
                  snd_seq_port_info_alloca(&pinfo);
                  snd_seq_port_info_set_client(pinfo, snd_seq_client_info_get_client(cinfo));
                  snd_seq_port_info_set_port(pinfo, -1);
#else
                  pinfo = (snd_seq_port_info_t*)alloca(sizeof(*pinfo));
                  memset(pinfo, 0, sizeof(*pinfo));
                  pinfo->client = cinfo->client;
                  pinfo->port = -1;
#endif
                  while (snd_seq_query_next_port(alsaSeq, pinfo) >= 0) {
                        const char* portname;
#ifdef snd_seq_port_info_alloca
                        portname = snd_seq_port_info_get_name(pinfo);
#else
                        portname = pinfo->name;
#endif
// printf("compare port <%s> <%s>\n", portname, buffer);
                        if (strcmp(portname, buffer) == 0) {
                              found = true;
                              break;
                              }
                        }
                  if (found)
                        break;
                  }
            if (!found)
                  break;
            }
      _name = strdup(buffer);

      // search muse port
      bool found = false;
      snd_seq_client_info_t* cinfo;
#ifdef snd_seq_client_info_alloca
      snd_seq_client_info_alloca(&cinfo);
      snd_seq_client_info_set_client(cinfo, -1);
#else
      cinfo = (snd_seq_client_info_t*)alloca(sizeof(*cinfo));
      memset(cinfo, 0, sizeof(*cinfo));
      cinfo->client = -1;
#endif
      while (snd_seq_query_next_client(alsaSeq, cinfo) >= 0) {
            found = false;
            snd_seq_port_info_t *pinfo;
#ifdef snd_seq_port_info_alloca
            snd_seq_port_info_alloca(&pinfo);
            snd_seq_port_info_set_client(pinfo, snd_seq_client_info_get_client(cinfo));
            snd_seq_port_info_set_port(pinfo, -1);
#else
            pinfo = (snd_seq_port_info_t*)alloca(sizeof(*pinfo));
            memset(pinfo, 0, sizeof(*pinfo));
            pinfo->client = cinfo->client;
            pinfo->port = -1;
#endif
            while (snd_seq_query_next_port(alsaSeq, pinfo) >= 0) {
                  const char* portname;
#ifdef snd_seq_port_info_alloca
                  portname = snd_seq_port_info_get_name(pinfo);
#else
                  portname = pinfo->name;
#endif
                  if (strcmp(portname, "MusE Port 0") == 0) {
                        found = true;
                  snd_seq_port_info_set_client(pinfo, snd_seq_client_info_get_client(cinfo));

                        _musePort.port   = snd_seq_port_info_get_port(pinfo);
                        _musePort.client = snd_seq_client_info_get_client(cinfo);
                        break;
                        }
                  }
            if (found)
                  break;
            }
      if (!found) {
            printf("muse port not found!\n");
            }

      //-----------------------------------------
      //    create port to alsa sequencer
      //-----------------------------------------

      int port  = snd_seq_create_simple_port(alsaSeq, _name,
         SND_SEQ_PORT_CAP_WRITE | SND_SEQ_PORT_CAP_SUBS_WRITE,
      SND_SEQ_PORT_TYPE_APPLICATION);
// printf("create new alsa port <%s>\n", _name);
      if (port < 0) {
            perror("create port");
            return;
            }
      _alsaPort.port   = port;
      _alsaPort.client = snd_seq_client_id(alsaSeq);

      // create the midi thread
      int err = pthread_create(&midiThread, 0, midi_run, this);
      if (err) {
            printf("Mess: Couldn't create midi thread: %s\n",
               strerror(errno));
            }
      }
#endif

#if (SND_LIB_MAJOR==0) && (SND_LIB_MINOR==5)
//---------------------------------------------------------
//   registerAlsa
//    ALSA 5 version
//---------------------------------------------------------

void Mess::registerAlsa()
      {
      //-----------------------------------------
      //  connect to ALSA and get poll
      //  descriptors
      //-----------------------------------------
      if (alsaSeq == 0) {
            int error = snd_seq_open(&alsaSeq, SND_SEQ_OPEN|SND_SEQ_NONBLOCK);
            if (error < 0) {
                  fprintf(stderr, "Could not open ALSA sequencer: %s\n",
                     snd_strerror(error));
                  alsaSeq = 0;
                  return;
                  }
            }
      int alsaSeqFd = snd_seq_file_descriptor(alsaSeq);
      npfd         = 1;
      pfd          = new pollfd;
      pfd->fd      = alsaSeqFd;
      pfd->events  = POLLIN;
      pfd->revents = 0;

      //-----------------------------------------
      //  find an unused alsa port name
      //-----------------------------------------

      snd_seq_system_info_t sysinfo;
      int error = snd_seq_system_info(alsaSeq, &sysinfo);
      if (error < 0) {
            fprintf(stderr, "Cound not get ALSA sequencer info: %s\n",
               snd_strerror(error));
            return;
            }

      char buffer[256];
      buffer[0] = 0;
      for (int i = 1;; ++i) {
            bool found = false;
            sprintf(buffer, "%s-%d", _className, i);
            for (int client = 0; client < sysinfo.clients; ++client) {
                  snd_seq_client_info_t cinfo;
                  if (snd_seq_get_any_client_info(alsaSeq, client, &cinfo) < 0)
                        continue;

                  for (int port = 0; port < sysinfo.ports; port++) {
                        snd_seq_port_info_t pinfo;

                        if (snd_seq_get_any_port_info(alsaSeq, client, port, &pinfo) < 0)
                              continue;
                        const char* portname = pinfo.name;
                        if (strcmp(portname, buffer) == 0) {
                              found = true;
                              break;
                              }
                        }
                  if (found)
                        break;
                  }
            if (!found)
                  break;
            }
      _name = strdup(buffer);

      // search muse port
      bool found;
      for (int client = 0; client < sysinfo.clients; ++client) {
            found = false;
            snd_seq_client_info_t cinfo;
            if (snd_seq_get_any_client_info(alsaSeq, client, &cinfo) < 0)
                  continue;
            for (int port = 0; port < sysinfo.ports; port++) {
                  snd_seq_port_info_t pinfo;

                  if (snd_seq_get_any_port_info(alsaSeq, client, port, &pinfo) < 0)
                        continue;
                  const char* portname = pinfo.name;
                  if (strcmp(portname, "MusE Port 0") == 0) {
                        found = true;
                        _musePort.port   = pinfo.port;
                        _musePort.client = cinfo.client;
                        break;
                        }
                  }
            if (found)
                  break;
            }
      if (!found) {
            printf("muse port not found!\n");
            }

      //-----------------------------------------
      //    create port to alsa sequencer
      //-----------------------------------------

      int port  = snd_seq_create_simple_port(alsaSeq, _name,
         SND_SEQ_PORT_CAP_WRITE | SND_SEQ_PORT_CAP_SUBS_WRITE,
      SND_SEQ_PORT_TYPE_APPLICATION);
// printf("create new alsa port <%s>\n", _name);
      if (port < 0) {
            perror("create port");
            return;
            }
      _alsaPort.port   = port;
      _alsaPort.client = snd_seq_client_id(alsaSeq);

      // create the midi thread
      int err = pthread_create(&midiThread, 0, midi_run, this);
      if (err) {
            printf("Mess: Couldn't create midi thread: %s\n",
               strerror(errno));
            }
      }
#endif

//---------------------------------------------------------
//   getPatchName
//---------------------------------------------------------

const char* Mess::getPatchName(int /*channel*/, int /*hbank*/, int /*lbank*/,
   int /*prog*/, MType /*type*/)
      {
      return "???";
      }

//---------------------------------------------------------
//    sysex
//---------------------------------------------------------

void Mess::sendSysex(const unsigned char* p, int n)
      {
      snd_seq_event_t event;
      int len             = n + sizeof(event) + 2;
      char* buf           = new char[len];
      memset(&event, 0, sizeof(event));
      event.type          = SND_SEQ_EVENT_SYSEX;
      event.flags         = SND_SEQ_EVENT_LENGTH_VARIABLE;
      event.data.ext.len  = n + 2;
      event.data.ext.ptr  = (void*)(buf + sizeof(event));
      event.tag           = 0;
      event.queue         = SND_SEQ_QUEUE_DIRECT;
      event.dest          = _musePort;
      event.source        = _alsaPort;

      memcpy(buf, &event, sizeof(event));
      char* pp = buf + sizeof(event);
      *pp++ = 0xf0;
      memcpy(pp, p, n);
      pp += n;
      *pp = 0xf7;

#if (SND_LIB_MAJOR==0) && (SND_LIB_MINOR==5)
      int alsaSeqFd = snd_seq_file_descriptor(alsaSeq);
      int error = write(alsaSeqFd, buf, len);
      if (error != len)
            fprintf(stderr, "sysex: midi write error: %s\n", snd_strerror(error));
#endif

#if (SND_LIB_MAJOR==0) && (SND_LIB_MINOR==9)
      int error = snd_seq_event_output_direct(alsaSeq, &event);
      if (error < 0)
            fprintf(stderr, "sysex: midi write error: %s\n", snd_strerror(error));
#endif
      delete buf;
      }

//---------------------------------------------------------
//   processreplace
//---------------------------------------------------------

void Mess::processreplace(int len)
      {
      pthread_mutex_lock(&lock);
      startFrame = curTime();
      pthread_mutex_unlock(&lock);

      int offset = 0;
      for (int channel = 0; channel < _channels; ++channel)
            memset(outputPorts[channel], 0, len * sizeof(float));

      //---------------------------------------------------
      //  process all midi events for this buffer
      //---------------------------------------------------

      for (;;) {
            //---------------------------------------------
            //  get next midi event
            //---------------------------------------------
            pthread_mutex_lock(&lock);

            if (events.empty()) {
                  pthread_mutex_unlock(&lock);
                  break;
                  }
            MEvent* e = events.front();
            int samplePos = e->samplePos();
            if (samplePos >= len) {
                  for (iMEvent i = events.begin(); i != events.end(); ++i) {
                        (*i)->adjustSamplePos(len);
                        }
                  pthread_mutex_unlock(&lock);
                  break;
                  }
            events.erase(events.begin());
            pthread_mutex_unlock(&lock);

            //---------------------------------------------
            //  generate samples up to this midi event
            //---------------------------------------------

            int samplesToSynthesize = samplePos - offset;

            write(samplesToSynthesize, outputPorts, offset);
            offset += samplesToSynthesize;
            processEvent(e);
            delete e;
            }

      //---------------------------------------------------
      //  any samples left?
      //---------------------------------------------------

      int samplesToSynthesize = len - offset;
      write(samplesToSynthesize, outputPorts, offset);
      }

//---------------------------------------------------------
//   MessMono
//---------------------------------------------------------

MessMono::MessMono(const char* name, int channels)
   : Mess(name, channels)
      {
      }

MessMono::~MessMono()
      {
      }

//---------------------------------------------------------
//   midiNoteOn
//---------------------------------------------------------

void MessMono::midiNoteOn(int channel, int pitch, int velo)
      {
      if (velo == 0) {
            midiNoteOff(channel, pitch);
            return;
            }
      pitchStack.push_back(PitchVelo(channel, pitch, velo));
      noteon(channel, pitch, velo);
      }

//---------------------------------------------------------
//   midiNoteOff
//---------------------------------------------------------

void MessMono::midiNoteOff(int channel, int pitch)
      {
      if (pitchStack.empty())
            return;
      if (pitchStack.back().pitch == pitch) {
            pitchStack.pop_back();
            if (pitchStack.empty()) {
                  noteoff(channel, pitch);
                  return;
                  }
            PitchVelo pv = pitchStack.back();
            noteon(pv.channel, pv.pitch, pv.velo);  // change pitch
            return;
            }
      for (std::list<PitchVelo>::iterator i = pitchStack.begin();
         i != pitchStack.end(); ++i) {
            if ((*i).pitch == pitch) {
                  pitchStack.erase(i);
                  return;
                  }
            }
      // no noteon found
      // emergency stop:
      noteoff(channel, pitch);
      }

//---------------------------------------------------------
//   processEvents
//    process received midi event
//---------------------------------------------------------

void MessMono::processEvent(MEvent* ev)
      {
      switch(ev->type()) {
            case SND_SEQ_EVENT_NOTEON:
            case SND_SEQ_EVENT_KEYPRESS:
                  midiNoteOn(ev->chan(), ev->dataA(), ev->dataB());
                  break;

            case SND_SEQ_EVENT_NOTEOFF:
                  midiNoteOff(ev->chan(), ev->dataA());
                  break;

		case SND_SEQ_EVENT_CONTROL14:
		case SND_SEQ_EVENT_NONREGPARAM:
		case SND_SEQ_EVENT_REGPARAM:
		case SND_SEQ_EVENT_CONTROLLER:
			switch(ev->dataA()) {
				case 0x62:	ctrlLo = ev->dataB(); break;
				case 0x63:	ctrlHi = ev->dataB(); break;
				case 0x06:	dataHi = ev->dataB(); break;
				case 0x026:	dataLo = ev->dataB(); break;
			}
			if(ev->dataA() == 0x06)
				setController(ev->chan(), ctrlLo+ctrlHi*128, dataLo+dataHi*128);
			break;

		case SND_SEQ_EVENT_SYSEX:
			sysex(ev->data(), ev->dataLen());
			break;
            case SND_SEQ_EVENT_CHANPRESS:
            case SND_SEQ_EVENT_PITCHBEND:
            case SND_SEQ_EVENT_PGMCHANGE:
                  break;
            default:
                  printf("MessMono: event type %d not processed\n", ev->type());
                  break;
            }
      }

//=========================================================
//    LADSPA
//=========================================================

static Mess* (*xinstantiate)(const char* name);

//---------------------------------------------------------
//   instantiate
//    construct a new synthesizer instance
//---------------------------------------------------------

static LADSPA_Handle instantiate(const LADSPA_Descriptor* descr, unsigned long sr)
      {
      Mess* synth = xinstantiate(descr->Label);
      synth->setSampleRate(sr);
      if (synth->init()) {
            delete synth;
            synth = 0;
            }
      return synth;
      }

//---------------------------------------------------------
//   run
//---------------------------------------------------------

static void run(LADSPA_Handle instance, unsigned long sampleCount)
      {
      ((Mess *)instance)->processreplace(sampleCount);
      }

//---------------------------------------------------------
//   connectPort
//    Connect a port to a data location.
//---------------------------------------------------------

static void connect(LADSPA_Handle Instance, unsigned long port, float* data)
      {
      ((Mess *)Instance)->setPort(port, data);
      }

//---------------------------------------------------------
//   activate
//---------------------------------------------------------

static void activate(LADSPA_Handle instance)
      {
      ((Mess *)instance)->init();
      ((Mess *)instance)->registerAlsa();
      }

//---------------------------------------------------------
//   cleanup
//---------------------------------------------------------

static void cleanup(LADSPA_Handle instance)
      {
      delete (Mess *)instance;
      }

//---------------------------------------------------------
//   portNames
//---------------------------------------------------------

static const char* portNames[] = {
      "Output",
      "Output",
      };

//---------------------------------------------------------
//   portDescriptors
//---------------------------------------------------------

static LADSPA_PortDescriptor portDescriptors[] = {
      LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO,
      LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO,
      };

//---------------------------------------------------------
//   portRangeHints
//---------------------------------------------------------

static LADSPA_PortRangeHint portRangeHints[] = {
      { 0, 0.0, 0.0 },
      { 0, 0.0, 0.0 },
      };

static LADSPA_Descriptor descriptor = {
      1223,
      "label",
      LADSPA_PROPERTY_HARD_RT_CAPABLE,
      "name",
      "maker",
      "copyright",
      1,
      portDescriptors,
      portNames,
      portRangeHints,
      0,                // impl. data
      instantiate,
      connect,
      activate,
      run,
      0,
      0,
      0,
      cleanup
      };

//---------------------------------------------------------
//   ladspa_descriptor
//    Return a descriptor of the requested plugin type.
//---------------------------------------------------------

const LADSPA_Descriptor* ladspa_descriptor(unsigned long i)
      {
      __init();         // create ladspa descritpor
      return (i == 0) ? &descriptor : 0;
      }

//---------------------------------------------------------
//   initMess
//    ports  -  1-Mono  2-Stereo
//---------------------------------------------------------

void Mess::initMess(unsigned long id, const char* label, const char* maker,
   const char* name, const char* copyright, unsigned long ports,
   Mess* (inst)(const char*))
      {
      descriptor.UniqueID  = id;
      descriptor.Label     = label;
      descriptor.Name      = name;
      descriptor.Maker     = maker;
      descriptor.Copyright = copyright;
      descriptor.PortCount = ports;
      xinstantiate          = inst;
      }

