/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Richard Hult <rhult@codefactory.se>
 * Copyright (C) 2001 Mikael Hallendal <micke@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Richard Hult
 * 
 */

#include <gtk/gtk.h>
#include "util/type-utils.h"
#include "id-map.h"

struct _IdMapPriv {
	GHashTable *id_hash;
	gint        next_id;
};

static void id_map_init		(IdMap		 *map);
static void id_map_class_init	(IdMapClass	 *klass);
static void id_map_destroy      (GtkObject       *object);

GNOME_CLASS_BOILERPLATE (IdMap, id_map, GtkObject, gtk_object);

static void
id_map_class_init (IdMapClass *klass)
{
	GtkObjectClass *object_class;
	
	object_class = (GtkObjectClass*) klass;
	object_class->destroy = id_map_destroy;
	
	parent_class = gtk_type_class (GTK_TYPE_OBJECT);
}

static void
id_map_init (IdMap *map)
{
	IdMapPriv *priv;
	
	priv = g_new0 (IdMapPriv, 1);
	map->priv = priv;

	priv->id_hash = g_hash_table_new (g_int_hash, g_int_equal);
	priv->next_id = 0;
}

static void
destroy_id (gpointer   key, 
	    gpointer   value, 
	    gpointer   user_data)
{
	g_free (key);
}

static void
id_map_destroy (GtkObject *object)
{
	IdMap *id_map;

	id_map = ID_MAP (object);
	
	g_hash_table_foreach (id_map->priv->id_hash, destroy_id, NULL);
	g_hash_table_destroy (id_map->priv->id_hash);
	
	g_free (id_map->priv);
	id_map->priv = NULL;

	GNOME_CALL_PARENT_HANDLER (GTK_OBJECT_CLASS, destroy, (object));
}

IdMap *
id_map_new (GNOME_MrProject_Id first_id)
{
	IdMap *map;

	map = gtk_type_new (TYPE_ID_MAP);
	map->priv->next_id = first_id;

	return map;
}

GNOME_MrProject_Id
id_map_insert (IdMap *map, gpointer data)
{
	gint     id, *key;

	id = map->priv->next_id++;
	while (1) {
		if (!g_hash_table_lookup_extended (map->priv->id_hash,
						   &id,
						   NULL,
						   NULL)) {
			/* We found an unsed id. */
			key = g_new (gint, 1);
			*key = id; 
			g_hash_table_insert (map->priv->id_hash, key, data);
			return id;
		}
		id++;
	}

	return -1;
}

gboolean
id_map_insert_id (IdMap *map, GNOME_MrProject_Id id, gpointer data)
{
	gint     *key, id_int;

	id_int = (int) id;
	if (g_hash_table_lookup_extended (map->priv->id_hash,
					  &id_int,
					  NULL,
					  NULL)) {
		/* We already have an entry for this id. */
		return FALSE;
	}

	key = g_new (gint, 1);
	*key = id_int;
	g_hash_table_insert (map->priv->id_hash, key, data);

	return TRUE;
}

gboolean
id_map_remove (IdMap *map, GNOME_MrProject_Id id)
{
	gint     *key, id_int;

	id_int = (int) id;
	if (!g_hash_table_lookup_extended (map->priv->id_hash,
					  &id_int,
					  (gpointer *) &key,
					  NULL)) {
		/* We don't have this id stored. */
		return FALSE;
	}
	
	g_hash_table_remove (map->priv->id_hash, &id_int);
	g_free (key);

	/* Set next id to the one we removed so that we will try to use
	 * it next.
	 */
	if (id_int < map->priv->next_id) {
		map->priv->next_id = id_int;
	}
	
	return TRUE;
}

gpointer
id_map_lookup (IdMap *map, GNOME_MrProject_Id id)
{
	gint id_int;

	id_int = (int) id;
	return g_hash_table_lookup (map->priv->id_hash, &id_int);
}

void
id_map_foreach (IdMap *map, GHFunc func, gpointer user_data)
{
	g_hash_table_foreach (map->priv->id_hash, func, user_data);
}

gint
id_map_size (IdMap *map) 
{
        return g_hash_table_size (map->priv->id_hash);
}

static void
get_key (gpointer key, gpointer value, gpointer user_data) 
{
        GSList   **list;
        
        g_return_if_fail (key != NULL);
        g_return_if_fail (user_data != NULL);
        
        list = (GSList **) user_data;
        
        *list = g_slist_prepend (*list, key);
}

GSList *             
id_map_get_keys (IdMap *map)
{
        GSList       *list = NULL;

        g_hash_table_foreach (map->priv->id_hash, get_key, &list);
        
        return list;
}

static void
get_object (gpointer key, gpointer value, gpointer user_data) 
{
        GSList   **list;
        
        g_return_if_fail (value != NULL);
        g_return_if_fail (user_data != NULL);
        
        list = (GSList **) user_data;
        
        *list = g_slist_prepend (*list, value);
}

GSList *             
id_map_get_objects (IdMap *map)
{
        GSList       *list = NULL;

        g_hash_table_foreach (map->priv->id_hash, get_object, &list);
        
        return list;
}

