/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Richard Hult <rhult@codefactory.se>
 * Copyright (C) 2001 Mikael Hallendal <micke@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <glib.h>
#include <bonobo/bonobo-exception.h>
#include <liboaf/liboaf.h>
#include "type-utils.h"
#include "corba-utils.h"


GM_Dependency *
corba_util_dependency_duplicate (const GM_Dependency *dependency)
{
	GM_Dependency *copy;

	g_return_val_if_fail (dependency != NULL, NULL);

	copy                = GNOME_MrProject_Dependency__alloc ();
	copy->taskId        = dependency->taskId;
	copy->predecessorId = dependency->predecessorId;
	copy->depId         = dependency->depId;
	copy->type          = dependency->type;
	copy->empty         = CORBA_string_dup ("");

	return copy;
}

void
corba_util_task_copy (GM_Task *dst, const GM_Task *src)
{
	g_return_if_fail (dst != NULL);
	g_return_if_fail (src != NULL);

	g_assert (src->name);

	dst->name            = CORBA_string_dup (src->name);
	dst->taskId          = src->taskId;
	dst->type            = src->type;
	dst->parentId        = src->parentId;
	dst->start           = src->start;
	dst->end             = src->end;
	dst->percentComplete = src->percentComplete;
}

GM_Task *
corba_util_task_duplicate (const GM_Task *task)
{
	GM_Task *copy;

	copy = GNOME_MrProject_Task__alloc ();
	corba_util_task_copy (copy, task);
	
	return copy;
}

void
corba_util_allocation_copy (GM_Allocation       *dst,
			    const GM_Allocation *src)
{
	g_return_if_fail (dst != NULL);
	g_return_if_fail (src != NULL);

	dst->taskId     = src->taskId;
	dst->resourceId = src->resourceId;
	dst->units      = src->units;
}

GM_Allocation *
corba_util_allocation_duplicate (const GM_Allocation *allocation)
{
	GM_Allocation *copy;

	copy = GNOME_MrProject_Allocation__alloc ();
	corba_util_allocation_copy (copy, allocation);
	
	return copy;
}

TaskChangeMask
corba_util_task_update (GM_Task        *original,
			const GM_Task  *updated,
			TaskChangeMask  mask)
{
	TaskChangeMask changed;

	g_return_val_if_fail (original != NULL, TASK_CHANGE_NONE);
	g_return_val_if_fail (updated != NULL, TASK_CHANGE_NONE);

	g_assert (original->name);
	g_assert (updated->name);
	
	changed = TASK_CHANGE_NONE;
	
	if (original->taskId != updated->taskId) {
		g_warning ("ID differs: %d, %d.", 
			   (int) original->taskId, 
			   (int) updated->taskId);
		return TASK_CHANGE_NONE;
	}

	if (mask & TASK_CHANGE_PARENT) {
		if (original->parentId != updated->parentId) {
			original->parentId = updated->parentId;
			changed |= TASK_CHANGE_PARENT;
		}
	}
	
	if (mask & TASK_CHANGE_NAME) {
		if (strcmp (original->name, updated->name)) {
			CORBA_free (original->name);
			original->name = CORBA_string_dup (updated->name);
			changed |= TASK_CHANGE_NAME;
		}
	}
	
	if (mask & TASK_CHANGE_START) {
		if (original->start != updated->start) {
			original->start = updated->start;
			changed |= TASK_CHANGE_START;
		}
	}

	if (mask & TASK_CHANGE_END) {
		if (original->end != updated->end) {
			original->end = updated->end;
			changed |= TASK_CHANGE_END;
		}
	}

	if (mask & TASK_CHANGE_TYPE) {
		if (original->type != updated->type) {
			original->type = updated->type;
			changed |= TASK_CHANGE_TYPE;
		}
	}
	
	if (mask & TASK_CHANGE_PERCENT_COMPLETE) {
		if (original->percentComplete != updated->percentComplete) {
			original->percentComplete = updated->percentComplete;
			changed |= TASK_CHANGE_PERCENT_COMPLETE;
		}
	}
	
	return changed;
}

GM_Resource *
corba_util_resource_duplicate (const GM_Resource   *resource)
{
	GM_Resource   *copy;

	g_return_val_if_fail (resource != NULL, NULL);
	
	copy = GNOME_MrProject_Resource__alloc ();
	corba_util_resource_copy (copy, resource);
	
	return copy;
}

void
corba_util_resource_copy (GM_Resource         *copy,
			  const GM_Resource   *original)
{
	g_return_if_fail (copy != NULL);
	g_return_if_fail (original != NULL);

	g_assert (original->name);
	
	copy->resourceId  = original->resourceId;
	copy->name        = CORBA_string_dup (original->name);
	copy->groupId     = original->groupId;
	copy->type        = original->type;
	copy->units       = original->units;
	copy->email       = CORBA_string_dup (original->email);
	copy->stdRate     = original->stdRate;
	copy->ovtRate     = original->ovtRate;
}

gboolean
corba_util_resource_update (GM_Resource         *original,
			    const GM_Resource   *updated)
{
	gboolean   changed = FALSE;

	g_return_val_if_fail (original != NULL, FALSE);
	g_return_val_if_fail (updated != NULL, FALSE);

	g_assert (original->name);
	g_assert (updated->name);
	
	if (original->resourceId != updated->resourceId) {
		g_warning ("Trying to update resource failed");
		return changed;
	}

	if (strcmp (original->name, updated->name)) {
		CORBA_free (original->name);
		original->name = CORBA_string_dup (updated->name);
		changed = TRUE;
	}
	
	if (original->groupId != updated->groupId) {
		original->groupId = updated->groupId;
		changed = TRUE;
	} 

	if (original->type != updated->type) {
		original->type = updated->type;
		changed = TRUE;
	}

	if (original->units != updated->units) {
		original->units = updated->units;
		changed = TRUE;
	}

	if (strcmp (original->email, updated->email)) {
		CORBA_free (original->email);
		original->email = CORBA_string_dup (updated->email);
		changed = TRUE;
	}

	if (original->stdRate != updated->stdRate) {
		original->stdRate = updated->stdRate;
		changed = TRUE;
	}

	if (original->ovtRate != updated->ovtRate) {
		original->ovtRate = updated->ovtRate;
		changed = TRUE;
	}

	return changed;
}

GM_ResourceGroup *
corba_util_resource_group_duplicate (const GM_ResourceGroup *group)
{
	GM_ResourceGroup   *copy;
	
	g_return_val_if_fail (group != NULL, NULL);

	copy = GNOME_MrProject_ResourceGroup__alloc ();
	corba_util_resource_group_copy (copy, group);
	
	return copy;
}

void
corba_util_resource_group_copy (GM_ResourceGroup  *copy,
				const GM_ResourceGroup  *original)
{
	g_return_if_fail (copy != NULL);
	g_return_if_fail (original != NULL);
	
	copy->groupId     = original->groupId;
	copy->name        = CORBA_string_dup (original->name);
 	copy->adminName   = CORBA_string_dup (original->adminName);
 	copy->adminPhone  = CORBA_string_dup (original->adminPhone);
 	copy->adminEmail  = CORBA_string_dup (original->adminEmail);
}

gboolean
corba_util_resource_group_update (GM_ResourceGroup         *original,
				  const GM_ResourceGroup   *updated)
{
	gboolean   changed = FALSE;

	g_return_val_if_fail (original != NULL, FALSE);
	g_return_val_if_fail (updated != NULL, FALSE);

	g_assert (original->name);
	g_assert (original->adminName);
	g_assert (original->adminPhone);
	g_assert (original->adminEmail);
	g_assert (updated->name);
	g_assert (updated->adminName);
	g_assert (updated->adminPhone);
	g_assert (updated->adminEmail);
	
	if (original->groupId != updated->groupId) {
		g_warning ("Trying to update resource group failed");
		return changed;
	}

	if (strcmp (original->name, updated->name)) {
		CORBA_free (original->name);
		original->name = CORBA_string_dup (updated->name);
		changed = TRUE;
	}

	if (strcmp (original->adminName, updated->adminName)) {
		CORBA_free (original->adminName);
		original->adminName = CORBA_string_dup (updated->adminName);
		changed = TRUE;
	}

	if (strcmp (original->adminPhone, updated->adminPhone)) {
		CORBA_free (original->adminPhone);
		original->adminPhone = CORBA_string_dup (updated->adminPhone);
		changed = TRUE;
	}

	if (strcmp (original->adminEmail, updated->adminEmail)) {
		CORBA_free (original->adminEmail);
		original->adminEmail = CORBA_string_dup (updated->adminEmail);
		changed = TRUE;
	}
	
	return changed;
}

/* Returns a CORBA sequence of ids from a GSList,
 * with integers stored in the list element pointers.
 */
GM_IdSeq *
corba_util_id_seq_from_list (GSList *list)
{
	GM_IdSeq *ids;
	guint     len, i;

	if (!list) {
		g_message ("Id list is empty.");
	}
	
	len = g_slist_length (list);

	ids = GNOME_MrProject_IdSeq__alloc ();
	ids->_buffer = CORBA_sequence_GNOME_MrProject_Id_allocbuf (len);
	ids->_length = len;
	ids->_maximum = len;
	CORBA_sequence_set_release (ids, CORBA_TRUE);

	for (i = 0; i < len; i++) {
		ids->_buffer[i] = (GNOME_MrProject_Id) GPOINTER_TO_INT (list->data);
		/*g_print ("seq_from_list: %d\n", GPOINTER_TO_INT (list->data));*/
		list = list->next;
	}

	return ids;
}

GSList *
corba_util_id_seq_to_list (const GM_IdSeq *id_seq)
{
	GSList    *list;
	guint      i;
	GM_Id      id;
	
	list = NULL;
	
	for (i = 0; i < id_seq->_length; ++i) {
		id = id_seq->_buffer[i];
		list = g_slist_prepend (list, GINT_TO_POINTER (id));
		/*g_print ("seq_to_list: %d\n", id);*/
	}

	return g_slist_reverse (list);
}

/* HACK ALERT: Returns a CORBA string of :-delimited ids from
 * an GSList with integers stored in the list element pointers.
 */
CORBA_char *
corba_util_id_string_from_list (GSList *list)
{
	GSList     *l;
	GString    *str;
	gboolean    first;
	gchar      *tmp;
	CORBA_char *ret;

	if (!list) {
		g_message ("Id list is empty.");
	}

	first = TRUE;
	str = g_string_new ("");
	for (l = list; l; l = l->next) {

		if (first) {
			first = FALSE;
		} else {
			g_string_append_c (str, ':');
		}
		
		tmp = g_strdup_printf ("%d", GPOINTER_TO_INT (l->data));
		g_string_append (str, tmp);
		g_free (tmp);
	}

	ret = CORBA_string_dup (str->str);
	g_string_free (str, TRUE);

	return ret;
}

/* HACK ALERT: Returns a GSList of ids from a string with :-delimited
 * ids.
  */
GSList *
corba_util_id_string_to_list (CORBA_char *ids)
{
	gchar  **strv;
	GSList  *list;
	guint    i;
	GM_Id    id;

	strv = g_strsplit (ids, ":", 0);

	list = NULL;
	for (i = 0; strv[i]; i++) {
		id = (int) strtol (strv[i], NULL, 10);
		list = g_slist_prepend (list, GINT_TO_POINTER (id));
	}

	g_strfreev (strv);
	
	return g_slist_reverse (list);
}

/* Returns a CORBA sequence of dependencies from a GSList.
 */
GM_DependencySeq *
corba_util_dependency_seq_from_list (GSList *list)
{
	GM_DependencySeq *dependencies;
	GM_Dependency    *dependency;
	guint             len, i;

	if (!list) {
		g_message ("Dependency list is empty.");
	}

	len = g_slist_length (list);
	
	dependencies = GNOME_MrProject_DependencySeq__alloc ();
	dependencies->_buffer = CORBA_sequence_GNOME_MrProject_Dependency_allocbuf (len);
	dependencies->_length = len;
	dependencies->_maximum = len;
	CORBA_sequence_set_release (dependencies, CORBA_TRUE);

	for (i = 0; i < len; i++, list = list->next) {
		dependency = list->data;

		/*g_print ("adding dep to seq: %d %d %d %d\n",
			 dependency->taskId,
			 dependency->predecessorId,
			 dependency->depId,
			 dependency->type);*/
		
		dependencies->_buffer[i].taskId = dependency->taskId;
		dependencies->_buffer[i].predecessorId = dependency->predecessorId;
		dependencies->_buffer[i].depId = dependency->depId;
		dependencies->_buffer[i].type = dependency->type;
		dependencies->_buffer[i].empty = CORBA_string_dup ("");
	}

	return dependencies;
}

/* Returns a CORBA sequence of tasks from a GSList.
 */
GM_TaskSeq *
corba_util_task_seq_from_list (GSList *list)
{
	GM_TaskSeq *tasks;
	GM_Task    *task;
	guint       len, i;

	len = g_slist_length (list);
	
	tasks           = GNOME_MrProject_TaskSeq__alloc ();
	tasks->_buffer  = CORBA_sequence_GNOME_MrProject_Task_allocbuf (len);
	tasks->_length  = len;
	tasks->_maximum = len;
	CORBA_sequence_set_release (tasks, CORBA_TRUE);

	for (i = 0; i < len; i++, list = list->next) {
		task = list->data;
		corba_util_task_copy (&tasks->_buffer[i], task);
	}

	return tasks;
}

/* Returns a CORBA sequence of resources from a GSList.
 */
GM_ResourceSeq *
corba_util_resource_seq_from_list (GSList *list)
{
	GM_ResourceSeq   *resources;
	GM_Resource      *resource;
	guint                          len, i;

	len = g_slist_length (list);
	
	resources           = GNOME_MrProject_ResourceSeq__alloc ();
	resources->_buffer  = CORBA_sequence_GNOME_MrProject_Resource_allocbuf (len);
	resources->_length  = len;
	resources->_maximum = len;
	CORBA_sequence_set_release (resources, CORBA_TRUE);
	
	for (i = 0; i < len; i++, list = list->next) {
		resource = list->data;
		corba_util_resource_copy (&resources->_buffer[i], resource);
	}

	return resources;
}

/* Returns a CORBA sequence of resource groups from a GSList.
 */
GM_ResourceGroupSeq *
corba_util_resource_group_seq_from_list (GSList *list)
{
	GM_ResourceGroupSeq   *groups;
	GM_ResourceGroup      *group;
	guint                               len, i;

	len = g_slist_length (list);
	
	groups           = GNOME_MrProject_ResourceGroupSeq__alloc ();
	groups->_buffer  = CORBA_sequence_GNOME_MrProject_ResourceGroup_allocbuf (len);
	groups->_length  = len;
	groups->_maximum = len;
	CORBA_sequence_set_release (groups, CORBA_TRUE);

	for (i = 0; i < len; i++, list = list->next) {
		group = list->data;
		groups->_buffer[i] = 
			(*(corba_util_resource_group_duplicate(group)));
	}

	return groups;
}

GSList *
corba_util_task_seq_to_list (GM_TaskSeq *tasks)
{
	GSList    *list;
	GM_Task   *task;
	guint      i;
	
	list = NULL;
	
	for (i = 0; i < tasks->_length; ++i) {
		task = corba_util_task_duplicate ((GM_Task *) &tasks->_buffer[i]);
		list = g_slist_prepend (list, task);
	}

	return list;
}

GSList *
corba_util_resource_seq_to_list (GM_ResourceSeq *resources)
{
	GSList        *list;
	GM_Resource   *resource;
	guint          i;
	
	list = NULL;
	
	for (i = 0; i < resources->_length; ++i) {
		resource = corba_util_resource_duplicate ((GM_Resource *) &resources->_buffer[i]);
		list = g_slist_prepend (list, resource);
	}

	return list;
}

GSList *
corba_util_resource_group_seq_to_list (GM_ResourceGroupSeq *groups)
{
	GSList             *list;
	GM_ResourceGroup   *group;
	guint               i;
	
	list = NULL;
	
	for (i = 0; i < groups->_length; ++i) {
		group = corba_util_resource_group_duplicate ((GM_ResourceGroup *) &groups->_buffer[i]);
		list = g_slist_prepend (list, group);
	}

	return list;
}

GSList *
corba_util_allocation_seq_to_list (GM_AllocationSeq *allocations)
{
	GSList          *list;
	GM_Allocation   *allocation;
	guint            i;
	
	list = NULL;
	
	for (i = 0; i < allocations->_length; ++i) {
		allocation = corba_util_allocation_duplicate ((GM_Allocation *) &allocations->_buffer[i]);
		list = g_slist_prepend (list, allocation);
	}

	return list;
}

/* Returns a CORBA sequence of allocations from a GSList.
 */
GM_AllocationSeq *
corba_util_allocation_seq_from_list (GSList *list)
{
	GM_AllocationSeq   *allocations;
	GM_Allocation      *allocation;
	guint               len, i;

	len = g_slist_length (list);
	
	allocations      = GNOME_MrProject_AllocationSeq__alloc ();
	allocations->_buffer  = CORBA_sequence_GNOME_MrProject_Allocation_allocbuf (len);
	allocations->_length  = len;
	allocations->_maximum = len;
	CORBA_sequence_set_release (allocations, CORBA_TRUE);

	for (i = 0; i < len; i++, list = list->next) {
		allocation = list->data;
		corba_util_allocation_copy (&allocations->_buffer[i], allocation);
	}

	return allocations;
}

GSList *
corba_util_dependency_seq_to_list (GM_DependencySeq *dependencys)
{
	GSList        *list;
	GM_Dependency *dependency;
	guint          len, i;

	len = dependencys->_length;

	list = NULL;
	for (i = 0; i < len; i++) {
		dependency = corba_util_dependency_duplicate ((GM_Dependency *) &dependencys->_buffer[i]);
		list = g_slist_prepend (list, dependency);
	}

	return list;
}

void
g_slist_CORBA_free (GSList *list)
{
	GSList *l;

	for (l = list; l; l = l->next) {
		CORBA_free (l->data);
	}
	
	g_slist_free (list);
}

static gchar *
exception_to_string (CORBA_Environment *ev, GLogLevelFlags *level)
{
	GM_GeneralError *error_data;
	gchar           *msg;

	error_data = NULL;
	msg = NULL;

	if (!BONOBO_EX (ev)) {
		return g_strdup ("");
	}

	if (level) {
		*level = G_LOG_LEVEL_CRITICAL;
	}

	if (ev->_major == CORBA_USER_EXCEPTION) {
		if (!strncmp ((ev)->_repo_id, "IDL:GNOME/MrProject/",
			      sizeof ("IDL:GNOME/MrProject/") - 1)) {
			/* Ok we can handle this nicely. */
			error_data = CORBA_exception_value (ev);
			if (error_data) {
				msg = g_strdup (error_data->msg.reason);

				if (level) {
					switch (error_data->msg.type) {
					case GNOME_MrProject_ERROR_INFO:
						*level = G_LOG_LEVEL_INFO;
						break;
					case GNOME_MrProject_ERROR_WARNING:
						*level = G_LOG_LEVEL_WARNING;
						break;
					case GNOME_MrProject_ERROR_CRITICAL:
						*level = G_LOG_LEVEL_CRITICAL;
					break;
					};
				}
			}
		}
	}

	if (!msg) {
		msg = bonobo_exception_get_text (ev);
	}
	
	return msg;
}
	
void	
g_log_exception (CORBA_Environment *ev, gchar *msg)
{
	GLogLevelFlags  level;
	gchar          *exception_string;

	exception_string = exception_to_string (ev, &level);

	if (msg && strlen (msg) > 0) {
		g_log (G_LOG_DOMAIN, level, "%s\n\n%s", msg, exception_string);
	} else {
		g_log (G_LOG_DOMAIN, level, exception_string);
	}
}

gboolean
corba_util_evolution_composer_exist (void)
{
	OAF_ServerInfoList *oaf_result;
	CORBA_Environment   ev;
	gchar              *query;
	gboolean            found = FALSE;
	
	query = "repo_ids.has ('IDL:GNOME/Evolution:Composer:1.0')";

	CORBA_exception_init (&ev);

	oaf_result = oaf_query (query, NULL, &ev);
	if (!BONOBO_EX (&ev) && oaf_result != NULL && oaf_result->_length > 0) {
		found = TRUE;
	}

	CORBA_free (oaf_result);
	CORBA_exception_free (&ev);

	return found;
}
