/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2000-2001 CodeFactory AB
 * Copyright (C) 2000-2001 Richard Hult <rhult@codefactory.se>
 * Copyright (C) 2001      Mikael Hallendal <micke@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <glib.h>
#include <gtk/gtkobject.h>
#include <bonobo/bonobo-xobject.h>
#include <bonobo/bonobo-property-bag.h>
#include <bonobo/bonobo-exception.h>
#include <gnome-xml/parser.h>
#include <gnome-xml/parserInternals.h>
#include <gnome-xml/xmlmemory.h>

#include "util/time-utils.h"
#include "task-manager.h"
#include "resource-manager.h"
#include "allocation-manager.h"
#include "backends/file-task-manager.h"
#include "backends/file-resource-manager.h"
#include "backends/file-allocation-manager.h"
#include "project.h"

#include "util/debug.h"

#define PARENT_TYPE BONOBO_X_OBJECT_TYPE
static GtkObjectClass *parent_class;

/* Bonobo properties. */
enum {
	PROP_START,
	PROP_FINISH,
	PROP_NAME,
	PROP_MANAGER,
	PROP_COMPANY,

	/* To make catching interface changes easier. */
	PROP_VERSION
};

static void project_init       (Project      *project);
static void project_class_init (ProjectClass *klass);


/* Private members. */
struct _ProjectPriv {
	BonoboPropertyBag   *property_bag; /* aggregated */
	GList               *shells;
	gchar               *uri;

	/* Properties. */
	gchar 		    *name, *company, *manager;
	time_t		     start, finish;
};

typedef struct {
	GNOME_MrProject_Shell         shell_co;
	Bonobo_EventSource_ListenerId listener_id;
} ShellInfo;

static void
impl_Project_addShell (PortableServer_Servant       servant, 
		       const GNOME_MrProject_Shell  shell, 
		       CORBA_Environment           *ev)
{
	Project                       *project;
	ProjectPriv                   *priv;
	Bonobo_Listener                listener_co;
	Bonobo_EventSource_ListenerId  id;
	ShellInfo                     *info;

	project = PROJECT (bonobo_x_object (servant));
	priv = project->priv;

	listener_co = Bonobo_Unknown_queryInterface (shell,
						     "IDL:Bonobo/Listener:1.0",
						     ev);
	if (listener_co == CORBA_OBJECT_NIL || BONOBO_EX (ev)) {
		g_warning ("Couldn't get shell listener.");
		return;
	}

	id = Bonobo_EventSource_addListener (BONOBO_OBJREF (project->event_source),
					     listener_co,
					     CORBA_OBJECT_NIL);

	bonobo_object_release_unref (listener_co, CORBA_OBJECT_NIL);

	info = g_new0 (ShellInfo, 1);
	info->shell_co = CORBA_Object_duplicate (shell, ev);
	info->listener_id = id;
	priv->shells = g_list_prepend (priv->shells, info);
}

static void
impl_Project_removeShell (PortableServer_Servant       servant, 
			  const GNOME_MrProject_Shell  shell, 
			  CORBA_Environment           *ev)
{
	Project            *project;
	ProjectPriv        *priv;
	gboolean            found;
	GList              *list;
	ShellInfo          *info;

	d(puts (__FUNCTION__));
	
	project = PROJECT (bonobo_x_object (servant));
	priv = project->priv;

	found = FALSE;
	info = NULL;
	for (list = priv->shells; list; list = list->next) {
		info = list->data;
		if (CORBA_Object_is_equivalent (info->shell_co, shell, ev)) {
			priv->shells = g_list_remove (priv->shells, info);
			found = TRUE;
			break;
		}
	}
	
	if (!found) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_GNOME_MrProject_Project_NoSuchShell,
				     NULL);
		return;
	}

	Bonobo_EventSource_removeListener (BONOBO_OBJREF (project->event_source),
					   info->listener_id,
					   ev);
	info->shell_co = NULL;
	info->listener_id = 0;
	g_free (info);
}

static GNOME_MrProject_ProjectState 
impl_Project_getState (PortableServer_Servant  servant,
		       CORBA_Environment      *ev)
{
	Project                      *project;
	ProjectPriv                  *priv;
	GNOME_MrProject_ProjectState  a_state, r_state, t_state;

	project  = PROJECT (bonobo_x_object (servant));
	priv     = project->priv;

	a_state = r_state = t_state = GNOME_MrProject_PROJECT_STATE_EMPTY;

	gtk_object_get (GTK_OBJECT (project->allocation_manager),
			"state",
			&a_state,
			NULL);

	gtk_object_get (GTK_OBJECT (project->resource_manager),
			"state",
			&r_state,
			NULL);

	gtk_object_get (GTK_OBJECT (project->task_manager),
			"state",
			&t_state,
			NULL);
	
	if (a_state == GNOME_MrProject_PROJECT_STATE_EMPTY &&
	    r_state == GNOME_MrProject_PROJECT_STATE_EMPTY &&
	    t_state == GNOME_MrProject_PROJECT_STATE_EMPTY) {
		return GNOME_MrProject_PROJECT_STATE_EMPTY;
	}

	if (a_state == GNOME_MrProject_PROJECT_STATE_DIRTY ||
	    r_state == GNOME_MrProject_PROJECT_STATE_DIRTY ||
	    t_state == GNOME_MrProject_PROJECT_STATE_DIRTY) {
		return GNOME_MrProject_PROJECT_STATE_DIRTY;
	}

	return GNOME_MrProject_PROJECT_STATE_SAVED;
}

static void
impl_Project_load (PortableServer_Servant  servant,
		   const CORBA_char       *uri,
		   CORBA_Environment      *ev)
{
	Project	     *project;
	ProjectClass *klass;
	CORBA_any    *any;

	project = PROJECT (bonobo_x_object (servant));
	klass   = PROJECT_CLASS (GTK_OBJECT (project)->klass);

	any = CORBA_any__alloc ();
	any->_type = TC_null;
	any->_value = CORBA_OBJECT_NIL;
	CORBA_any_set_release (any, FALSE);
	
	bonobo_event_source_notify_listeners (project->event_source,
					      "GNOME/MrProject:project:load_begin",
					      any,
					      ev);
	
	klass->load (project, uri, ev);
	if (!BONOBO_EX (ev)) {
		project->priv->uri = g_strdup (uri);
	}

	bonobo_event_source_notify_listeners (project->event_source,
					      "GNOME/MrProject:project:load_end",
					      any,
					      ev);

	CORBA_free (any);
}

static void
impl_Project_save (PortableServer_Servant  servant,
		   const CORBA_char       *uri,
		   CORBA_Environment      *ev)
{
	Project      *project;
	ProjectClass *klass;
	
	project  = PROJECT (bonobo_x_object (servant));

	klass = PROJECT_CLASS (GTK_OBJECT (project)->klass);

	if ((uri == CORBA_OBJECT_NIL || strlen (uri) == 0)) {
		if (!project->priv->uri) { /* No new URI and no old one. */
			/* FIXME: raise an exception if we
			 * don't have a valid location.
			 */
			return;
		}
	} else if (!project->priv->uri) {
		/* New URI, but no old one -> use the new one. */
		project->priv->uri = g_strdup (uri);
	} else if (strcmp (uri, project->priv->uri)) {
		/* New URI and old one that differs ->
		 * free old one and use new one.
		 */
		g_free (project->priv->uri);
		project->priv->uri = g_strdup (uri);
	}
	
	klass->save (project, uri, ev);
}

static CORBA_char *
impl_Project_getURI (PortableServer_Servant  servant,
		     CORBA_Environment      *ev)
{
	Project *project;
	
	project = PROJECT (bonobo_x_object (servant));

	/* URI should never ever be NULL. Maybe "", but never NULL. */
	g_assert (project->priv->uri != NULL);
	
	return CORBA_string_dup (project->priv->uri);
}

static void
impl_Project_setURI (PortableServer_Servant  servant,
		     const CORBA_char       *uri,
		     CORBA_Environment      *ev)
{
	Project     *project;
	ProjectPriv *priv;
	
	project = PROJECT (bonobo_x_object (servant));
	priv    = project->priv;

	if (priv->uri) {
		g_free (priv->uri);
	}
		
	priv->uri = g_strdup (uri);
}

static void
project_destroy (GtkObject *object)
{
	Project     *project;
	ProjectPriv *priv;

	d(puts (__FUNCTION__));
	
	project = PROJECT (object);
	priv    = project->priv;

	g_free (priv->uri);
	g_free (priv->name);
	g_free (priv->company);
	g_free (priv->manager);
	
	priv->uri = NULL;
	priv->name = NULL;
	priv->company = NULL;
	priv->manager = NULL;
	
	priv->property_bag = NULL;
	
	if (priv->shells != NULL) {
		g_warning ("Shells still added to project under destruction!");
	}
	
	g_free (project->priv);
	project->priv = NULL;

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
project_class_init (ProjectClass *klass)
{
	POA_GNOME_MrProject_Project__epv *epv = &klass->epv;
	GtkObjectClass *object_class = (GtkObjectClass *) klass;

	parent_class = gtk_type_class (PARENT_TYPE);

	object_class->destroy = project_destroy;

	epv->addShell         = impl_Project_addShell;
	epv->removeShell      = impl_Project_removeShell;
	epv->getState         = impl_Project_getState;
	epv->load             = impl_Project_load;
	epv->save             = impl_Project_save;
	epv->getURI           = impl_Project_getURI;
	epv->setURI           = impl_Project_setURI;
}

BONOBO_X_TYPE_FUNC_FULL (Project, 
			 GNOME_MrProject_Project,
			 PARENT_TYPE,
			 project);

static void
set_prop (BonoboPropertyBag *bag,
	  const BonoboArg   *arg,
	  guint              arg_id,
	  CORBA_Environment *ev,
	  gpointer           user_data)
{
	Project     *project;
	ProjectPriv *priv;
	gchar   *str;
	time_t   t;

	g_return_if_fail (user_data != NULL);
	g_return_if_fail (IS_PROJECT (user_data));

	project = PROJECT (user_data);
	priv    = project->priv;
	
	switch (arg_id) {
	case PROP_NAME:
		str = BONOBO_ARG_GET_STRING (arg);
		g_free (priv->name);
		priv->name = g_strdup (str);
		break;

	case PROP_COMPANY:
		str = BONOBO_ARG_GET_STRING (arg);
		g_free (priv->company);
		priv->company = g_strdup (str);
		break;

	case PROP_MANAGER:
		str = BONOBO_ARG_GET_STRING (arg);
		g_free (priv->manager);
		priv->manager = g_strdup (str);
		break;

	case PROP_START:
		t = BONOBO_ARG_GET_INT (arg);
		priv->start = t;
		break;

	case PROP_FINISH:
		t = BONOBO_ARG_GET_INT (arg);
		priv->finish = t;
		break;
		
	default:
		g_print ("Project FIXME: set prop %d unimplemented.\n", arg_id);
		break;
	}
}

static void
get_prop (BonoboPropertyBag *bag,
	  BonoboArg         *arg,
	  guint              arg_id,
	  CORBA_Environment *ev,
	  gpointer           user_data)
{
	Project     *project;
	ProjectPriv *priv;
	gchar       *str;

	g_return_if_fail (user_data != NULL);
	g_return_if_fail (IS_PROJECT (user_data));

	project = PROJECT (user_data);
	priv    = project->priv;

	switch (arg_id) {
	case PROP_START:
		BONOBO_ARG_SET_GENERAL (arg,
					priv->start,
					TC_CORBA_long,
					GNOME_MrProject_Time,
					NULL);
		break;

	case PROP_FINISH:
		BONOBO_ARG_SET_GENERAL (arg,
					priv->finish,
					TC_CORBA_long,
					GNOME_MrProject_Time,
					NULL);
		break;

	case PROP_NAME:
		str = priv->name;
		BONOBO_ARG_SET_STRING (arg, str); 
		break;

	case PROP_COMPANY:
		str = priv->company;
		BONOBO_ARG_SET_STRING (arg, str); 
		break;

	case PROP_MANAGER:
		str = priv->manager;
		BONOBO_ARG_SET_STRING (arg, str); 
		break;

	case PROP_VERSION:
		BONOBO_ARG_SET_STRING (arg, VERSION);
		break;

	default:
		break;
	}
}

static void
project_init (Project *project)
{
	ProjectPriv       *priv;
	BonoboPropertyBag *property_bag;
	BonoboEventSource *event_source;
	
	priv = g_new0 (ProjectPriv, 1);
	project->priv = priv;

	priv->uri     = g_strdup ("");
	priv->name    = g_strdup ("");
	priv->company = g_strdup ("");
	priv->manager = g_strdup ("");

	event_source = bonobo_event_source_new ();
	project->event_source = event_source;

	/* FIXME: any trouble adding here and then having the property bag
	 * add it again?
	 */
	/*bonobo_object_add_interface (BONOBO_OBJECT (project),
	  BONOBO_OBJECT (event_source));*/

	property_bag = bonobo_property_bag_new_full (get_prop,
						     set_prop,
						     event_source,
						     project);
	
	priv->property_bag = property_bag;

	bonobo_object_add_interface (BONOBO_OBJECT (project),
				     BONOBO_OBJECT (property_bag));

	bonobo_property_bag_add (property_bag,
				 "Start",
				 PROP_START,
				 TC_CORBA_long,
				 NULL,
				 "The project start date",
				 BONOBO_PROPERTY_READABLE | BONOBO_PROPERTY_WRITEABLE);
	bonobo_property_bag_add (property_bag,
				 "Finish",
				 PROP_FINISH,
				 TC_CORBA_long,
				 NULL,
				 "The project finish date",
				 BONOBO_PROPERTY_READABLE | BONOBO_PROPERTY_WRITEABLE);
	bonobo_property_bag_add (property_bag,
				 "Name",
				 PROP_NAME,
				 BONOBO_ARG_STRING,
				 NULL,
				 "The name of the project",
				 BONOBO_PROPERTY_READABLE | BONOBO_PROPERTY_WRITEABLE);
	bonobo_property_bag_add (property_bag,
				 "Company",
				 PROP_COMPANY,
				 BONOBO_ARG_STRING,
				 NULL,
				 "The company that drives the project",
				 BONOBO_PROPERTY_READABLE | BONOBO_PROPERTY_WRITEABLE);
	bonobo_property_bag_add (property_bag,
				 "Manager",
				 PROP_MANAGER,
				 BONOBO_ARG_STRING,
				 NULL,
				 "The project manager",
				 BONOBO_PROPERTY_READABLE | BONOBO_PROPERTY_WRITEABLE);
	bonobo_property_bag_add (property_bag,
				 "Version",
				 PROP_VERSION,
				 BONOBO_ARG_STRING,
				 NULL,
				 "Version of project engine",
				 BONOBO_PROPERTY_READABLE);
}
