/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Mikael Hallendal <micke@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Mikael Hallendal <micke@codefactory.se>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "libmrproject/GNOME_MrProject.h"
#include "util/id-map.h"
#include "util/type-utils.h"
#include "util/corba-utils.h"
#include "file-allocation-manager.h"

#define d(x)

static void file_allocation_manager_init       (FileAllocationManager      *m);
static void file_allocation_manager_class_init (FileAllocationManagerClass *k);

static void  fam_destroy    (GtkObject   *object);

static GM_Allocation *     fam_allocate   (AllocationManager     *am,
                                           GM_Id                  task_id,
                                           GM_Id                  resource_id,
                                           guint                  units);

static void                fam_deallocate (AllocationManager     *fam,
					   const GM_Allocation   *allocation);

static GSList *
fam_get_allocations_by_task  (AllocationManager   *am,
                              GM_Id                task_id);
static GSList *
fam_get_allocations_by_resource      (AllocationManager   *am,
				      GM_Id                resource_id);

static GSList *
fam_get_all_allocations      (AllocationManager    *am);

static GSList *  
fam_deallocate_all_resources  (AllocationManager   *am,
                               GM_Id                task_id);

static GSList * 
fam_deallocate_all_tasks      (AllocationManager   *am,
                               GM_Id                resource_id);

static void
fam_foreach_allocation        (AllocationManager   *am,
			       GFunc                func,
			       gpointer             user_data);

static GSList *
fam_get_list_node             (FileAllocationManager   *fam,
			       GM_Id                    task_id,
			       GM_Id                    resource_id);

struct _FileAllocationManagerPriv {

	/* This will not be very efficient with lots of allocations */
	GSList   *allocations;
};

GNOME_CLASS_BOILERPLATE (FileAllocationManager, file_allocation_manager,
                         AllocationManager, allocation_manager);

static void
file_allocation_manager_init (FileAllocationManager *fam)
{
        FileAllocationManagerPriv   *priv;
        
        priv               = g_new0 (FileAllocationManagerPriv, 1);
        priv->allocations  = NULL;
        fam->priv          = priv;
}

static void 
file_allocation_manager_class_init (FileAllocationManagerClass *klass)
{
        GtkObjectClass           *object_class;
	AllocationManagerClass   *amc;
	
        object_class = (GtkObjectClass *) klass;
	amc          = (AllocationManagerClass *) klass;

        /* CORBA methods. */
        amc->allocate                    = fam_allocate;
        amc->deallocate                  = fam_deallocate;
        amc->get_allocations_by_task     = fam_get_allocations_by_task;
        amc->get_allocations_by_resource = fam_get_allocations_by_resource;
        amc->get_all_allocations         = fam_get_all_allocations;
        amc->deallocate_all_resources    = fam_deallocate_all_resources;
        amc->deallocate_all_tasks        = fam_deallocate_all_tasks;

	/* C methods. */
	amc->foreach_allocation          = fam_foreach_allocation;
	 
        object_class->destroy            = fam_destroy;
}

static void
fam_destroy (GtkObject *object)
{
        FileAllocationManager       *fam;
        FileAllocationManagerPriv   *priv;
	GSList                      *node;
	
        g_return_if_fail (object != NULL);
        g_return_if_fail (IS_FILE_ALLOCATION_MANAGER (object));
        
        fam  = FILE_ALLOCATION_MANAGER (object);
        priv = fam->priv;

	if (priv->allocations) {
		for (node = priv->allocations; node; node = node->next) {
			CORBA_free (node->data);
		}
		
		g_slist_free (priv->allocations);
		priv->allocations = NULL;
        }
	
        GNOME_CALL_PARENT_HANDLER (GTK_OBJECT_CLASS, destroy, (object));
}

static GM_Allocation *
fam_allocate (AllocationManager   *am,
              GM_Id                task_id,
              GM_Id                resource_id,
              guint                units)
{
        FileAllocationManager       *fam;
        FileAllocationManagerPriv   *priv;
        GM_Allocation               *allocation;
        
        g_return_val_if_fail (am != NULL, NULL);
        g_return_val_if_fail (IS_FILE_ALLOCATION_MANAGER (am), NULL);
        
        fam  = FILE_ALLOCATION_MANAGER (am);
        priv = fam->priv;

	if (fam_get_list_node (fam, task_id, resource_id)) {
		d(g_print ("Already allocated\n"));
		return NULL;
	}

        allocation             = GNOME_MrProject_Allocation__alloc ();
        allocation->taskId     = task_id;
        allocation->resourceId = resource_id;

	/* Default to 100 units (units are in procents, so 100 units equals
	   one actual units) */
	allocation->units      = 100;
	
	priv->allocations = g_slist_prepend (priv->allocations, allocation);
        
	d(g_print ("Allocated, now %d allocations\n", 
		   g_slist_length (priv->allocations)));

        return allocation;
}

static void
fam_deallocate (AllocationManager     *am,
                const GM_Allocation   *allocation)
{
        FileAllocationManager       *fam;
        FileAllocationManagerPriv   *priv;
	GSList                      *node;
        GM_Allocation               *tmp_allocation;
        
        g_return_if_fail (am != NULL);
        g_return_if_fail (IS_FILE_ALLOCATION_MANAGER (am));
        
        fam  = FILE_ALLOCATION_MANAGER (am);
        priv = fam->priv;

	d(g_print ("Trying to remove [%d <-> %d]\n",
		   allocation->taskId, allocation->resourceId));

	node = fam_get_list_node (fam, 
				  allocation->taskId, allocation->resourceId);

	if (node) {
		tmp_allocation = (GM_Allocation *) node->data;
		
		priv->allocations = g_slist_remove_link (priv->allocations, 
							 node);

		d(g_print ("FileAllocationManager::deallocate: Removed [ %d <-> %d], length now %d\n",
			   tmp_allocation->taskId, tmp_allocation->resourceId,
			   g_slist_length (priv->allocations)));
		
		g_slist_free (node);

		CORBA_free (tmp_allocation);
	}
}


static GSList *
fam_get_allocations_by_task (AllocationManager    *am,
                             GM_Id                 task_id)
{
        FileAllocationManager       *fam;
        FileAllocationManagerPriv   *priv;
	GSList                      *node;
	GSList                      *list = NULL;
        GM_Allocation               *allocation;
        
        g_return_val_if_fail (am != NULL, NULL);
        g_return_val_if_fail (IS_FILE_ALLOCATION_MANAGER (am), NULL);
        
        fam  = FILE_ALLOCATION_MANAGER (am);
        priv = fam->priv;

	d(g_print ("Got %d allocations\n", 
		   g_slist_length (priv->allocations)));

	for (node = priv->allocations; node; node = node->next) {
		allocation = (GM_Allocation *) node->data;
		
		if (allocation->taskId == task_id) {
			list = g_slist_prepend (list, allocation);
		}
	}

	return list;
}

static GSList *
fam_get_allocations_by_resource (AllocationManager   *am,
				 GM_Id                resource_id)
{
        FileAllocationManager       *fam;
        FileAllocationManagerPriv   *priv;
	GSList                      *node;
        GSList                      *list = NULL;
        GM_Allocation               *allocation;
        
        g_return_val_if_fail (am != NULL, NULL);
        g_return_val_if_fail (IS_FILE_ALLOCATION_MANAGER (am), NULL);
        
        fam  = FILE_ALLOCATION_MANAGER (am);
        priv = fam->priv;

	for (node = priv->allocations; node; node = node->next) {
                allocation = (GM_Allocation *) node->data;
                
                if (allocation->resourceId == resource_id) {
                        list = g_slist_prepend (list, allocation);
                }
	}

        return list;
}

static GSList *
fam_get_all_allocations (AllocationManager *am)
{
        FileAllocationManager     *fam;
        FileAllocationManagerPriv *priv;
        
        g_return_val_if_fail (am != NULL, NULL);
        g_return_val_if_fail (IS_FILE_ALLOCATION_MANAGER (am), NULL);
        
        fam  = FILE_ALLOCATION_MANAGER (am);
        priv = fam->priv;
 
        return g_slist_copy (priv->allocations);
}

static GSList *  
fam_deallocate_all_resources (AllocationManager   *am,
                              GM_Id                task_id)
{ 
        FileAllocationManager       *fam;
        FileAllocationManagerPriv   *priv;
	GSList                      *node;
        GSList                      *list = NULL;
        GM_Allocation               *allocation;
        
        g_return_val_if_fail (am != NULL, NULL);
        g_return_val_if_fail (IS_FILE_ALLOCATION_MANAGER (am), NULL);
        
        fam  = FILE_ALLOCATION_MANAGER (am);
        priv = fam->priv;

	for (node = priv->allocations; node; node = node->next) {
                allocation = (GM_Allocation *) node->data;

		if (allocation->taskId == task_id) {
			
			priv->allocations = g_slist_remove (priv->allocations,
							    allocation);

			/* This list should be freed by caller */
                        list = g_slist_prepend (list, allocation);
		}
	}
 
	return list;
}

static GSList * 
fam_deallocate_all_tasks (AllocationManager   *am,
                          GM_Id                resource_id)
{
        FileAllocationManager       *fam;
        FileAllocationManagerPriv   *priv;
	GSList                      *node;
        GSList                      *list = NULL;
        GM_Allocation               *allocation;
        
        g_return_val_if_fail (am != NULL, NULL);
        g_return_val_if_fail (IS_FILE_ALLOCATION_MANAGER (am), NULL);
        
        fam  = FILE_ALLOCATION_MANAGER (am);
        priv = fam->priv;
 
	for (node = priv->allocations; node; node = node->next) {
                allocation = (GM_Allocation *) node->data;
                
                if (allocation->resourceId == resource_id) {
			priv->allocations = g_slist_remove (priv->allocations,
							    allocation);
			
			/* This list should be freed by caller */
                        list = g_slist_prepend (list, allocation);
                }
	}

        return list;
}

static void
fam_foreach_allocation        (AllocationManager    *am,
			       GFunc                 func,
			       gpointer              user_data)
{
	FileAllocationManager *fam;

	fam = FILE_ALLOCATION_MANAGER (am);
	
	g_slist_foreach (fam->priv->allocations, func, user_data);
}

static GSList *
fam_get_list_node             (FileAllocationManager   *fam,
			       GM_Id                    task_id,
			       GM_Id                    resource_id)
{
	FileAllocationManagerPriv   *priv;
	GSList                      *node;
	GM_Allocation               *allocation;
	
	priv = fam->priv;

	for (node = priv->allocations; node; node = node->next) {
		allocation = (GM_Allocation *) node->data;
		
		if (allocation->resourceId == resource_id &&
		    allocation->taskId     == task_id) {
			return node;
		}
	}

	return NULL;
}

AllocationManager *
file_allocation_manager_new (BonoboEventSource *event_source)
{
	AllocationManager *am;
	
	am = ALLOCATION_MANAGER (gtk_type_new (TYPE_FILE_ALLOCATION_MANAGER));

	allocation_manager_construct (am, event_source);
	
	return am;
}


                            

                                                        
                                                        

                             
                             


