/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2000-2001 CodeFactory AB
 * Copyright (C) 2000-2001 Richard Hult <rhult@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <bonobo.h>
#include <bonobo/bonobo-listener.h>
#include <bonobo/bonobo-property-bag.h>
#include <liboaf/liboaf.h>
#include "util/type-utils.h"
#include "util/corba-utils.h"
#include "client/widgets/mr-message-box.h"
#include "shell.h"
#include "project-window.h"

#define DEBUG 0
#include "util/debug.h"

#define PROJECT_OAFIID "OAFIID:GNOME_MrProject_Project"
#define PARENT_TYPE BONOBO_X_OBJECT_TYPE


static GtkObjectClass *parent_class;

static void     shell_init           (Shell      *shell);
static void     shell_class_init     (ShellClass *klass);
static void     activate_components  (Shell      *shell);
static void     terminate_components (Shell      *shell);
static gboolean check_version        (Shell      *shell);


/* A shell component and its control and widget. */
typedef struct {
	Bonobo_Control     control;
	GM_ShellComponent  component;
	GtkWidget         *widget;
} ControlInfo;

/* Private members. */
struct _ShellPriv {
	ProjectWindow     *project_window;

	GM_Project         project_co;       /* The Project object for this shell. */ 
	BonoboUIContainer *container;

	BonoboEventSource *event_source;
	BonoboListener    *listener;

	GSList            *components;
	
	GHashTable        *controls;       /* All the controls that the shell maintains, OAFIID as key. */
	ControlInfo       *active_control; /* The component that is currently activated with menus merged etc. */
};

static void
impl_addControl (PortableServer_Servant   servant,
		 const GM_ShellComponent  component,
		 const Bonobo_Control     control,
		 const CORBA_char         *id,
		 const CORBA_char        *title,
		 const CORBA_char        *iconFilename,
		 CORBA_Environment       *ev)
{
	Shell          *shell;
	ControlInfo  *info;
	gchar          *key;
	GdkPixbuf      *pixbuf;

	shell = SHELL (bonobo_x_object (servant));
	
	if (control == CORBA_OBJECT_NIL || component == CORBA_OBJECT_NIL || strlen (id) == 0) {
		g_warning ("Trying to add a NULL component.");
		return;
	}

	if (g_hash_table_lookup (shell->priv->controls, id)) {
		g_warning ("Trying to add the same component twice.");
		return;
	}

	info = g_new0 (ControlInfo, 1);

	key = g_strdup (id);
	info->control = CORBA_Object_duplicate (control, NULL);
	info->component = CORBA_Object_duplicate (component, NULL);

	g_hash_table_insert (shell->priv->controls, key, info);

	pixbuf = gdk_pixbuf_new_from_file (iconFilename);

	info->widget = bonobo_widget_new_control_from_objref (info->control,
							      BONOBO_OBJREF (shell->priv->container));
	
	project_window_add_control (shell->priv->project_window,
				    info->widget,
				    key,
				    pixbuf,
				    title);

	gdk_pixbuf_unref (pixbuf);
}

static void
impl_removeControl (PortableServer_Servant   servant,
		    const CORBA_char        *id,
		    CORBA_Environment       *ev)
{
	Shell         *shell;
	ControlInfo *info;
	gpointer       key, value;

	shell = SHELL (bonobo_x_object (servant));

	if (id == NULL || strlen (id) == 0) {
		g_warning ("Trying to remove a NULL component Id.");
		return;
	}

	if (!g_hash_table_lookup_extended (shell->priv->controls, id, &key, &value)) {
		g_warning ("Trying to remove a component that is not added.");
		return;
	}

	info = value;
	
	g_hash_table_remove (shell->priv->controls, id);
	g_free (key);

	GNOME_MrProject_ShellComponent_removeFromShell (info->component, ev);

	CORBA_Object_release (info->control, NULL);
	CORBA_Object_release (info->component, NULL);

	/* This is to unref the ref we got when we first activated. */
	Bonobo_Unknown_unref (info->component, NULL);

	gtk_object_destroy (GTK_OBJECT (info->widget));
	
	info->control = NULL;
	info->component = NULL;
	info->widget = NULL;
}

static GM_Project
impl_getProject (PortableServer_Servant  servant,
		 CORBA_Environment      *ev)
{
	Shell      *shell;
	ShellPriv  *priv;
	GM_Project  project_co;

	shell = SHELL (bonobo_x_object (servant));
	priv = shell->priv;

	if (CORBA_Object_is_nil (priv->project_co, NULL)) {
		return CORBA_OBJECT_NIL;
	}

	project_co = CORBA_Object_duplicate (priv->project_co, ev);
	return project_co;
}

static Bonobo_UIContainer
impl_getUIContainer (PortableServer_Servant servant, CORBA_Environment *ev)
{
	Shell *shell;

	shell = SHELL (bonobo_x_object (servant));

	if (!shell->priv->container) {
		return CORBA_OBJECT_NIL;
	}

	return bonobo_object_dup_ref (BONOBO_OBJREF (shell->priv->container), NULL);
}

static Bonobo_EventSource
impl_getProxyEventSource (PortableServer_Servant servant, CORBA_Environment *ev)
{
	Shell *shell;

	shell = SHELL (bonobo_x_object (servant));

	if (!shell->priv->event_source) {
		return CORBA_OBJECT_NIL;
	}

	return bonobo_object_dup_ref (BONOBO_OBJREF (shell->priv->event_source), NULL);
}

static void
shell_destroy (GtkObject *object)
{
	Shell     *shell;
	ShellPriv *priv;

	d(puts (__FUNCTION__));
	
	shell = SHELL (object);
	priv = shell->priv;

	terminate_components (shell);

	if (priv->project_co != CORBA_OBJECT_NIL) {
		shell_remove_from_project (shell);
	}

	bonobo_object_unref (BONOBO_OBJECT (shell->priv->event_source));

	/* FIXME: free the keys/values. */
	g_hash_table_destroy (shell->priv->controls);
	
	g_free (shell->priv);
	shell->priv = NULL;

	(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
shell_class_init (ShellClass *klass)
{
	POA_GNOME_MrProject_Shell__epv *epv = &klass->epv;
	GtkObjectClass *object_class = (GtkObjectClass *) klass;

	parent_class = gtk_type_class (PARENT_TYPE);
	
	object_class->destroy = shell_destroy;

	epv->addControl          = impl_addControl;
	epv->removeControl       = impl_removeControl;
	epv->getProject          = impl_getProject;
	epv->getUIContainer      = impl_getUIContainer;
	epv->getProxyEventSource = impl_getProxyEventSource;
}

static void shell_init (Shell *shell)
{
	shell->priv = g_new0 (ShellPriv, 1);

	shell->priv->controls = g_hash_table_new (g_str_hash, g_str_equal);
}

static void
listener_callback (BonoboListener    *listener,
		   char              *event_name, 
		   CORBA_any         *any,
		   CORBA_Environment *ev,
		   gpointer           user_data)
{
	Shell             *shell;
	CORBA_Environment  e;
	CORBA_any         *dup;
	
	shell = SHELL (user_data);

	CORBA_exception_init (&e);

	/* Note: The dup/copy should not be needed. */
	dup = CORBA_any__alloc ();
	CORBA_any__copy (dup, any);
	CORBA_any_set_release (dup, CORBA_TRUE);

	d(g_print ("proxying event: '%s'\n", event_name);
	if (strstr (event_name, "task:ins")) {
		g_print ("** INSERT: %d\n", ((GM_Task*)any->_value)->taskId);
	});
	
	bonobo_event_source_notify_listeners (shell->priv->event_source,
					      event_name,
					      dup,
					      &e);
	if (BONOBO_EX (&e)) {
		g_warning ("Exception when proxying event.");
	}

	CORBA_free (dup);
	
	CORBA_exception_free (&e);
}

static void
activate_components (Shell *shell)
{
	CORBA_Environment      ev;
	char                  *sort[2];
	const gchar           *query;
	OAF_ServerInfoList    *oaf_result;
	guint                  i;
	GNOME_MrProject_Shell  shell_co;
	OAF_ServerInfo         server;
	GM_ShellComponent      component_co;
	CORBA_boolean          really_add = FALSE;
	
	CORBA_exception_init (&ev);
	
	query =	"repo_ids.has ('IDL:GNOME/MrProject/ShellComponent:1.0')";
	
	/* Hack to get the gantt chart first. Should look into some
	 * priority scheme or some such.
	 */
	sort[0] = "mrproject:supported_views.has ('gantt_chart')";
	sort[1] = NULL;
	
	oaf_result = oaf_query (query, sort, &ev);
	
	if (!BONOBO_EX (&ev) && oaf_result != NULL) {
		for (i = 0; i < oaf_result->_length; i++) {
			server = oaf_result->_buffer[i];
			
			component_co = oaf_activate_from_id (server.iid,
							     0,	   /* OAF_ActivationFlags */
							     NULL, /* OAF_ActivationID* */
							     &ev);
			
			if (BONOBO_EX (&ev) || !component_co) {
				gchar *msg = g_strdup_printf (
					"Could not activate the component: %s\n"
					"Make sure you followed the instructions in the README file.", 
					server.iid);
				g_message (msg);
				g_free (msg);
				
				if (ev._major == CORBA_USER_EXCEPTION) {
					if (!strncmp ((ev)._repo_id, "IDL:OAF/GeneralError:1.0",
						      sizeof ("IDL:OAF/GeneralError:1.0") - 1)) {
						OAF_GeneralError *general_error;
						
						general_error = CORBA_exception_value (&ev);
						g_print ("OAF error: %s\n", general_error->description);
					}
				}						
				else {
					g_print ("%s", bonobo_exception_get_text (&ev));
				}
				
				CORBA_exception_free (&ev);
				continue;
			}
			
			shell_co = BONOBO_OBJREF (BONOBO_OBJECT (shell));
			really_add = GNOME_MrProject_ShellComponent_addToShell (component_co,
										shell_co,
										&ev);
			if (BONOBO_EX (&ev) || !really_add) {
				gchar *str;
				str = bonobo_exception_get_text (&ev);

				g_print ("Component '%s' not added. (%s)\n",
					 server.iid,
					 str);
				g_free (str);
				CORBA_exception_free (&ev);
				
				bonobo_object_release_unref (component_co, NULL);
			}
			else {
				shell->priv->components = g_slist_prepend (shell->priv->components, component_co);
			}
		}
	}
	
	if (oaf_result) {
		CORBA_free (oaf_result);
	}
	
	CORBA_exception_free (&ev);
}

static void
terminate_components (Shell *shell)
{
	CORBA_Environment  ev;
	GSList            *l;

	d(puts (__FUNCTION__));

	CORBA_exception_init (&ev);
	
	for (l = shell->priv->components; l; l = l->next) {
		CORBA_Object component;

		component = l->data;
		
		GNOME_MrProject_ShellComponent_removeFromShell (component, &ev);
		if (BONOBO_EX (&ev)) {
			CORBA_exception_free (&ev);
		}

		CORBA_Object_release (component, NULL);
	}

	g_slist_free (shell->priv->components);
	shell->priv->components = NULL;

	CORBA_exception_free (&ev);
}

static Shell *
shell_construct (Shell             *shell,
		 ProjectWindow     *window,
		 BonoboUIContainer *ui_container)
{
	BonoboEventSource *event_source;
	BonoboListener    *listener;

	g_return_val_if_fail (shell != NULL, NULL);
	g_return_val_if_fail (IS_SHELL (shell), NULL);

	event_source = bonobo_event_source_new ();
	shell->priv->event_source = event_source;

	listener = bonobo_listener_new (listener_callback, shell);
	bonobo_object_add_interface (BONOBO_OBJECT (shell), BONOBO_OBJECT (listener));
	shell->priv->listener = listener;

	shell->priv->project_window = window;
	shell->priv->container = ui_container;

	activate_components (shell);

	return shell;
}

Shell *
shell_new (ProjectWindow     *window,
	   BonoboUIContainer *ui_container)
{
	Shell *shell;
	
	shell = gtk_type_new (shell_get_type ());
	if (!shell_construct (shell, window, ui_container)) {
		return NULL;
	}

	return shell;
}

void
shell_activate_control (Shell *shell, const gchar *id)
{
	CORBA_Environment  ev;
	ControlInfo     *info;

	g_return_if_fail (shell != NULL);
	g_return_if_fail (IS_SHELL (shell));
	g_return_if_fail (id != NULL);
	
	CORBA_exception_init (&ev);

	info = g_hash_table_lookup (shell->priv->controls, id);
	if (!info) {
		g_warning ("Trying to switch to non-existant component.");
		return;
	}

	/* Don't do anything if this control is already active. */
	if (shell->priv->active_control && 
	    (CORBA_Object_is_equivalent (info->control, shell->priv->active_control->control, &ev) || BONOBO_EX (&ev))) {
		CORBA_exception_free (&ev);
		return;
	}

	Bonobo_Control_activate (info->control, TRUE, &ev);
	if (BONOBO_EX (&ev)) {
		g_warning ("Could not activate component '%s'.", id);
	} else {
		if (shell->priv->active_control && shell->priv->active_control->control) {
			Bonobo_Control_activate (shell->priv->active_control->control, FALSE, &ev);
		}
		shell->priv->active_control = info;
	}

	CORBA_exception_free (&ev);
}

void
shell_open (Shell *shell, const gchar *uri, CORBA_Environment *ev)
{
	ShellPriv             *priv;
	GNOME_MrProject_Shell  shell_co;
	CORBA_any	      *any;
	
	g_return_if_fail (shell != NULL);
	g_return_if_fail (IS_SHELL (shell));

	d(puts (__FUNCTION__));

	priv = shell->priv;
	shell_co = BONOBO_OBJREF (shell);

	/* a project only present, destroy it */
	if (priv->project_co) {
		d(g_print ("Removing old project\n"));
		
		any = CORBA_any__alloc ();
		any->_type = TC_null; /*TC_GNOME_MrProject_Project;*/
		any->_value = CORBA_OBJECT_NIL; /*priv->project_co;*/
		CORBA_any_set_release (any, FALSE);
		
		bonobo_event_source_notify_listeners (priv->event_source,
						      "GNOME/MrProject:project:unset",
						      any,
						      NULL);
		CORBA_free (any);
		GNOME_MrProject_Project_removeShell (priv->project_co, shell_co, ev);

		bonobo_object_release_unref (priv->project_co, ev);
		priv->project_co = CORBA_OBJECT_NIL;
	} 

	priv->project_co = oaf_activate_from_id (PROJECT_OAFIID,
						 0,    /* OAF_ActivationFlags */
						 NULL, /* OAF_ActivationID */
						 ev);

	if (BONOBO_EX (ev) || priv->project_co == CORBA_OBJECT_NIL) {
		g_warning ("Could not activate Project.");
		return;
	}

	if (!check_version (shell)) {
		gtk_main_quit ();
		return;
	}

#if 0
	{
		CORBA_Environment  ev;
		CORBA_char        *ior;

		CORBA_exception_init (&ev);

		ior = CORBA_ORB_object_to_string (bonobo_orb (), priv->project_co, &ev);
		g_print ("ior %s\n", ior);

		CORBA_free (ior);
		CORBA_exception_free (&ev);
	}
#endif
	
	GNOME_MrProject_Project_addShell (priv->project_co,
					  shell_co,
					  ev);
	if (BONOBO_EX (ev)) {
		g_warning ("Could not add shell to project.");
		bonobo_object_release_unref (priv->project_co, ev);
		priv->project_co = CORBA_OBJECT_NIL;
		return;
	}
	
	any = CORBA_any__alloc ();
	any->_type = TC_null, /*TC_GNOME_MrProject_Project;*/
	any->_value = CORBA_OBJECT_NIL; /*priv->project_co;*/
	CORBA_any_set_release (any, FALSE);
	
	bonobo_event_source_notify_listeners (priv->event_source,
					      "GNOME/MrProject:project:set",
					      any,
					      NULL);
	
	CORBA_free (any);

	if (uri != NULL && strlen (uri) > 0) {
		GNOME_MrProject_Project_load (priv->project_co,
					      uri,
					      ev);
	}
}

void
shell_save (Shell *shell, const gchar *uri, CORBA_Environment *ev)
{
	GNOME_MrProject_Shell shell_co;

	g_return_if_fail (shell != NULL);
	g_return_if_fail (IS_SHELL (shell));
	g_return_if_fail (uri != NULL);

	/* Save the project data and the state of the shell and
	 * its controls.
	 */

	/* Plan:
	* 1. Tell the project to save itself.
	* 2. Pass a stream for the shell to add to the storage, plus a path.
	* 3. Pass streams+paths for all the controls.
	*/
	
	shell_co = bonobo_object_corba_objref (BONOBO_OBJECT (shell));
	GNOME_MrProject_Project_save (shell->priv->project_co, uri, ev);
	if (BONOBO_EX (ev)) {
		return;
	}

	/* FIXME: Don't hardcode the path here, should be the name used for the shell. */
	/*GNOME_MrProject_Project_saveSubdirectory (shell->priv->project_co, shell_co, "Shell", ev);*/
	if (BONOBO_EX (ev)) {
		return;
	}

	/* FIXME: save controls here. */
}

GNOME_MrProject_Project
shell_get_project (Shell *shell)
{
	g_return_val_if_fail (shell != NULL, CORBA_OBJECT_NIL);
	g_return_val_if_fail (IS_SHELL (shell), CORBA_OBJECT_NIL);

	return shell->priv->project_co;
}

void
shell_set_range (Shell *shell, time_t first, time_t last)
{
	BonoboArg         *arg;
	CORBA_Environment  ev;

	g_return_if_fail (shell != NULL);
	g_return_if_fail (IS_SHELL (shell));

	CORBA_exception_init (&ev);
	
	arg = bonobo_arg_new (BONOBO_ARG_INT);

	BONOBO_ARG_SET_INT (arg, first);
	bonobo_event_source_notify_listeners (shell->priv->event_source,
					      "Bonobo/Property:change:FirstTime",
					      (CORBA_any *) arg,
					      &ev);
	if (BONOBO_EX (&ev)) {
		g_warning ("Could not set start time.");
		CORBA_exception_free (&ev);
	}
	
	BONOBO_ARG_SET_INT (arg, last);
	bonobo_event_source_notify_listeners (shell->priv->event_source,
					      "Bonobo/Property:change:LastTime",
					      (CORBA_any *) arg,
					      &ev);
	if (BONOBO_EX (&ev)) {
		g_warning ("Could not set end time.");
		CORBA_exception_free (&ev);
	}

	CORBA_exception_free (&ev);
	bonobo_arg_release (arg);
}

BONOBO_X_TYPE_FUNC_FULL (Shell,
			 GNOME_MrProject_Shell,
			 PARENT_TYPE,
			 shell);



#if 0
void
shell_remove_component (Shell *shell, const gchar *id)
{
	ControlInfo        *info;
	gpointer            key, value;
	BonoboControlFrame *frame;
	CORBA_Environment   ev;

	if (id == NULL || strlen (id) == 0) {
		g_warning ("Trying to remove a NULL component Id.");
		return;
	}

	if (!g_hash_table_lookup_extended (shell->priv->controls, id, &key, &value)) {
		g_warning ("Trying to remove a component that is not added.");
		return;
	}

	info = value;
	
	g_hash_table_remove (shell->priv->controls, id);
	g_free (key);

	CORBA_exception_init (&ev);
	
	GNOME_MrProject_ShellComponent_removeFromShell (info->component, &ev);

	frame = bonobo_widget_get_control_frame (BONOBO_WIDGET (info->widget));
	bonobo_object_unref (BONOBO_OBJECT (frame));

	/* Release our CORBA Object reference. */
	CORBA_Object_release (info->component, &ev);

	CORBA_exception_free (&ev);

	info->control = NULL;
	info->component = NULL;
	info->widget = NULL;
}
#endif

void
shell_remove_from_project (Shell *shell)
{
	CORBA_Environment ev;

	d(puts (__FUNCTION__));
	
	CORBA_exception_init (&ev);

	GNOME_MrProject_Project_removeShell (shell->priv->project_co,
					     BONOBO_OBJREF (shell),
					     &ev);
	if (BONOBO_EX (&ev)) {
		/*g_warning ("Could not remove shell from project.");*/
		CORBA_exception_free (&ev);
	}

	bonobo_object_release_unref (shell->priv->project_co, NULL);

	shell->priv->project_co = CORBA_OBJECT_NIL;

	CORBA_exception_free (&ev);
}

/* Check that we have a matching version of the project engine. */
static gboolean
check_version (Shell *shell)
{
	CORBA_Environment  ev;
	CORBA_Object       obj;
	gchar             *version;

	CORBA_exception_init (&ev);

	/* Note: using property_bag_client directly should work, but it
	 * doesn't so we do the queryInterface ourselves.
	 */
	
	obj = Bonobo_Unknown_queryInterface (shell->priv->project_co,
					     "IDL:Bonobo/PropertyBag:1.0",
					     &ev);
	if (!obj || BONOBO_EX (&ev)) {
		CORBA_exception_free (&ev);
		return FALSE;
	}

	CORBA_exception_free (&ev);

	version = bonobo_property_bag_client_get_value_string (obj,
							       "Version",
							       NULL);
	bonobo_object_release_unref (obj, NULL);
	
	if (!version || strcmp (version, VERSION)) {
		g_warning ("The project engine version (%s) does not match the front end version (%s)."
			   "Try exiting MrProject and terminating the engine, then try again.",
			   version, VERSION);
		g_free (version);
		return FALSE;
	}

	g_free (version);
	return TRUE;
}
