/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Mikael Hallendal <micke@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Mikael Hallendal <micke@codefactory.se>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <bonobo.h>
#include <bonobo/bonobo-listener.h>
#include <gtk/gtksignal.h>
#include "util/type-utils.h"
#include "resource-manager-client.h"

#define d(x)

static void  resource_mc_init        (ResourceManagerClient        *rmc);
static void  resource_mc_class_init  (ResourceManagerClientClass   *klass);
static void  rmc_destroy             (GtkObject                    *object);

static void  rmc_listener            (ManagerClient                *mc,
				      gchar                        *subtype,
				      CORBA_any                    *any);
static void  rmc_project_set         (ManagerClient                *rmc);
static void  rmc_project_unset       (ManagerClient                *rmc);

enum {
	RESOURCE_INSERTED,
	RESOURCE_UPDATED,
	RESOURCES_REMOVED,
	GROUP_INSERTED,
	GROUP_UPDATED,
	GROUP_REMOVED,
	GROUP_DEFAULT_SET,
	LAST_SIGNAL
};

static gint signals[LAST_SIGNAL] = { 0 };

struct _ResourceManagerClientPriv {
        GM_ResourceManager manager;
};

GNOME_CLASS_BOILERPLATE (ResourceManagerClient, resource_mc,
                         ManagerClient,         manager_client);

static void
resource_mc_init (ResourceManagerClient *rmc)
{
        ResourceManagerClientPriv *priv;
        
	d(puts (__FUNCTION__));

        priv = g_new0 (ResourceManagerClientPriv, 1);
        
        rmc->priv = priv;
}

static void
resource_mc_class_init (ResourceManagerClientClass *klass)
{
	GtkObjectClass     *object_class;
	ManagerClientClass *mcc;
	
	object_class = (GtkObjectClass *) klass;
	mcc          = MANAGER_CLIENT_CLASS (klass);

	mcc->listener         = rmc_listener;
	mcc->project_set      = rmc_project_set;
	mcc->project_unset    = rmc_project_unset;

        object_class->destroy = rmc_destroy;

	d(puts (__FUNCTION__));

	signals[RESOURCE_INSERTED] = 
		gtk_signal_new ("resource_inserted",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (ResourceManagerClientClass,
						   resource_inserted),
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE,
				1, GTK_TYPE_POINTER);
	signals[RESOURCE_UPDATED] = 
		gtk_signal_new ("resource_updated",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (ResourceManagerClientClass,
						   resource_updated),
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE,
				1, GTK_TYPE_POINTER);
	signals[RESOURCES_REMOVED] = 
		gtk_signal_new ("resources_removed",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (ResourceManagerClientClass,
						   resources_removed),
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE,
				1, GTK_TYPE_POINTER);
	signals[GROUP_INSERTED] = 
		gtk_signal_new ("group_inserted",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (ResourceManagerClientClass,
						   group_inserted),
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE,
				1, GTK_TYPE_POINTER);
	signals[GROUP_UPDATED] = 
		gtk_signal_new ("group_updated",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (ResourceManagerClientClass,
						   group_updated),
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE,
				1, GTK_TYPE_POINTER);
	signals[GROUP_REMOVED] = 
		gtk_signal_new ("group_removed",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (ResourceManagerClientClass,
						   group_removed),
				gtk_marshal_NONE__INT,
				GTK_TYPE_NONE,
				1, GTK_TYPE_INT);
	signals[GROUP_DEFAULT_SET] = 
		gtk_signal_new ("group_default_set",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (ResourceManagerClientClass,
						   group_default_set),
				gtk_marshal_NONE__INT,
				GTK_TYPE_NONE,
				1, GTK_TYPE_INT);

	gtk_object_class_add_signals (object_class, signals, LAST_SIGNAL);
}

static void
rmc_destroy (GtkObject *object)
{
        ResourceManagerClient     *rmc;
        ResourceManagerClientPriv *priv;
	
        g_return_if_fail (object != NULL);
        g_return_if_fail (IS_RESOURCE_MANAGER_CLIENT (object));
        
	d(puts (__FUNCTION__));

        rmc  = RESOURCE_MANAGER_CLIENT (object);
	priv = rmc->priv;
	
	bonobo_object_release_unref (priv->manager, NULL);
	priv->manager = NULL;

	g_free (rmc->priv);
        rmc->priv = NULL;
	
	GNOME_CALL_PARENT_HANDLER (GTK_OBJECT_CLASS, destroy, (object));
}

static void 
rmc_listener (ManagerClient *mc, gchar *subtype, CORBA_any *any)
{
	ResourceManagerClient     *rmc;
	ResourceManagerClientPriv *priv;
	GM_Resource               *resource;
	GM_ResourceGroup          *group;
	
	g_return_if_fail (mc != NULL);
	g_return_if_fail (IS_RESOURCE_MANAGER_CLIENT (mc));
	
	d(puts (__FUNCTION__));

	rmc  = RESOURCE_MANAGER_CLIENT (mc);
	priv = rmc->priv;
	
	if (!strcmp (subtype, "inserted")) {
		resource = (GM_Resource *)(any->_value);
		gtk_signal_emit (GTK_OBJECT (rmc), 
				 signals[RESOURCE_INSERTED],
				 resource);
	}
	else if (!strcmp (subtype, "updated")) {
		resource = (GM_Resource *)(any->_value);
		gtk_signal_emit (GTK_OBJECT (rmc),
				 signals[RESOURCE_UPDATED],
				 resource);
	}
	else if (!strcmp (subtype, "removed_seq")) {
		BonoboArg *arg = (BonoboArg *) any;
		gchar     *id_str;
		GSList    *list;
		
		id_str = BONOBO_ARG_GET_STRING (arg);
		list   = corba_util_id_string_to_list (id_str);
		
		gtk_signal_emit (GTK_OBJECT (rmc),
				 signals[RESOURCES_REMOVED],
				 list);

		g_slist_free (list);
	}
	else if (!strcmp (subtype, "group_inserted")) {
		group = (GM_ResourceGroup *)(any->_value);
		gtk_signal_emit (GTK_OBJECT (rmc),
				 signals[GROUP_INSERTED],
				 group);
	} 	
	else if (!strcmp (subtype, "group_updated")) {
		group = (GM_ResourceGroup *)(any->_value);
		gtk_signal_emit (GTK_OBJECT (rmc),
				 signals[GROUP_UPDATED],
				 group);
	} 	
	else if (!strcmp (subtype, "group_removed")) {
		GNOME_MrProject_EventResourceGroupRemoved *event;
		event = (GNOME_MrProject_EventResourceGroupRemoved *)(any->_value);
		gtk_signal_emit (GTK_OBJECT (rmc),
				 signals[GROUP_REMOVED],
				 event->id);
	}
	else if (!strcmp (subtype, "group_default_set")) {
		GNOME_MrProject_EventResourceGroupDefaultSet *event;
		event = (GNOME_MrProject_EventResourceGroupDefaultSet *)(any->_value);
		gtk_signal_emit (GTK_OBJECT (rmc),
				 signals[GROUP_DEFAULT_SET],
				 event->id);
	} else {
		d(g_print ("ResourceManagerClient: got unhandled event: %s\n",
			   subtype));
	}
}

static void
rmc_project_set (ManagerClient *mc) 
{
	ResourceManagerClient     *rmc;
	ResourceManagerClientPriv *priv;
	GM_Project                 project_co;
	CORBA_Environment          ev;
	
	g_return_if_fail (mc != NULL);
	g_return_if_fail (IS_RESOURCE_MANAGER_CLIENT (mc));

	d(puts (__FUNCTION__));

	rmc  = RESOURCE_MANAGER_CLIENT (mc);
	priv = rmc->priv;

	if (priv->manager) {
		bonobo_object_release_unref (priv->manager, NULL);
	}

	if (!mc->shell) {
		g_warning ("ResourceManagerClient doesn't have a shell");
		return;
	}

	CORBA_exception_init (&ev);
	
	project_co = GNOME_MrProject_Shell_getProject (mc->shell, &ev);
	
	if (BONOBO_EX (&ev) || !project_co) {
		g_warning ("Shell doesn't have a project");
		CORBA_exception_free (&ev);
		return;
	}

	priv->manager = Bonobo_Unknown_queryInterface (project_co,
						       "IDL:GNOME/MrProject/ResourceManager:1.0",
						       &ev);
	
	if (BONOBO_EX (&ev)) {
		g_warning ("Couldn't get ResourceManager interface through project");
	}

	CORBA_Object_release (project_co, NULL);

	CORBA_exception_free (&ev);
}

static void
rmc_project_unset (ManagerClient *mc)
{
	ResourceManagerClient     *rmc;
	ResourceManagerClientPriv *priv;

	d(puts (__FUNCTION__));

	g_return_if_fail (mc != NULL);
	g_return_if_fail (IS_RESOURCE_MANAGER_CLIENT (mc));

	rmc  = RESOURCE_MANAGER_CLIENT (mc);
	priv = rmc->priv;

	if (priv->manager) {
		bonobo_object_release_unref (priv->manager, NULL);
	}

	priv->manager = CORBA_OBJECT_NIL;
}

ManagerClient *
resource_mc_new (GM_Shell shell, gboolean set, CORBA_Environment *ev)
{
	ResourceManagerClient     *rmc;
	ResourceManagerClientPriv *priv;
	
	d(puts (__FUNCTION__));

	rmc  = gtk_type_new (RESOURCE_MANAGER_CLIENT_TYPE);
	priv = rmc->priv;
	
	manager_client_set_shell (MANAGER_CLIENT (rmc),
				  shell,
				  "resource",
				  ev);
	
	if (set) {
		rmc_project_set (MANAGER_CLIENT (rmc));
	}
 	
	return MANAGER_CLIENT(rmc);
}

GM_Resource *  
resource_mc_create_resource (ResourceManagerClient *rmc, CORBA_Environment *ev)
{
	ResourceManagerClientPriv *priv;
	GM_Resource               *resource;
	
	g_return_val_if_fail (rmc != NULL, NULL);
	g_return_val_if_fail (IS_RESOURCE_MANAGER_CLIENT(rmc), NULL);

	d(puts (__FUNCTION__));

	priv = rmc->priv;
	
	resource = GNOME_MrProject_ResourceManager_createResource (priv->manager,
								   ev);
	
	/* FIX: Check for exceptions */
	
	return resource;
}

GM_Id
resource_mc_insert_resource (ResourceManagerClient *rmc,
			     GM_Resource           *resource,
			     CORBA_Environment     *ev)
{
	ResourceManagerClientPriv *priv;
	gboolean                   free_resource = FALSE;
	GM_Id                      id;
	
	g_return_val_if_fail (rmc != NULL, -1);
	g_return_val_if_fail (IS_RESOURCE_MANAGER_CLIENT(rmc), -1);

	d(puts (__FUNCTION__));

	priv = rmc->priv;
	
	if (!resource) {
		resource = resource_mc_create_resource (rmc, ev);
		free_resource = TRUE;
	}
	
	id = GNOME_MrProject_ResourceManager_insertResource (priv->manager,
							     resource,
							     ev);
	
	/* FIX: Check for exceptions */

	if (free_resource) {
		CORBA_free (resource);
	}
	
	return id;
}

GM_Resource *  
resource_mc_get_resource (ResourceManagerClient *rmc,
			  GM_Id                  res_id,
			  CORBA_Environment     *ev)
{
	ResourceManagerClientPriv *priv;
	GM_Resource               *resource;

	g_return_val_if_fail (rmc != NULL, NULL);
	g_return_val_if_fail (IS_RESOURCE_MANAGER_CLIENT(rmc), NULL);

	d(puts (__FUNCTION__));

	priv = rmc->priv;

	resource = GNOME_MrProject_ResourceManager_getResource (priv->manager,
								res_id,
								ev);
	
	/* FIX: Check for exceptions */

	return resource;
}

GSList *       
resource_mc_get_all_resources (ResourceManagerClient *rmc, 
			       CORBA_Environment     *ev)
{
	ResourceManagerClientPriv *priv;
	GM_ResourceSeq            *resources;
	GSList                    *list;
	
	g_return_val_if_fail (rmc != NULL, NULL);
	g_return_val_if_fail (IS_RESOURCE_MANAGER_CLIENT(rmc), NULL);

	d(puts (__FUNCTION__));

	priv = rmc->priv;

	resources = GNOME_MrProject_ResourceManager_getAllResources (
		priv->manager, ev);
	
	/* FIX: Check for exceptions */

	list = corba_util_resource_seq_to_list (resources);

	CORBA_free (resources);
	
	return list;
}

void           
resource_mc_update_resource (ResourceManagerClient *rmc,
			     GM_Resource           *resource,
			     CORBA_Environment     *ev)
{
	ResourceManagerClientPriv *priv;
	
	g_return_if_fail (rmc != NULL);
	g_return_if_fail (IS_RESOURCE_MANAGER_CLIENT(rmc));

	d(puts (__FUNCTION__));

	priv = rmc->priv;

	GNOME_MrProject_ResourceManager_updateResource (priv->manager,
							resource,
							ev);
	
	/* FIX: Check for exceptions */
}

void
resource_mc_remove_resources (ResourceManagerClient *rmc,
			      GSList                *res_ids,
			      CORBA_Environment     *ev)
{
	ResourceManagerClientPriv *priv;
	GM_IdSeq                  *resources;
	
	g_return_if_fail (rmc != NULL);
	g_return_if_fail (IS_RESOURCE_MANAGER_CLIENT(rmc));

	d(puts (__FUNCTION__));

	priv = rmc->priv;

	resources = corba_util_id_seq_from_list (res_ids);

	GNOME_MrProject_ResourceManager_removeResources (priv->manager,
							 resources,
							 ev);
	
	/* FIX: Check for exceptions */
}

GM_ResourceGroup * 
resource_mc_create_group (ResourceManagerClient *rmc, CORBA_Environment *ev)
{
	ResourceManagerClientPriv *priv;
	GM_ResourceGroup          *group;
	
	g_return_val_if_fail (rmc != NULL, NULL);
	g_return_val_if_fail (IS_RESOURCE_MANAGER_CLIENT(rmc), NULL);

	d(puts (__FUNCTION__));

	priv = rmc->priv;

	group = GNOME_MrProject_ResourceManager_createGroup (priv->manager,
							     ev);
	
	/* FIX: Check for exceptions */

	return group;
}

GM_Id
resource_mc_insert_group (ResourceManagerClient *rmc,
			  GM_ResourceGroup      *group,
			  CORBA_Environment     *ev)
{
	ResourceManagerClientPriv *priv;
	GM_Id                      id;
	gboolean                   free_group = FALSE;
	
	g_return_val_if_fail (rmc != NULL, -1);
	g_return_val_if_fail (IS_RESOURCE_MANAGER_CLIENT(rmc), -1);

	d(puts (__FUNCTION__));

	priv = rmc->priv;

	if (!group) {
		group = resource_mc_create_group (rmc, ev);
		free_group = TRUE;
	}

 	id = GNOME_MrProject_ResourceManager_insertGroup (priv->manager,
							  group,
							  ev);
	
	/* FIX: Check for exceptions */
	
	if (free_group) {
		CORBA_free (group);
	}
	
	return id;
}

GM_ResourceGroup *
resource_mc_get_group (ResourceManagerClient *rmc,
		       GM_Id                  group_id,
		       CORBA_Environment     *ev)
{
	ResourceManagerClientPriv *priv;
	GM_ResourceGroup          *group;
	
	g_return_val_if_fail (rmc != NULL, NULL);
	g_return_val_if_fail (IS_RESOURCE_MANAGER_CLIENT(rmc), NULL);

	d(puts (__FUNCTION__));

	priv = rmc->priv;

	group = GNOME_MrProject_ResourceManager_getGroup (priv->manager,
							  group_id,
							  ev);
	
	/* FIX: Check for exceptions */

	return group;
}

GSList *
resource_mc_get_all_groups (ResourceManagerClient *rmc, CORBA_Environment *ev)
{
	ResourceManagerClientPriv *priv;
	GM_ResourceGroupSeq       *groups;
	GSList                    *list;
	
	g_return_val_if_fail (rmc != NULL, NULL);
	g_return_val_if_fail (IS_RESOURCE_MANAGER_CLIENT(rmc), NULL);

	d(puts (__FUNCTION__));

	priv = rmc->priv;

	groups = GNOME_MrProject_ResourceManager_getAllGroups (priv->manager,
							       ev);
	
	/* FIX: Check for exceptions */

	list = corba_util_resource_group_seq_to_list (groups);

	CORBA_free (groups);
	
	return list;
}

void
resource_mc_update_group (ResourceManagerClient *rmc,
			  GM_ResourceGroup      *group,
			  CORBA_Environment     *ev)
{
	ResourceManagerClientPriv *priv;
	
	g_return_if_fail (rmc != NULL);
	g_return_if_fail (IS_RESOURCE_MANAGER_CLIENT(rmc));
	
	d(puts (__FUNCTION__));

	priv = rmc->priv;
	
	GNOME_MrProject_ResourceManager_updateGroup (priv->manager,
						     group,
						     ev);
	
	/* FIX: Check for exceptions */
}

void
resource_mc_remove_group (ResourceManagerClient *rmc,
			  GM_Id                  group_id,
			  CORBA_Environment     *ev)
{
	ResourceManagerClientPriv *priv;
	
	g_return_if_fail (rmc != NULL);
	g_return_if_fail (IS_RESOURCE_MANAGER_CLIENT(rmc));

	d(puts (__FUNCTION__));

	priv = rmc->priv;
 
	GNOME_MrProject_ResourceManager_removeGroup (priv->manager,
						     group_id,
						     ev);
	
	/* FIX: Check for exceptions */
}

GM_Id
resource_mc_get_default_group (ResourceManagerClient *rmc, 
			       CORBA_Environment     *ev)
{
	ResourceManagerClientPriv *priv;
	GM_Id                      id;
	
	g_return_val_if_fail (rmc != NULL, -1);
	g_return_val_if_fail (IS_RESOURCE_MANAGER_CLIENT(rmc), -1);

	d(puts (__FUNCTION__));

	priv = rmc->priv;
 
	id = GNOME_MrProject_ResourceManager_getDefaultGroup (priv->manager,
							      ev);
	
	/* FIX: Check for exceptions */

	return id;
}

void
resource_mc_set_default_group (ResourceManagerClient *rmc,
			       GM_Id                  group_id,
			       CORBA_Environment     *ev)
{
	ResourceManagerClientPriv *priv;
	
	g_return_if_fail (rmc != NULL);
	g_return_if_fail (IS_RESOURCE_MANAGER_CLIENT(rmc));

	d(puts (__FUNCTION__));

	priv = rmc->priv;
 
	GNOME_MrProject_ResourceManager_setDefaultGroup (priv->manager,
							 group_id,
							 ev);
	
	/* FIX: Check for exceptions */
}
