/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Mikael Hallendal <micke@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Mikael Hallendal <micke@codefactory.se>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "util/type-utils.h"
#include "manager-client.h"

#define DEBUG 0
#include "util/debug.h"

#define IS_EVENT(event_name, str) (!strncmp (event_name, str, sizeof (str)-1))

static void  manager_client_init       (ManagerClient    *mc);
static void  manager_client_class_init (GtkObjectClass   *klass);
static void  mc_destroy                (GtkObject        *object);
static void  mc_listener               (BonoboListener             *listener,
					char                       *event_name,
					CORBA_any                  *any,
					CORBA_Environment          *ev,
					gpointer                    user_data);

enum {
        ITEMS_CLEAR,
        PROJECT_SET,
        PROJECT_UNSET,
	LOAD_BEGIN,
	LOAD_END,
        LAST_SIGNAL
};

static gint signals[LAST_SIGNAL] = { 0 };

struct _ManagerClientPriv {
	Bonobo_EventSource_ListenerId  listener_id;
        BonoboListener                *listener;
        
        gchar                         *extra_kind;
};

GNOME_CLASS_BOILERPLATE (ManagerClient, manager_client,
                         GtkObject,     gtk_object);


static void
manager_client_init (ManagerClient *mc)
{
        ManagerClientPriv   *priv;
        
        priv = g_new0 (ManagerClientPriv, 1);
        
        mc->priv = priv;
}

static void
manager_client_class_init (GtkObjectClass *klass)
{
	d(puts(__FUNCTION__));
	
        klass->destroy = mc_destroy;
        
        signals[ITEMS_CLEAR] = 
                gtk_signal_new ("items_clear",
                                GTK_RUN_LAST,
                                klass->type,
                                GTK_SIGNAL_OFFSET (ManagerClientClass,
                                                   items_clear),
                                gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE,
                                0);
        
        signals[PROJECT_SET] =
                gtk_signal_new ("project_set",
                                GTK_RUN_LAST,
                                klass->type,
                                GTK_SIGNAL_OFFSET (ManagerClientClass,
                                                   project_set),
                                gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE,
                                0);
        
        signals[PROJECT_UNSET] =
                gtk_signal_new ("project_unset",
                                GTK_RUN_LAST,
                                klass->type,
                                GTK_SIGNAL_OFFSET (ManagerClientClass,
                                                   project_set),
                                gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE,
                                0);

	signals[LOAD_BEGIN] =
		gtk_signal_new ("load_begin",
                                GTK_RUN_LAST,
                                klass->type,
                                GTK_SIGNAL_OFFSET (ManagerClientClass,
                                                   load_begin),
                                gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE,
                                0);

	signals[LOAD_END] =
		gtk_signal_new ("load_end",
                                GTK_RUN_LAST,
                                klass->type,
                                GTK_SIGNAL_OFFSET (ManagerClientClass,
                                                   load_end),
                                gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE,
                                0);

        gtk_object_class_add_signals (klass, signals, LAST_SIGNAL);
}

static void
mc_destroy (GtkObject *object)
{
        ManagerClient        *mc;
        ManagerClientPriv    *priv;
        CORBA_Environment     ev;
	Bonobo_EventSource    es;

	d(puts (__FUNCTION__));
	
        g_return_if_fail (object != NULL);
        g_return_if_fail (IS_MANAGER_CLIENT (object));
        
        mc   = MANAGER_CLIENT (object);
        priv = mc->priv;
        
	if (priv->extra_kind) {
		g_free (priv->extra_kind);
		priv->extra_kind = NULL;
	}
 	
        CORBA_exception_init (&ev);

	es = GNOME_MrProject_Shell_getProxyEventSource (mc->shell, &ev);

        if (!BONOBO_EX (&ev) && es) {
                Bonobo_EventSource_removeListener (es,
                                                   priv->listener_id,
                                                   &ev);
                bonobo_object_release_unref (es, NULL);
	}
	
	if (priv->listener) {
		bonobo_object_unref (BONOBO_OBJECT (priv->listener));
		priv->listener = NULL;
	}
	
	if (mc->shell) {
		CORBA_Object_release (mc->shell, NULL);
		mc->shell = NULL;
	}
	
        CORBA_exception_free (&ev);

        g_free (priv);
        mc->priv = NULL;
}

static void
mc_listener (BonoboListener      *listener,
             char                *event_name,
             CORBA_any           *any,
             CORBA_Environment   *ev,
             gpointer             user_data)
{
        ManagerClient        *mc;
        ManagerClientPriv    *priv;
        ManagerClientClass   *klass;
        gchar                *kind, *subtype;

        g_return_if_fail (user_data != NULL);
        g_return_if_fail (IS_MANAGER_CLIENT (user_data));

	d(puts (__FUNCTION__));
        
        mc   = MANAGER_CLIENT (user_data);
        priv = mc->priv;

        kind    = bonobo_event_kind    (event_name);
        subtype = bonobo_event_subtype (event_name);
        
        if (IS_EVENT (event_name, "GNOME/MrProject:")) {
                if (IS_EVENT (kind, priv->extra_kind)) {
                        klass = MANAGER_CLIENT_CLASS (GTK_OBJECT (mc)->klass);
                        
                        if (klass->listener) {
                                klass->listener (mc, subtype, any);
                        }
                }
                else if (IS_EVENT (kind, "project")) {
                        if (!strcmp (subtype, "set")) {
                                gtk_signal_emit (GTK_OBJECT (mc),
                                                 signals[PROJECT_SET],
                                                 NULL);
                        }
                        else if (!strcmp (subtype, "unset")) {
                                gtk_signal_emit (GTK_OBJECT (mc),
                                                 signals[PROJECT_UNSET],
                                                 NULL);
				
				gtk_signal_emit (GTK_OBJECT (mc),
						 signals[ITEMS_CLEAR],
						 NULL);
                        }
                        else if (!strcmp (subtype, "load_begin")) {
                                gtk_signal_emit (GTK_OBJECT (mc),
                                                 signals[LOAD_BEGIN],
                                                 NULL);
                        }
                        else if (!strcmp (subtype, "load_end")) {
                                gtk_signal_emit (GTK_OBJECT (mc),
                                                 signals[LOAD_END],
                                                 NULL);
                        }
                }
        }
        
        g_free (kind);
        g_free (subtype);
}

void
manager_client_set_shell (ManagerClient       *mc,
                          GM_Shell             shell, 
                          gchar               *eventname_kind,
                          CORBA_Environment   *ev)
{
        ManagerClientPriv    *priv;
        Bonobo_EventSource    event_source_co;
        Bonobo_Listener       listener_co;
        gchar                *listener_mask;
        
        g_return_if_fail (mc != NULL);
        g_return_if_fail (IS_MANAGER_CLIENT (mc));

	d(puts(__FUNCTION__));

        priv = mc->priv;
        
        priv->extra_kind = g_strdup (eventname_kind);
        mc->shell        = CORBA_Object_duplicate (shell, ev);

        if (BONOBO_EX (ev) || !mc->shell) {
                g_warning ("Couldn't duplicate shell");
                return;
        }

	event_source_co = GNOME_MrProject_Shell_getProxyEventSource (mc->shell,
								     ev);
        
        if (BONOBO_EX (ev) || !event_source_co) {
                g_warning ("Couldn't get shell event source");
                return;
        }
        
        priv->listener = bonobo_listener_new (mc_listener, mc);
        listener_co = BONOBO_OBJREF (priv->listener);
	
        listener_mask = g_strconcat ("GNOME/MrProject:project,",
				     "GNOME/MrProject:",
				     priv->extra_kind,
				     NULL);
        
	d(g_print ("Adding listener with mask: %s\n",
		   listener_mask));

        priv->listener_id = 
		Bonobo_EventSource_addListenerWithMask (event_source_co,
							listener_co,
							listener_mask,
							ev);
        
        g_free (listener_mask);
        
        if (BONOBO_EX (ev)) {
                g_warning ("Couldn't add listener");
                bonobo_object_unref (BONOBO_OBJECT (priv->listener));
        }

        bonobo_object_release_unref (event_source_co, NULL);
}

